﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "./ngc_BinaryReader.h"
#include "./ngc_InputStream.h"

namespace nn { namespace ngc { namespace detail {

#define IS_READING_LITTLE_ENDIAN !m_SwapEndian

errno_t BinaryReader::Init(EndianSetting endian) NN_NOEXCEPT
{
    if (m_Errno != -1)
    {
        return EALREADY;
    }
    m_SwapEndian = (endian == EndianSetting_EndianBig);
    m_Errno = 0;
    return 0;
}

errno_t BinaryReader::Open(InputStream* stream) NN_NOEXCEPT
{
    if (m_Errno != 0)
    {
        if (m_Errno != -1)
        {
            return EINVAL;
        }
        errno_t e = Init();
        if (e != 0) return e;
    }
    if (!stream)
    {
        return EINVAL;
    }
    if (m_Stream)
    {
        return EEXIST;
    }
    m_Stream = stream;
    return 0;
}

bool BinaryReader::Skip_(size_t n) NN_NOEXCEPT
{
    if (m_Stream->Skip(n) != n)
    {
        if (!(*m_Stream))
        {
            this->SetError(EIO);
        }
        return false;
    }
    return true;
}

bool BinaryReader::Read_(uint8_t* x) NN_NOEXCEPT
{
    int c = m_MiniBuf.Read(m_Stream);
    if (c == -1)
    {
        if (!(*m_Stream))
        {
            this->SetError(EIO);
        }
        return false;
    }
    *x = static_cast<uint8_t>(c);
    return true;
}

bool BinaryReader::Read_(uint16_t* x) NN_NOEXCEPT
{
    if (!m_MiniBuf.Prefetch(m_Stream, 2))
    {
        if (!(*m_Stream))
        {
            this->SetError(EIO);
        }
        return false;
    }
    if (IS_READING_LITTLE_ENDIAN)
    {
        *x = static_cast<uint16_t>(m_MiniBuf[0] | (m_MiniBuf[1] << 8));
    }
    else
    {
        *x = static_cast<uint16_t>(m_MiniBuf[1] | (m_MiniBuf[0] << 8));
    }
    m_MiniBuf.Advance(2);
    return true;
}

bool BinaryReader::Read_(uint32_t* x) NN_NOEXCEPT
{
    if (!m_MiniBuf.Prefetch(m_Stream, 4))
    {
        if (!(*m_Stream))
        {
            this->SetError(EIO);
        }
        return false;
    }
    if (IS_READING_LITTLE_ENDIAN)
    {
        *x = static_cast<uint32_t>(m_MiniBuf[0] | (m_MiniBuf[1] << 8) |
                                   (m_MiniBuf[2] << 16) | (m_MiniBuf[3] << 24));
    }
    else
    {
        *x = static_cast<uint32_t>(m_MiniBuf[3] | (m_MiniBuf[2] << 8) |
                                   (m_MiniBuf[1] << 16) | (m_MiniBuf[0] << 24));
    }
    m_MiniBuf.Advance(4);
    return true;
}

bool BinaryReader::Read_(uint64_t* x) NN_NOEXCEPT
{
    if (!m_MiniBuf.Prefetch(m_Stream, 8))
    {
        if (!(*m_Stream))
        {
            this->SetError(EIO);
        }
        return false;
    }
    if (IS_READING_LITTLE_ENDIAN)
    {
        *x = static_cast<uint64_t>(m_MiniBuf[0]) | static_cast<uint64_t>(m_MiniBuf[1]) << 8 |
            static_cast<uint64_t>(m_MiniBuf[2]) << 16 | static_cast<uint64_t>(m_MiniBuf[3]) << 24 |
            static_cast<uint64_t>(m_MiniBuf[4]) << 32 | static_cast<uint64_t>(m_MiniBuf[5]) << 40 |
            static_cast<uint64_t>(m_MiniBuf[6]) << 48 | static_cast<uint64_t>(m_MiniBuf[7]) << 56;

    }
    else
    {
        *x = static_cast<uint64_t>(m_MiniBuf[7]) | static_cast<uint64_t>(m_MiniBuf[6]) << 8 |
            static_cast<uint64_t>(m_MiniBuf[5]) << 16 | static_cast<uint64_t>(m_MiniBuf[4]) << 24 |
            static_cast<uint64_t>(m_MiniBuf[3]) << 32 | static_cast<uint64_t>(m_MiniBuf[2]) << 40 |
            static_cast<uint64_t>(m_MiniBuf[1]) << 48 | static_cast<uint64_t>(m_MiniBuf[0]) << 56;
    }
    m_MiniBuf.Advance(8);
    return true;
}

size_t BinaryReader::ReadArray(unsigned char* x, size_t n) NN_NOEXCEPT
{
    if (n > RSIZE_MAX)
    {
        this->SetError(EINVAL);
        return 0;
    }
    return m_MiniBuf.ReadBytes(m_Stream, x, n);
}

size_t BinaryReader::ReadArray(unsigned short* x, size_t n) NN_NOEXCEPT
{  // NOLINT
    if (n > RSIZE_MAX)
    {
        this->SetError(EINVAL);
        return 0;
    }
    size_t rval = m_MiniBuf.Read(m_Stream, x, n);
    if (m_SwapEndian)
    {
        SwapEndian16(x, rval);
    }
    return rval;
}

size_t BinaryReader::ReadArray(unsigned int* x, size_t n) NN_NOEXCEPT
{
    if (n > RSIZE_MAX)
    {
        this->SetError(EINVAL);
        return 0;
    }
    size_t rval = m_MiniBuf.Read(m_Stream, x, n);
    if (m_SwapEndian)
    {
        SwapEndian32(x, rval);
    }
    return rval;
}

size_t BinaryReader::ReadArray(unsigned long long* x, size_t n) NN_NOEXCEPT
{  // NOLINT
    if (n > RSIZE_MAX)
    {
        this->SetError(EINVAL);
        return 0;
    }
    size_t rval = m_MiniBuf.Read(m_Stream, x, n);
    if (m_SwapEndian)
    {
        SwapEndian64(reinterpret_cast<uint64_t*>(x), rval);
    }
    return rval;
}

}}} // nn::ngc::detail
