﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/ngc/ngc_ProfanityFilterPatternList.h>


namespace nn {
namespace ngc {
namespace detail {

    class ContentsReader
    {
    public:

        /**
         * @brief   読み込む AC(AhoCorasick) バイナリの種類
         */
        enum AcType
        {
            AcType_NotBoundary = 0,     // 単語境界(\b)を持たないキーワードの AC バイナリ
            AcType_BoundaryScraped,     // 単語境界(\b)を持ち、登録時に単語境界を抜いて格納された AC バイナリ
            AcType_BoundaryNotScraped   // 単語境界(\b)を持ち、登録時に単語境界を抜かずに格納された AC バイナリ
        };
    public:
        // コンストラクタ・デストラクタ
        ContentsReader() NN_NOEXCEPT;
        ~ContentsReader() NN_NOEXCEPT;

        // マウント・アンマウント
        nn::Result MountContents(void* pCacheBuffer, size_t cacheBufferSize) NN_NOEXCEPT;
        nn::Result UnmountContents() NN_NOEXCEPT;

        // バージョンデータ取得
        nn::Result GetVersionData(void* data, uint32_t size) NN_NOEXCEPT;
        nn::Result GetVersionDataSize(uint32_t* size) NN_NOEXCEPT;

        // succinct 用
        /**
         * @brief   読み込む AC バイナリのパスを pOutPath に入れます。
         */
        bool MakeMountPath(char* pOutPath, size_t maxLength, AcType type, int lang) NN_NOEXCEPT;
        /**
         * @brief   ファイルパスの先頭がシステムデータのマウント名と一致するか確認します。
         */
        bool CheckMountPrefix(const char* filename) NN_NOEXCEPT;

    private:
        // ファイルの内容を取得
        nn::Result GetFileContent(void* data, char* path, uint32_t size) NN_NOEXCEPT;

        // ファイルサイズ取得
        nn::Result GetFileSize(uint32_t* pSize, char* path) NN_NOEXCEPT;

        // コンテンツのファイルパス取得
        nn::Result GetFilterFilePath(char* filePath, ProfanityFilterPatternList filterPattern, uint32_t size) NN_NOEXCEPT;
        nn::Result GetVersionFilePath(char* filePath, uint32_t size) NN_NOEXCEPT;

        // ファイルパスの最大長
        static const int FilePathMax = 128;

        // マウント済みかどうか
        bool m_ContentMounted;

        // マウントを行った回数を保存
        uint32_t m_MountCount;
    };

}
}
}

