﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs.h>
#include <nn/fs/fs_SystemData.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_ResEndian.h>
#include <nn/ngc/ngc_Result.h>
#include "ngc_Contents.h"
#include "../ngc_String.h"
#include "../ngc_Private.h"

#if defined( NN_BUILD_CONFIG_OS_WIN )
#include "ngc_GetFilterPatternData-os.win32.h"
#endif

#define NN_NGC_RETURN_IF_FAILED(r) NN_RESULT_DO(r)
// データを読み出す設定
// ファイルパス（フォーマット）
#define MOUNT_NAME              "NgWord"
#define VERSION_FILE_PATH       MOUNT_NAME ":" "/version.dat"
#define FILTER_FILE_FORMAT      MOUNT_NAME ":" "/%d.txt"
#define FILTER_COMMON_FORMAT    MOUNT_NAME ":" "/common.txt"

namespace nn
{
namespace ngc
{
namespace detail
{

namespace
{
    const nn::Bit64 NgcSystemDataId = NN_NGC_SYSTEM_DATA_ID;
}

/*!--------------------------------------------------------------------------*
  Name:         ContentsReader
  @brief        コンストラクタです。メンバ変数の初期化を行います。
 *---------------------------------------------------------------------------*/
ContentsReader::ContentsReader() NN_NOEXCEPT
{
    // マウント処理のフラグを初期化
    m_ContentMounted = false;

    // マウント処理の回数を初期化
    m_MountCount = 0;
}

/*!--------------------------------------------------------------------------*
  Name:         ContentsReader
  @brief        デストラクタです。何も行いません。
 *---------------------------------------------------------------------------*/
ContentsReader::~ContentsReader() NN_NOEXCEPT
{
}


/*!--------------------------------------------------------------------------*
  Name:         MountContents
  @brief        コンテンツを含んだアーカイブをマウントします。
  @return       結果が返ります。
 *---------------------------------------------------------------------------*/
nn::Result ContentsReader::MountContents(void* pCacheBuffer, size_t cacheBufferSize) NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    const nn::ncm::SystemDataId systemDataId = { NgcSystemDataId };
    size_t cacheSize;

    auto mountResult = nn::fs::QueryMountSystemDataCacheSize(&cacheSize, systemDataId);
    if(mountResult.IsFailure())
    {
        NN_SDK_LOG("Failed to mount 0x%016llx (module = 0x%08x, desc = 0x%08x). \n", NgcSystemDataId, mountResult.GetModule(),mountResult.GetDescription());
        NN_ABORT_UNLESS_RESULT_SUCCESS( mountResult );
    }
    if (cacheBufferSize < cacheSize)
    {
        NN_SDK_LOG("cacheBufferSize: %zu\n", cacheBufferSize);
        NN_SDK_LOG("cacheSize: %zu\n", cacheSize);
        NN_ABORT_UNLESS(cacheBufferSize >= cacheSize);
    }

    mountResult = nn::fs::MountSystemData(MOUNT_NAME, systemDataId, pCacheBuffer, cacheBufferSize);
    if(mountResult.IsFailure())
    {
        NN_SDK_LOG("Failed to mount 0x%016llx (module = 0x%08x, desc = 0x%08x). \n", NgcSystemDataId, mountResult.GetModule(),mountResult.GetDescription());
        NN_ABORT_UNLESS_RESULT_SUCCESS( mountResult );
    }
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    //何も実施しない。
    NN_UNUSED(pCacheBuffer);
    NN_UNUSED(cacheBufferSize);
#else
#error "unsupported os"
#endif

    return nn::ResultSuccess();
}

/*!--------------------------------------------------------------------------*
  Name:         UnmountContents
  @brief        コンテンツを含んだアーカイブをアンマウントします。
  @return       結果が返ります。
 *---------------------------------------------------------------------------*/
nn::Result ContentsReader::UnmountContents() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    nn::fs::Unmount(MOUNT_NAME);
#elif defined( NN_BUILD_CONFIG_OS_WIN )
#else
#error "unsupported os"
#endif
    return nn::ResultSuccess();
}

/*!--------------------------------------------------------------------------*
  Name:         MakeMountPath
  @brief        読み込む AC バイナリのパスを pOutPath に入れます。
  @return       成功したら true を返します。
 *---------------------------------------------------------------------------*/
bool ContentsReader::MakeMountPath(char* pOutPath, size_t maxLength, ContentsReader::AcType type, int lang) NN_NOEXCEPT
{
    char pPath[64];
    memset(pPath, 0, 64);
    switch (type)
    {
    case ContentsReader::AcType_NotBoundary:
        if (lang >= 0)
        {
            nn::util::SNPrintf(pPath, maxLength, MOUNT_NAME":/ac_%d_not_b_nx", lang);
        }
        else
        {
            nn::util::SNPrintf(pPath, maxLength, MOUNT_NAME":/ac_common_not_b_nx", lang);
        }
        break;
    case ContentsReader::AcType_BoundaryScraped:
        if (lang >= 0)
        {
            nn::util::SNPrintf(pPath, maxLength, MOUNT_NAME":/ac_%d_b1_nx", lang);
        }
        else
        {
            nn::util::SNPrintf(pPath, maxLength, MOUNT_NAME":/ac_common_b1_nx", lang);
        }
        break;
    case ContentsReader::AcType_BoundaryNotScraped:
        if (lang >= 0)
        {
            nn::util::SNPrintf(pPath, maxLength, MOUNT_NAME":/ac_%d_b2_nx", lang);
        }
        else
        {
            nn::util::SNPrintf(pPath, maxLength, MOUNT_NAME":/ac_common_b2_nx", lang);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    if (maxLength > strlen(pPath))
    {
        strncpy(pOutPath, pPath, 64);
        pOutPath[strlen(pPath)] = '\0';
        return true;
    }
    return false;
}

/*!--------------------------------------------------------------------------*
  Name:         MakeMountPath
  @brief        読み込む AC バイナリのパスを pOutPath に入れます。
  @return       成功したら true を返します。
 *---------------------------------------------------------------------------*/
bool ContentsReader::CheckMountPrefix(const char* filename) NN_NOEXCEPT
{
    const char* mountPrefix = MOUNT_NAME ":/";
    if (strncmp(filename, mountPrefix, strnlen(mountPrefix, 8)) != 0)
    {
        return false;
    }
    return true;
}

/*!--------------------------------------------------------------------------*
  Name:         GetFileContent
  @brief        指定したパターンのフィルタデータを取得します。
  @return       結果が返ります。
 *---------------------------------------------------------------------------*/
nn::Result ContentsReader::GetFileContent(void* data, char* path, uint32_t size) NN_NOEXCEPT
{
    nn::fs::FileHandle fileHandle;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(fileHandle);
    };

    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::fs::ReadFile(fileHandle, 0, data, size) );

    return nn::ResultSuccess();
}

/*!--------------------------------------------------------------------------*
  Name:         GetSize
  @brief        指定したパターンのフィルタデータを取得します。
  @return       結果が返ります。
 *---------------------------------------------------------------------------*/
nn::Result ContentsReader::GetFileSize(uint32_t* pSize, char* path) NN_NOEXCEPT
{
    nn::fs::FileHandle fileHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(fileHandle);
    };

    int64_t size;

    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::fs::GetFileSize(&size, fileHandle) );
    *pSize = 0xFFFFFFFF & size;

    return nn::ResultSuccess();
}

/*!--------------------------------------------------------------------------*
  Name:         GetFilterFilePath
  @brief        指定したパターンのフィルタデータを取得します。
  @return       結果が返ります。
 *---------------------------------------------------------------------------*/
nn::Result ContentsReader::GetFilterFilePath
            (char* filePath, ProfanityFilterPatternList filterPattern, uint32_t size) NN_NOEXCEPT
{
    if(filterPattern == NgcProfanityFilterPatternListCommon)
    {
        nn::ngc::strncpy(filePath,FILTER_COMMON_FORMAT,size);
    }
    else
    {
        int fileNo;
        for(fileNo = 0;fileNo < nn::ngc::ProfanityFilterPatternList_Max; fileNo++ )
        {
            if(filterPattern == (1 << fileNo))
            {
                break;
            }
        }

        nn::util::SNPrintf( filePath, size, FILTER_FILE_FORMAT, fileNo );
    }
    return nn::ResultSuccess();
}

/*!--------------------------------------------------------------------------*
  Name:         GetVersionFileName
  @brief        指定したパターンのフィルタデータを取得します。
  @return       結果が返ります。
 *---------------------------------------------------------------------------*/
nn::Result ContentsReader::GetVersionFilePath(char* filePath, uint32_t size) NN_NOEXCEPT
{
    nn::ngc::strncpy( filePath, VERSION_FILE_PATH, size );
    return nn::ResultSuccess();
}

/*!--------------------------------------------------------------------------*
  Name:         GetVersionData
  @brief        指定したパターンのフィルタデータを取得します。
  @return       結果が返ります。
 *---------------------------------------------------------------------------*/
nn::Result ContentsReader::GetVersionData(void* data, uint32_t size) NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char filePath[FilePathMax] = {};

    GetVersionFilePath(filePath, FilePathMax);

    nn::Result result = GetFileContent(data, filePath, size);
    NN_NGC_RETURN_IF_FAILED(result);
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    nn::Result result = nn::ngc::detail::GetFilterVersionData( data, size );
    NN_NGC_RETURN_IF_FAILED(result);
#else
#error "unsupported os"
#endif

    return nn::ResultSuccess();
}

/*!--------------------------------------------------------------------------*
  Name:         GetVersionDataSize
  @brief        指定したパターンのフィルタデータを取得します。
  @return       結果が返ります。
 *---------------------------------------------------------------------------*/
nn::Result ContentsReader::GetVersionDataSize(uint32_t* size) NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    char filePath[FilePathMax] = {};

    GetVersionFilePath(filePath, FilePathMax);

    nn::Result result = GetFileSize(size, filePath);
    NN_NGC_RETURN_IF_FAILED(result);
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    nn::Result result = nn::ngc::detail::GetFilterVersionDataLength( size );
    NN_NGC_RETURN_IF_FAILED(result);
#else
#error "unsupported os"
#endif

    return nn::ResultSuccess();
}


}   // namespace detail
}   // namespace ngc
}   // namespace nn

