﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/util/util_FormatString.h>
#include <nn/nfc/server/core/nfc_SaveData.h>
#include "nfp_ScopedFileStreamOpen.h"
#include "nfp_BackupDataStream.h"
#include "nfp_Util.h"
#include "nfp_Date.h"
#include <nn/nfc/server/util/nfc_ScopedMutexLock.h>

namespace nn {
namespace nfp {
namespace server {

namespace {
}

BackupDataStream::BackupDataStream() NN_NOEXCEPT
    : m_FileHandle(-1), m_IsInitialized(false)
{
    nn::os::InitializeMutex(&m_Mutex, true, 0);
}

BackupDataStream::~BackupDataStream() NN_NOEXCEPT
{
    if(m_IsInitialized)
    {
        if(m_FileHandle != -1)
        {
            // ファイルは自体は nn::nfc::server::core::SaveDataのデストラクタでまとめて閉じられる
            m_FileHandle = -1;
        }
    }

    nn::os::FinalizeMutex(&m_Mutex);
}

nn::Result BackupDataStream::Initialize() NN_NOEXCEPT
{
    nn::os::LockMutex(&m_Mutex);//Initialize したもののみが利用可能なようにロック
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    NN_ABORT_UNLESS(!m_IsInitialized);
    NN_ABORT_UNLESS(nn::nfc::server::core::SaveData::GetInstance().IsInitialized());

    // バックアップデータをオープンします。失敗時にはフォーマットします。
    NN_RESULT_TRY(CreateAndOpenBackupData())
        NN_RESULT_CATCH_ALL
        {
            nn::nfc::server::core::SaveData::Delete(nn::nfc::server::core::SaveData::FilePathNfpBackup);
            nn::Result result = CreateAndOpenBackupData();
            if(result.IsFailure())
            {
                nn::os::UnlockMutex(&m_Mutex);
                NN_RESULT_THROW(result);
            }
        }
    NN_RESULT_END_TRY

    m_IsInitialized = true;
    NN_RESULT_SUCCESS;
}

nn::Result BackupDataStream::Finalize() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    NN_ABORT_UNLESS(m_IsInitialized);

    if(m_FileHandle != -1)
    {
        nn::nfc::server::core::SaveData::GetInstance().Close(m_FileHandle);
        m_FileHandle = -1;
    }

    m_IsInitialized = false;
    nn::os::UnlockMutex(&m_Mutex);//Initialize したもののみが利用可能な状態を解除
    NN_RESULT_SUCCESS;
}

nn::Result BackupDataStream::Format() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    Finalize();

    nn::nfc::server::core::SaveData::Delete(nn::nfc::server::core::SaveData::FilePathNfpBackup);

    return Initialize();
}

nn::Result BackupDataStream::Write(const void* pData, size_t dataSize, const nn::nfc::TagId& tagId) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    nn::Result result;

    NN_ABORT_UNLESS(m_IsInitialized);

    // 対象のタグを検索し、見つからなかった場合には新規登録します。
    int index;
    BackupTocItem toc;
    bool isFoundEntry = true;
    NN_RESULT_TRY(FindEntry(&index, &toc, tagId))
        NN_RESULT_CATCH(nn::nfc::ResultTagNotFound)
        {
            isFoundEntry = false;

            // 新しいタグを登録するためヘッダを更新します。
            index = m_Header.nextEntryIndex;
            m_Header.nextEntryIndex = (m_Header.nextEntryIndex + 1) % BackupDataStream::BackupEntryMax;
            if (m_Header.entryNum < BackupDataStream::BackupEntryMax)
            {
                ++m_Header.entryNum;
            }
            m_Header.Recalculate();

            // ヘッダを書き込みます。
            NN_RESULT_DO(nn::nfc::server::core::SaveData::GetInstance().SetPosition(m_FileHandle, 0));
            NN_RESULT_DO(nn::nfc::server::core::SaveData::GetInstance().Write(m_FileHandle, &m_Header, sizeof(m_Header), true));

            // タグデータの索引を作成します。
            toc.Clear();
            toc.uidLength = tagId.length;
            std::memcpy(toc.uid, tagId.uid, toc.uidLength);
            toc.entryRegisterDate = Date::GetNow().GetTagDate();
            toc.Recalculate();

            // タグデータの索引を書き込みます。
            int64_t tocPos = BackupDataStream::BackupEntryTocPos + sizeof(BackupTocItem) * index;
            NN_RESULT_DO(nn::nfc::server::core::SaveData::GetInstance().SetPosition(m_FileHandle, tocPos));
            NN_RESULT_DO(nn::nfc::server::core::SaveData::GetInstance().Write(m_FileHandle, &toc, sizeof(toc), true));
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    int64_t entryPos = BackupDataStream::BackupEntryPos + BackupDataStream::BackupEntrySize * index;
    if (isFoundEntry)
    {
        // バックアップデータを読み込んで差分が存在しない場合は書き込みを行いません。
        // これは NAND の書き込み回数を減らし寿命を延ばすことを意図した処理です。
        std::unique_ptr<nn::Bit8[]> backupBuffer(new nn::Bit8[BackupDataStream::BackupEntrySize]);
        NN_RESULT_DO(nn::nfc::server::core::SaveData::GetInstance().SetPosition(m_FileHandle, entryPos));
        NN_RESULT_DO(nn::nfc::server::core::SaveData::GetInstance().Read(backupBuffer.get(), m_FileHandle, dataSize));
        if (std::memcmp(pData, backupBuffer.get(), dataSize) == 0)
        {
            NN_RESULT_SUCCESS;
        }
    }

    // エントリを書き込みます。
    NN_RESULT_DO(nn::nfc::server::core::SaveData::GetInstance().SetPosition(m_FileHandle, entryPos));
    NN_RESULT_DO(nn::nfc::server::core::SaveData::GetInstance().Write(m_FileHandle, pData, dataSize, true));

    // システムセーブデータに反映します。
    nn::nfc::server::core::SaveData::GetInstance().Close(m_FileHandle);//コミットするために一旦閉じる
    m_FileHandle = -1;
    NN_RESULT_DO(nn::nfc::server::core::SaveData::Commit());
    NN_RESULT_DO(nn::nfc::server::core::SaveData::GetInstance().Open(&m_FileHandle, nn::nfc::server::core::SaveData::FilePathNfpBackup, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write));
    NN_RESULT_SUCCESS;
}

nn::Result BackupDataStream::Read(void* pOutBuffer, size_t* pOutSize, size_t bufferSize) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    NN_ABORT_UNLESS(m_IsInitialized);

    // バックアップデータを読み込みます。
    NN_RESULT_DO(nn::nfc::server::core::SaveData::GetInstance().SetPosition(m_FileHandle, 0));
    NN_RESULT_DO(nn::nfc::server::core::SaveData::GetInstance().Read(pOutSize, pOutBuffer, m_FileHandle, bufferSize));
    NN_RESULT_SUCCESS;
}

nn::Result BackupDataStream::Write(const void* pData, size_t dataSize) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    NN_ABORT_UNLESS(m_IsInitialized);

    // バックアップデータを書き込みます。(コミット可能なサイズ単位で書き込みます。)
    const bool flush = true;
    int loop = static_cast<int>(dataSize / nn::nfc::server::core::SaveData::SystemSaveDataCommitSize);
    size_t mod = static_cast<size_t>(dataSize % nn::nfc::server::core::SaveData::SystemSaveDataCommitSize);
    loop += (mod == 0 ? 0 : 1);
    size_t writeSize = static_cast<size_t>(nn::nfc::server::core::SaveData::SystemSaveDataCommitSize);
    int64_t pos = 0;
    for(auto i = 0; i < loop; ++i)
    {
        if(i == loop - 1 && mod != 0)
        {
            writeSize = mod;
        }
        NN_RESULT_DO(nn::nfc::server::core::SaveData::GetInstance().SetPosition(m_FileHandle, pos));
        NN_RESULT_DO(nn::nfc::server::core::SaveData::GetInstance().Write(m_FileHandle, reinterpret_cast<const char*>(pData) + pos, writeSize, flush));
        nn::nfc::server::core::SaveData::GetInstance().Close(m_FileHandle);//コミットするために一旦閉じる
        m_FileHandle = -1;
        NN_RESULT_DO(nn::nfc::server::core::SaveData::Commit());
        NN_RESULT_DO(nn::nfc::server::core::SaveData::GetInstance().Open(&m_FileHandle, nn::nfc::server::core::SaveData::FilePathNfpBackup, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write));
        pos += writeSize;
    }

    NN_RESULT_SUCCESS;
}

nn::Result BackupDataStream::Read(void* pOutBuffer, size_t readAndBufferSize, const nn::nfc::TagId& tagId) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    int64_t    pos;

    NN_ABORT_UNLESS(m_IsInitialized);

    // バックアップデータを検索します。
    int index;
    BackupTocItem toc;
    NN_RESULT_TRY(FindEntry(&index, &toc, tagId))
        NN_RESULT_CATCH(nn::nfc::ResultTagNotFound)
        {
            return nn::nfc::ResultBackupReadFailed();
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    // バックアップデータを読み込みます。
    pos = BackupDataStream::BackupEntryPos + BackupEntrySize * index;
    NN_RESULT_DO(nn::nfc::server::core::SaveData::GetInstance().SetPosition(m_FileHandle, pos));
    return nn::nfc::server::core::SaveData::GetInstance().Read(pOutBuffer, m_FileHandle, readAndBufferSize);
}

bool BackupDataStream::HasBackupData(const nn::nfc::TagId& tagId) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    int index;
    BackupTocItem toc;
    if (FindEntry(&index, &toc, tagId).IsFailure())
    {
        return false;
    }
    return toc.Verify().IsSuccess();
}

nn::Result BackupDataStream::CreateAndOpenBackupData() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    bool   isCreated = true;

    // バックアップデータが存在しない場合には作成します。
    NN_RESULT_TRY(nn::nfc::server::core::SaveData::GetInstance().Create(nn::nfc::server::core::SaveData::FilePathNfpBackup, BackupDataStream::BackupTotalSize))
        NN_RESULT_CATCH(nn::nfc::server::core::ResultFsPathAlreadyExists)
        {
            isCreated = false;
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_THROW(nn::nfc::ResultBackupSystemError());
        }
    NN_RESULT_END_TRY

    // バックアップデータをオープンします。
    ScopedFileStreamOpen scopedFileStreamOpen(&m_FileHandle);
    NN_RESULT_DO(scopedFileStreamOpen.Initialize(nn::nfc::server::core::SaveData::FilePathNfpBackup, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write));

    // ファイルサイズを確認します。
    int64_t fileSize;

    NN_RESULT_DO(nn::nfc::server::core::SaveData::GetInstance().GetSize(&fileSize, m_FileHandle));

    if (fileSize != BackupDataStream::BackupTotalSize)
    {
        return nn::nfc::ResultBackupSystemError();
    }

    // ヘッダ部分を読み込んでキャッシュします。
    if (isCreated)
    {
        m_Header.Clear();
        NN_RESULT_DO(nn::nfc::server::core::SaveData::GetInstance().Write(m_FileHandle, &m_Header, sizeof(m_Header), false));
        NN_RESULT_DO(nn::nfc::server::core::SaveData::GetInstance().Flush(m_FileHandle));
        scopedFileStreamOpen.Finalize();//コミットするために一旦閉じる
        NN_RESULT_DO(nn::nfc::server::core::SaveData::Commit());
        NN_RESULT_DO(scopedFileStreamOpen.Initialize(nn::nfc::server::core::SaveData::FilePathNfpBackup, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write));
    }
    else
    {
        NN_RESULT_DO(nn::nfc::server::core::SaveData::GetInstance().Read(&m_Header, m_FileHandle, sizeof(m_Header)));
    }

    // ヘッダの有効性を検証します。
    NN_RESULT_DO(m_Header.Verify(BackupDataStream::BackupEntryMax));

    scopedFileStreamOpen.Release();//Openしたままにする
    NN_RESULT_SUCCESS;
}

nn::Result BackupDataStream::FindEntry(int* pOutIndex, BackupTocItem* pOutToc, const nn::nfc::TagId& tagId) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    size_t    size;

    // バックアップデータ内の索引の先頭位置までシークします。
    NN_RESULT_DO(nn::nfc::server::core::SaveData::GetInstance().SetPosition(m_FileHandle, BackupDataStream::BackupEntryTocPos));

    // 全ての索引から指定されたタグを検索します。
    int i = 0;
    while (i < m_Header.entryNum)
    {
        // バッファに読み込めるだけまとめて索引を読み込みます。
        std::unique_ptr<BackupTocItem[]> tocBuffer(new BackupTocItem[64]);
        NN_RESULT_DO(nn::nfc::server::core::SaveData::GetInstance().Read(&size, tocBuffer.get(), m_FileHandle, sizeof(BackupTocItem) * 64));
        int count = static_cast<int>(size / sizeof(BackupTocItem));

        // 読み込んだ索引に対して UID の照合を行います。
        for (int j = 0; j < count && i < m_Header.entryNum; ++i, ++j)
        {
            const BackupTocItem& item = tocBuffer[j];
            if (item.uidLength == tagId.length &&
                std::memcmp(item.uid, tagId.uid, tagId.length) == 0 &&
                item.Verify().IsSuccess())
            {
                *pOutIndex = i;
                *pOutToc   = item;
                NN_RESULT_SUCCESS;
            }
        }
    }
    return nn::nfc::ResultTagNotFound();
}

} // end of namespace server
} // end of namespace nfp
} // end of namespace nn
