﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkText.h> // for NN_TEXT
#include <nn/nfp/nfp_TagApi.h>
#include <nn/nfp/nfp_EventApi.h>
#include <nn/nfc/nfc_PrivateResult.h>
#include <nn/nfc/nfc_NfpResult.internal.h>
#include <nn/nfp/nfp_PrivateResult.h>

#include <nn/fs/detail/fs_AccessLog.h>

#include "client/nfp_CommonInterface.h"
#include "detail/nfp_Util.h"

#define NN_DETAIL_FS_ACCESS_LOG_NAMESPACE_NFP NN_DETAIL_FS_ACCESS_LOG_FORMAT_NAMESPACE(nfp)

namespace nn { namespace nfp {

namespace
{
}

nn::Result ListDevices(DeviceHandle* pOutBuffer, int* pOutCount, int bufferCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES(0 < bufferCount);

    int32_t count;
    nn::sf::OutArray<nn::nfc::DeviceHandle> devices(reinterpret_cast<nn::nfc::DeviceHandle*>(pOutBuffer), bufferCount);
    NN_RESULT_DO(NN_NFP_CLIENT_GET_COMMON_INTERFACE(ListDevices, (devices, &count)));
    *pOutCount = static_cast<int>(count);

    NN_RESULT_SUCCESS;
}

nn::Result StartDetection(const DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    NN_NFP_DETAIL_RETURN(NN_NFP_CLIENT_GET_COMMON_INTERFACE(StartDetection, (reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle))));
}

nn::Result StopDetection(const DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    NN_NFP_DETAIL_RETURN(NN_NFP_CLIENT_GET_COMMON_INTERFACE(StopDetection, (reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle))));
}

nn::Result Mount(const DeviceHandle& deviceHandle, ModelType modelType, MountTarget mountTarget) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(ModelType_Amiibo, modelType);
    NN_NFP_DETAIL_RETURN(
        NN_DETAIL_FS_ACCESS_LOG(
            NN_NFP_CLIENT_GET_COMMON_INTERFACE(
                Mount,
                (reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle), modelType, mountTarget)
            ),
            nullptr,
            NN_DETAIL_FS_ACCESS_LOG_NAMESPACE_NFP
            NN_DETAIL_FS_ACCESS_LOG_FORMAT_NFP_MOUNT_WITHTARGET(modelType, mountTarget)
        )
    );
}

nn::Result Mount(const DeviceHandle& deviceHandle, ModelType modelType) NN_NOEXCEPT
{
    return Mount(deviceHandle, modelType, MountTarget_All);
}

nn::Result Unmount(const DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    NN_NFP_DETAIL_RETURN(NN_NFP_CLIENT_GET_COMMON_INTERFACE(Unmount, (reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle))));
}

nn::Result OpenApplicationArea(const DeviceHandle& deviceHandle, nn::Bit32 accessId) NN_NOEXCEPT
{
    NN_NFP_DETAIL_RETURN(NN_NFP_CLIENT_GET_COMMON_INTERFACE_WITHOUT_SYSTEM(OpenApplicationArea, (reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle), accessId)));
}

nn::Result GetApplicationArea(void* pOutBuffer, size_t* pOutSize, const DeviceHandle& deviceHandle, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);
    NN_SDK_REQUIRES(0 < bufferSize);

    /*
      (SIGLO-22952)
      アプリケーション領域以上の領域に変更がかからないように、
      まずアプリケーション領域サイズを取得し、nn::sf::OutBuffer でサービスフレームワークに渡す領域を最低限の領域にします。
     */
    uint32_t applicationAreaSize;
    NN_NFP_DETAIL_RESULT_DO(NN_NFP_CLIENT_GET_COMMON_INTERFACE_WITHOUT_SYSTEM(GetApplicationAreaSize, (&applicationAreaSize, reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle))));

    size_t readableSize;
    if(bufferSize < applicationAreaSize)
    {
        readableSize = bufferSize;
    }
    else
    {
        readableSize = static_cast<size_t>(applicationAreaSize);
    }

    nn::sf::OutBuffer buffer(reinterpret_cast<char*>(pOutBuffer), readableSize);
    uint32_t outSize = 0;
    NN_NFP_DETAIL_RESULT_DO(NN_NFP_CLIENT_GET_COMMON_INTERFACE_WITHOUT_SYSTEM(GetApplicationArea, (buffer, &outSize, reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle))));
    *pOutSize = static_cast<size_t>(outSize);
    NN_RESULT_SUCCESS;
}

nn::Result SetApplicationArea(const DeviceHandle& deviceHandle, const void* pData, size_t dataSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pData);
    NN_SDK_REQUIRES(0 < dataSize);

    nn::sf::InBuffer data(reinterpret_cast<char*>(const_cast<void*>(pData)), dataSize);

    nn::Result result;
    const char* abortMessage = NN_TEXT("指定されたアプリケーション専用領域のサイズが不正です。");
    NN_UNUSED(abortMessage);
    result = NN_NFP_CLIENT_GET_COMMON_INTERFACE_WITHOUT_SYSTEM(SetApplicationArea, (reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle), data));
    NN_SDK_REQUIRES(!nn::nfp::ResultInvalidSize::Includes(result), abortMessage);
    NN_NFP_DETAIL_RETURN(result);
}

nn::Result RecreateApplicationArea(const DeviceHandle& deviceHandle, const ApplicationAreaCreateInfo& createInfo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(createInfo.pInitialData);
    NN_SDK_REQUIRES(0 < createInfo.initialDataSize);

    nn::sf::InBuffer initialData(reinterpret_cast<char*>(createInfo.pInitialData), createInfo.initialDataSize);

    nn::Result result;
    const char* abortMessage = NN_TEXT("指定されたアプリケーション専用領域のサイズが不正です。");
    NN_UNUSED(abortMessage);
    result = NN_NFP_CLIENT_GET_COMMON_INTERFACE_WITHOUT_SYSTEM(RecreateApplicationArea, (reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle), createInfo.accessId, initialData));
    NN_SDK_REQUIRES(!nn::nfp::ResultInvalidSize::Includes(result), abortMessage);
    NN_NFP_DETAIL_RETURN(result);
}

nn::Result Flush(const DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    NN_NFP_DETAIL_RETURN(
        NN_DETAIL_FS_ACCESS_LOG(
            NN_NFP_CLIENT_GET_COMMON_INTERFACE(
                Flush, (reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle))
            ),
            nullptr, NN_DETAIL_FS_ACCESS_LOG_NAMESPACE_NFP
        )
    );
}

nn::Result Restore(const DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    NN_NFP_DETAIL_RETURN(
        NN_DETAIL_FS_ACCESS_LOG(
            NN_NFP_CLIENT_GET_COMMON_INTERFACE(
                Restore, (reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle))
            ),
            nullptr, NN_DETAIL_FS_ACCESS_LOG_NAMESPACE_NFP
        )
    );
}

nn::Result CreateApplicationArea(const DeviceHandle& deviceHandle, const ApplicationAreaCreateInfo& createInfo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(createInfo.pInitialData);
    NN_SDK_REQUIRES(0 < createInfo.initialDataSize);

    nn::sf::InBuffer initialData(reinterpret_cast<char*>(createInfo.pInitialData), createInfo.initialDataSize);

    nn::Result result;
    const char* abortMessage = NN_TEXT("指定されたアプリケーション専用領域のサイズが不正です。");
    NN_UNUSED(abortMessage);
    // 以下のコードを CreateApplicationArea() から移動する場合は注意
    // NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_FUNCTION_NAME を使用して、明示的に "CreateApplicationArea" を指定してください
    result = NN_DETAIL_FS_ACCESS_LOG(
                 NN_NFP_CLIENT_GET_COMMON_INTERFACE_WITHOUT_SYSTEM(
                     CreateApplicationArea,
                     (
                         reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle),
                         createInfo.accessId, initialData
                     )
                 ),
                 nullptr,
                 NN_DETAIL_FS_ACCESS_LOG_NAMESPACE_NFP
                 NN_DETAIL_FS_ACCESS_LOG_FORMAT_NFP_CREATEAPPLICATIONAREA(createInfo)
             );
    NN_SDK_REQUIRES(!nn::nfp::ResultInvalidSize::Includes(result), abortMessage);
    NN_NFP_DETAIL_RETURN(result);
}

nn::Result GetTagInfo(TagInfo* pOutTagInfo, const DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutTagInfo);

    NN_NFP_DETAIL_RETURN(NN_NFP_CLIENT_GET_COMMON_INTERFACE(GetTagInfo, (reinterpret_cast<nn::nfc::TagInfo*>(pOutTagInfo), reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle))));
}

nn::Result GetRegisterInfo(RegisterInfo* pOutRegisterInfo, const DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutRegisterInfo);

    NN_NFP_DETAIL_RETURN(NN_NFP_CLIENT_GET_COMMON_INTERFACE(GetRegisterInfo, (pOutRegisterInfo, reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle))));
}

nn::Result GetCommonInfo(CommonInfo* pOutCommonInfo, const DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCommonInfo);

    NN_NFP_DETAIL_RETURN(NN_NFP_CLIENT_GET_COMMON_INTERFACE(GetCommonInfo, (pOutCommonInfo, reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle))));
}

nn::Result GetModelInfo(ModelInfo* pOutModelInfo, const DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutModelInfo);

    NN_NFP_DETAIL_RETURN(NN_NFP_CLIENT_GET_COMMON_INTERFACE(GetModelInfo, (pOutModelInfo, reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle))));
}

}}
