﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SdkText.h> // for NN_TEXT
#include <nn/nfp/nfp_PrivateApi.h>
#include <nn/nfp/nfp_ErrorApi.h>
#include <nn/applet/applet.h>
#include "client/nfp_CommonInterface.h"
#include "client/nfp_McuVersionData.h"
#include "detail/nfp_AmiiboSettings.h"
#include "detail/nfp_Util.h"

namespace nn { namespace nfp {

void InitializeSystem() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(!client::IsCreatedDebugInterface());
    NN_ABORT_UNLESS(!client::IsCreatedUserInterface());
    NN_ABORT_UNLESS(!client::IsCreatedSystemInterface());

    client::CreateSystemInterface();

    nn::sf::InArray<nn::xcd::McuVersionDataForNfc> requiredVersions(
        client::RequiredMcuVersionData,
        client::RequiredMcuVersionDataCount);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::applet::AppletResourceUserId id = nn::applet::GetAppletResourceUserId();
    NN_ABORT_UNLESS_RESULT_SUCCESS(client::GetSystemInterface()->InitializeNfp(id, 0, requiredVersions));
#else
    nn::applet::AppletResourceUserId id = nn::applet::AppletResourceUserId::GetInvalidId();
    NN_ABORT_UNLESS_RESULT_SUCCESS(client::GetSystemInterface()->InitializeNfp(id, 1, requiredVersions));
#endif

    SetErrorVisible(true);
}

void FinalizeSystem() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(client::GetSystemInterface()->FinalizeNfp());
    client::DestroySystemInterface();
}

nn::Result Format(const DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    NN_NFP_DETAIL_RETURN(NN_NFP_CLIENT_GET_COMMON_INTERFACE_WITHOUT_USER(Format, (reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle))));
}

nn::Result GetAdminInfo(AdminInfo* pOutAdminInfo, const DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutAdminInfo);

    NN_NFP_DETAIL_RETURN(NN_NFP_CLIENT_GET_COMMON_INTERFACE_WITHOUT_USER(GetAdminInfo, (pOutAdminInfo, reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle))));
}

nn::Result GetRegisterInfo(RegisterInfoPrivate* pOutRegisterInfo, const DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutRegisterInfo);

    NN_NFP_DETAIL_RETURN(NN_NFP_CLIENT_GET_COMMON_INTERFACE_WITHOUT_USER(GetRegisterInfoPrivate, (pOutRegisterInfo, reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle))));
}

nn::Result SetRegisterInfo(const DeviceHandle& deviceHandle, const RegisterInfoPrivate& regInfo) NN_NOEXCEPT
{
    nn::Result result;
    const char* nicknameMessage = NN_TEXT("指定されたニックネームが不正です。");
    const char* miiMessage = NN_TEXT("指定された Mii データが不正です。");
    NN_UNUSED(nicknameMessage);
    NN_UNUSED(miiMessage);
    result = NN_NFP_CLIENT_GET_COMMON_INTERFACE_WITHOUT_USER(SetRegisterInfoPrivate, (reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle), regInfo));
    NN_SDK_REQUIRES(!nn::nfp::ResultInvalidNickname::Includes(result), nicknameMessage);
    NN_SDK_REQUIRES(!nn::nfp::ResultInvalidMii::Includes(result), miiMessage);
    NN_NFP_DETAIL_RETURN(result);
}

nn::Result DeleteRegisterInfo(const DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    NN_NFP_DETAIL_RETURN(NN_NFP_CLIENT_GET_COMMON_INTERFACE_WITHOUT_USER(DeleteRegisterInfo, (reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle))));
}

nn::Result DeleteApplicationArea(const DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    NN_NFP_DETAIL_RETURN(NN_NFP_CLIENT_GET_COMMON_INTERFACE_WITHOUT_USER(DeleteApplicationArea, (reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle))));
}

nn::Result ExistsApplicationArea(bool* outValue, const DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);

    NN_NFP_DETAIL_RETURN(NN_NFP_CLIENT_GET_COMMON_INTERFACE_WITHOUT_USER(ExistsApplicationArea, (outValue, reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle))));
}

nn::Result StartNicknameAndOwnerSettings(TagInfo* pOutTagInfo, DeviceHandle* pOutDeviceHandle, bool* pOutIsRegistered, RegisterInfo* pOutRegisterInfo, const AmiiboSettingsStartParam& startParam) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutTagInfo);
    NN_SDK_REQUIRES_NOT_NULL(pOutDeviceHandle);
    NN_SDK_REQUIRES_NOT_NULL(pOutIsRegistered);
    NN_SDK_REQUIRES_NOT_NULL(pOutRegisterInfo);

    NN_SDK_REQUIRES(NN_NFP_CLIENT_CHECK_COMMON_INTERFACE_WITHOUT_USER());

    StartParamForAmiiboSettings* param;
    ReturnValueForAmiiboSettings* value;

    std::lock_guard<nn::os::Mutex> lock(nn::nfp::detail::RefMutexForAmiiboSettings());
    nn::nfp::detail::GetBufferForAmiiboSettings(&value, &param);

    std::memset(param, 0, sizeof(*param));
    param->version = nn::nfp::detail::StartParamVersionForAmiiboSettings;
    param->mode = static_cast<nn::Bit8>(AmiiboSettingsMode_NicknameAndOwnerSettings);
    param->infoFlags = static_cast<nn::Bit8>(AmiiboSettingsInfoFlags_DeviceHandle);
    param->optionFlags = startParam.optionFlags;
    param->deviceHandle = startParam.deviceHandle;
    std::memcpy(param->_debug, startParam._debug, sizeof(param->_debug));

    NN_RESULT_DO(nn::nfp::detail::StartAmiiboSettings(value, *param));

    *pOutTagInfo = value->tagInfo;
    *pOutDeviceHandle = value->deviceHandle;
    *pOutIsRegistered = ((value->infoFlags & AmiiboSettingsInfoFlags_RegisterInfo) != 0);
    if(*pOutIsRegistered)
    {
        *pOutRegisterInfo = value->registerInfo;
    }

    NN_RESULT_SUCCESS;
}

nn::Result StartGameDataEraser(TagInfo* pOutTagInfo, DeviceHandle* pOutDeviceHandle, const AmiiboSettingsStartParam& startParam) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutTagInfo);
    NN_SDK_REQUIRES_NOT_NULL(pOutDeviceHandle);

    NN_SDK_REQUIRES(NN_NFP_CLIENT_CHECK_COMMON_INTERFACE_WITHOUT_USER());

    StartParamForAmiiboSettings* param;
    ReturnValueForAmiiboSettings* value;

    std::lock_guard<nn::os::Mutex> lock(nn::nfp::detail::RefMutexForAmiiboSettings());
    nn::nfp::detail::GetBufferForAmiiboSettings(&value, &param);

    std::memset(param, 0, sizeof(*param));
    param->version = nn::nfp::detail::StartParamVersionForAmiiboSettings;
    param->mode = static_cast<nn::Bit8>(AmiiboSettingsMode_GameDataEraser);
    param->infoFlags = static_cast<nn::Bit8>(AmiiboSettingsInfoFlags_DeviceHandle);
    param->optionFlags = startParam.optionFlags;
    param->deviceHandle = startParam.deviceHandle;
    std::memcpy(param->_debug, startParam._debug, sizeof(param->_debug));

    NN_RESULT_DO(nn::nfp::detail::StartAmiiboSettings(value, *param));

    *pOutTagInfo = value->tagInfo;
    *pOutDeviceHandle = value->deviceHandle;

    NN_RESULT_SUCCESS;
}

nn::Result StartRestorer(TagInfo* pOutTagInfo, DeviceHandle* pOutDeviceHandle, const AmiiboSettingsStartParam& startParam) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutTagInfo);
    NN_SDK_REQUIRES_NOT_NULL(pOutDeviceHandle);

    NN_SDK_REQUIRES(NN_NFP_CLIENT_CHECK_COMMON_INTERFACE_WITHOUT_USER());

    StartParamForAmiiboSettings* param;
    ReturnValueForAmiiboSettings* value;

    std::lock_guard<nn::os::Mutex> lock(nn::nfp::detail::RefMutexForAmiiboSettings());
    nn::nfp::detail::GetBufferForAmiiboSettings(&value, &param);

    std::memset(param, 0, sizeof(*param));
    param->version = nn::nfp::detail::StartParamVersionForAmiiboSettings;
    param->mode = static_cast<nn::Bit8>(AmiiboSettingsMode_Restorer);
    param->infoFlags = static_cast<nn::Bit8>(AmiiboSettingsInfoFlags_DeviceHandle);
    param->optionFlags = startParam.optionFlags;
    param->deviceHandle = startParam.deviceHandle;
    std::memcpy(param->_debug, startParam._debug, sizeof(param->_debug));

    NN_RESULT_DO(nn::nfp::detail::StartAmiiboSettings(value, *param));

    *pOutTagInfo = value->tagInfo;
    *pOutDeviceHandle = value->deviceHandle;

    NN_RESULT_SUCCESS;
}

nn::Result StartFormatter(TagInfo* pOutTagInfo, DeviceHandle* pOutDeviceHandle, const AmiiboSettingsStartParam& startParam) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutTagInfo);
    NN_SDK_REQUIRES_NOT_NULL(pOutDeviceHandle);

    NN_SDK_REQUIRES(NN_NFP_CLIENT_CHECK_COMMON_INTERFACE_WITHOUT_USER());

    StartParamForAmiiboSettings* param;
    ReturnValueForAmiiboSettings* value;

    std::lock_guard<nn::os::Mutex> lock(nn::nfp::detail::RefMutexForAmiiboSettings());
    nn::nfp::detail::GetBufferForAmiiboSettings(&value, &param);

    std::memset(param, 0, sizeof(*param));
    param->version = nn::nfp::detail::StartParamVersionForAmiiboSettings;
    param->mode = static_cast<nn::Bit8>(AmiiboSettingsMode_Formatter);
    param->infoFlags = static_cast<nn::Bit8>(AmiiboSettingsInfoFlags_DeviceHandle);
    param->optionFlags = startParam.optionFlags;
    param->deviceHandle = startParam.deviceHandle;
    std::memcpy(param->_debug, startParam._debug, sizeof(param->_debug));

    NN_RESULT_DO(nn::nfp::detail::StartAmiiboSettings(value, *param));

    *pOutTagInfo = value->tagInfo;
    *pOutDeviceHandle = value->deviceHandle;

    NN_RESULT_SUCCESS;
}

}}
