﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nfc/server/core/nfc_CoreManager.h>
#include <nn/nfc/server/util/nfc_ScopedLockShared.h>
#include <nn/nfc/server/util/nfc_UtilUtil.h>
#include <nn/nfc/server/nfc_Manager.h>

namespace nn { namespace nfc {namespace server {

namespace
{
const int LinkMonitoringThreadRetryInterval = 50;

void SignalAvailabilityChangeEvent() NN_NOEXCEPT
{
    nn::nfc::server::core::Manager::GetInstance().SignalAvailabilityChangeEvent();
}
}

Manager::Manager() NN_NOEXCEPT : m_State(nn::nfc::State_None), m_IsRunningThread(false), m_InitializeCount(0), m_Nfp(this), m_NfcMifare(this), m_NfcPt(this)
{
    std::lock_guard<nn::os::ReaderWriterLock> lock(m_ReaderWriterLock);

    for(auto i = 0; i < nn::nfc::DeviceCountMax; ++i)
    {
        if(m_DeviceList[i])
        {
            m_DeviceList[i].reset(nullptr);
        }
    }

    m_LinkMonitoringThreadStack.reset(new nn::nfc::server::util::ThreadStack(4 * 1024));

    nn::os::InitializeMultiWait(&m_LinkMonitoringMultiWait);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateSystemEvent(&m_LinkEvent, nn::os::EventClearMode_ManualClear, false));
    nn::os::InitializeMultiWaitHolder(&m_LinkHolder, &m_LinkEvent);
    nn::os::LinkMultiWaitHolder(&m_LinkMonitoringMultiWait, &m_LinkHolder);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateSystemEvent(&m_TerminateLinkMonitoringEvent, nn::os::EventClearMode_ManualClear, false));
    nn::os::InitializeMultiWaitHolder(&m_TerminateLinkMonitoringHolder, &m_TerminateLinkMonitoringEvent);
    nn::os::LinkMultiWaitHolder(&m_LinkMonitoringMultiWait, &m_TerminateLinkMonitoringHolder);

    UpdateDevices();

    nn::os::CreateThread(&m_LinkMonitoringThread, LinkMonitoringThread, this, m_LinkMonitoringThreadStack->Get(), m_LinkMonitoringThreadStack->GetSize(), NN_SYSTEM_THREAD_PRIORITY(nfc, ServerManager));
    m_IsRunningThread = true;
    nn::os::SetThreadNamePointer(&m_LinkMonitoringThread, NN_SYSTEM_THREAD_NAME(nfc, ServerManager));
    nn::os::StartThread(&m_LinkMonitoringThread);

    nn::nfc::server::core::Manager::GetInstance().SetEvent(&m_LinkEvent);
}

Manager::~Manager() NN_NOEXCEPT
{
    m_IsRunningThread = false;
    nn::os::SignalSystemEvent(&m_TerminateLinkMonitoringEvent);

    nn::os::WaitThread(&m_LinkMonitoringThread);

    nn::nfc::server::core::Manager::GetInstance().SetEvent(nullptr);

    nn::os::DestroyThread(&m_LinkMonitoringThread);

    for(auto i = 0; i < nn::nfc::DeviceCountMax; ++i)
    {
        if(m_DeviceList[i])
        {
            m_DeviceList[i].reset(nullptr);
        }
    }

    nn::os::UnlinkAllMultiWaitHolder(&m_LinkMonitoringMultiWait);
    nn::os::FinalizeMultiWait(&m_LinkMonitoringMultiWait);

    nn::os::FinalizeMultiWaitHolder(&m_LinkHolder);
    nn::os::DestroySystemEvent(&m_LinkEvent);

    nn::os::FinalizeMultiWaitHolder(&m_TerminateLinkMonitoringHolder);
    nn::os::DestroySystemEvent(&m_TerminateLinkMonitoringEvent);
}

nn::Result Manager::CheckService(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    return nn::nfc::server::core::Manager::GetInstance().CheckService(service);
}

nn::Result Manager::CheckNfcEnabled(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    return nn::nfc::server::core::Manager::GetInstance().CheckNfcEnabled(service);
}

nn::Result Manager::CheckAwake() NN_NOEXCEPT
{
    return nn::nfc::server::core::Manager::GetInstance().CheckAwake();
}

nn::Result Manager::InitializePrivate(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    nn::nfc::server::core::Manager::GetInstance().Initialize(service);

    if(!IsInitialized())
    {
        for(auto i = 0; i < nn::nfc::DeviceCountMax; ++i)
        {
            if(m_DeviceList[i])
            {
                m_DeviceList[i]->Initialize();
            }
        }
    }

    ++m_InitializeCount;

    SetState(nn::nfc::State_Init);

    NN_RESULT_SUCCESS;
}

nn::Result Manager::Initialize(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    std::lock_guard<nn::os::ReaderWriterLock> lock(m_ReaderWriterLock);
    return InitializePrivate(service);
}

void Manager::FinalizePrivate(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(IsInitialized());

    --m_InitializeCount;

    if(!IsInitialized())
    {
        for(auto i = 0; i < nn::nfc::DeviceCountMax; ++i)
        {
            if(m_DeviceList[i])
            {
                if(m_DeviceList[i]->IsInitialized())
                {
                    m_DeviceList[i]->Finalize();
                }
            }
        }

        SetState(nn::nfc::State_None);
    }
    else if(nn::nfc::server::core::Manager::GetInstance().CheckService(service).IsSuccess())
    {
        for(auto i = 0; i < nn::nfc::DeviceCountMax; ++i)
        {
            if(m_DeviceList[i])
            {
                if(m_DeviceList[i]->IsInitialized())
                {
                    m_DeviceList[i]->Reset(service);
                }
            }
        }
    }

    nn::nfc::server::core::Manager::GetInstance().Finalize(service);
}

void Manager::Finalize(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    std::lock_guard<nn::os::ReaderWriterLock> lock(m_ReaderWriterLock);
    FinalizePrivate(service);
}

nn::Result Manager::ListDevices(nn::nfc::DeviceHandle* pOutBuffer, int* pOutCount, nn::nfc::server::core::Service* service, int bufferCount, bool forApi) NN_NOEXCEPT
{
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pOutBuffer);
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pOutCount);
    NN_NFC_SERVER_UTIL_REQUIRES(0 < bufferCount);

    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    NN_RESULT_DO(CheckAwake());
    NN_RESULT_DO(CheckNfcEnabled(service));
    NN_RESULT_DO(CheckService(service));

    int count = 0;
    for(auto& device : m_DeviceList)
    {
        if(!device || (forApi && device->IsHidden()))
        {
            continue;
        }
        pOutBuffer[count] = device->GetHandle();
        ++count;
        if(bufferCount <= count)
        {
            break;
        }
    }

    if(0 < count)
    {
        *pOutCount = count;
        NN_RESULT_SUCCESS;
    }

    return nn::nfc::ResultNfcDeviceNotFound();
}

nn::Result Manager::StartDetection(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, nn::nfc::NfcProtocol protocolFilter) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(this, service, deviceHandle, StartDetection(service, protocolFilter));
}

nn::Result Manager::StopDetection(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(this, service, deviceHandle, StopDetection(service));
}

nn::Result Manager::GetTagInfo(nn::nfc::TagInfo* pOutTagInfo, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(this, service, deviceHandle, GetTagInfo(pOutTagInfo, service));
}

State Manager::GetState(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    NN_UNUSED(service);
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    return m_State;
}

nn::nfc::server::Device::State Manager::GetDeviceState(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);
    Device* device;
    nn::Result result = GetDevice(&device, service, deviceHandle, false);
    if(result.IsFailure())
    {
        return nn::nfc::server::Device::State_Unexpected;
    }

    return device->GetState();
}

nn::Result Manager::GetActivateEventHandle(nn::os::NativeHandle* handle, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    std::unique_ptr<nn::nfc::DeviceHandle []> devices(new nn::nfc::DeviceHandle[nn::nfc::DeviceCountMax]);
    int outCount;

    auto resultList = ListDevices(devices.get(), &outCount, service, nn::nfc::DeviceCountMax, false);
    if(resultList.IsFailure())
    {
        outCount = 0;
    }

    auto resultGet = service->GetActivateEventHandle(handle, deviceHandle, devices.get(), outCount);
    NN_RESULT_DO(resultList);
    return resultGet;
}

nn::Result Manager::GetDeactivateEventHandle(nn::os::NativeHandle* handle, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    std::unique_ptr<nn::nfc::DeviceHandle []> devices(new nn::nfc::DeviceHandle[nn::nfc::DeviceCountMax]);
    int outCount;

    auto resultList = ListDevices(devices.get(), &outCount, service, nn::nfc::DeviceCountMax, false);
    if(resultList.IsFailure())
    {
        outCount = 0;
    }

    auto resultGet = service->GetDeactivateEventHandle(handle, deviceHandle, devices.get(), outCount);
    NN_RESULT_DO(resultList);
    return resultGet;
}

nn::Result Manager::GetAvailabilityChangeEventHandle(nn::os::NativeHandle* handle, nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    return service->GetAvailabilityChangeEventHandle(handle);
}

nn::Result Manager::InitializeSystem(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    std::lock_guard<nn::os::ReaderWriterLock> lock(m_ReaderWriterLock);
    NN_RESULT_DO(InitializePrivate(service));
    NN_RESULT_SUCCESS;
}

void Manager::FinalizeSystem(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    std::lock_guard<nn::os::ReaderWriterLock> lock(m_ReaderWriterLock);
    FinalizePrivate(service);
}

nn::Result Manager::GetNpadId(nn::hid::NpadIdType* pOutNpadId, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(this, service, deviceHandle, GetNpadId(pOutNpadId, service));
}

nn::Result Manager::Mount(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, nn::nfp::ModelType modelType, nn::nfp::MountTarget mountTarget) NN_NOEXCEPT
{
    return m_Nfp.Mount(service, deviceHandle, modelType, mountTarget);
}

nn::Result Manager::Unmount(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    return m_Nfp.Unmount(service, deviceHandle);
}

nn::Result Manager::OpenApplicationArea(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, nn::Bit32 accessId) NN_NOEXCEPT
{
    return m_Nfp.OpenApplicationArea(service, deviceHandle, accessId);
}

nn::Result Manager::GetApplicationArea(void* pOutBuffer, size_t* pOutSize, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, size_t bufferSize) NN_NOEXCEPT
{
    return m_Nfp.GetApplicationArea(pOutBuffer, pOutSize, service, deviceHandle, bufferSize);
}

nn::Result Manager::SetApplicationArea(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const void* pData, size_t dataSize) NN_NOEXCEPT
{
    return m_Nfp.SetApplicationArea(service, deviceHandle, pData, dataSize);
}

nn::Result Manager::RecreateApplicationArea(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const nn::nfp::ApplicationAreaCreateInfo& createInfo) NN_NOEXCEPT
{
    return m_Nfp.RecreateApplicationArea(service, deviceHandle, createInfo);
}

nn::Result Manager::Flush(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    return m_Nfp.Flush(service, deviceHandle);
}

nn::Result Manager::Restore(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    return m_Nfp.Restore(service, deviceHandle);
}

nn::Result Manager::CreateApplicationArea(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const nn::nfp::ApplicationAreaCreateInfo& createInfo) NN_NOEXCEPT
{
    return m_Nfp.CreateApplicationArea(service, deviceHandle, createInfo);
}

nn::Result Manager::GetRegisterInfo(nn::nfp::RegisterInfo* pOutRegisterInfo, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    return m_Nfp.GetRegisterInfo(pOutRegisterInfo, service, deviceHandle);
}

nn::Result Manager::GetCommonInfo(nn::nfp::CommonInfo* pOutCommonInfo, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    return m_Nfp.GetCommonInfo(pOutCommonInfo, service, deviceHandle);
}

nn::Result Manager::GetModelInfo(nn::nfp::ModelInfo* pOutModelInfo, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    return m_Nfp.GetModelInfo(pOutModelInfo, service, deviceHandle);
}

nn::Result Manager::Format(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    return m_Nfp.Format(service, deviceHandle);
}

nn::Result Manager::GetAdminInfo(nn::nfp::AdminInfo* pOutAdminInfo, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    return m_Nfp.GetAdminInfo(pOutAdminInfo, service, deviceHandle);
}

nn::Result Manager::GetRegisterInfo(nn::nfp::RegisterInfoPrivate* pOutRegisterInfo, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    return m_Nfp.GetRegisterInfo(pOutRegisterInfo, service, deviceHandle);
}

nn::Result Manager::SetRegisterInfo(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const nn::nfp::RegisterInfoPrivate& regInfo) NN_NOEXCEPT
{
    return m_Nfp.SetRegisterInfo(service, deviceHandle, regInfo);
}

nn::Result Manager::DeleteRegisterInfo(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    return m_Nfp.DeleteRegisterInfo(service, deviceHandle);
}

nn::Result Manager::DeleteApplicationArea(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    return m_Nfp.DeleteApplicationArea(service, deviceHandle);
}

nn::Result Manager::ExistsApplicationArea(bool* outValue, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    return m_Nfp.ExistsApplicationArea(outValue, service, deviceHandle);
}

nn::Result Manager::GetAll(nn::nfp::NfpData* pOutNfpData, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    return m_Nfp.GetAll(pOutNfpData, service, deviceHandle);
}

nn::Result Manager::SetAll(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const nn::nfp::NfpData& nfpData) NN_NOEXCEPT
{
    return m_Nfp.SetAll(service, deviceHandle, nfpData);
}

nn::Result Manager::FlushDebug(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    return m_Nfp.FlushDebug(service, deviceHandle);
}

nn::Result Manager::BreakTag(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, nn::nfp::BreakType breakType) NN_NOEXCEPT
{
    return m_Nfp.BreakTag(service, deviceHandle, breakType);
}

nn::Result Manager::ReadBackupData(void* pOutBuffer, size_t* pOutSize, nn::nfc::server::core::Service* service, size_t bufferSize) NN_NOEXCEPT
{
    return m_Nfp.ReadBackupData(pOutBuffer, pOutSize, service, bufferSize);
}

nn::Result Manager::WriteBackupData(nn::nfc::server::core::Service* service, const void* pData, size_t dataSize) NN_NOEXCEPT
{
    return m_Nfp.WriteBackupData(service, pData, dataSize);
}

nn::Result Manager::WriteNtf(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const void* pData, size_t dataSize, nn::nfp::NtfWriteType ntfWriteType) NN_NOEXCEPT
{
    return m_Nfp.WriteNtf(service, deviceHandle, pData, dataSize, ntfWriteType);
}

size_t Manager::GetApplicationAreaSize() NN_NOEXCEPT
{
    return m_Nfp.GetApplicationAreaSize();
}

nn::Result Manager::ReadMifare(nn::nfc::MifareReadBlockData* pOutBlockData, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const nn::nfc::MifareReadBlockParameter* pBlockParameter, size_t blockCount) NN_NOEXCEPT
{
    return m_NfcMifare.Read(pOutBlockData, service, deviceHandle, pBlockParameter, blockCount);
}

nn::Result Manager::WriteMifare(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const nn::nfc::MifareWriteBlockParameter* pBlockParameter, size_t blockCount) NN_NOEXCEPT
{
    return m_NfcMifare.Write(service, deviceHandle, pBlockParameter, blockCount);
}

nn::Result Manager::SendCommandByPassThrough(void* pOutBuffer, size_t* pOutSize, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const void* pData, size_t dataSize, size_t bufferSize, nn::TimeSpan timeout) NN_NOEXCEPT
{
    return m_NfcPt.SendCommand(pOutBuffer, pOutSize, service, deviceHandle, pData, dataSize, bufferSize, timeout);
}

nn::Result Manager::KeepPassThroughSession(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    return m_NfcPt.KeepSession(service, deviceHandle);
}

nn::Result Manager::ReleasePassThroughSession(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    return m_NfcPt.ReleaseSession(service, deviceHandle);
}

void Manager::SetState(State state) NN_NOEXCEPT
{
    m_State = state;
}

nn::Result Manager::GetDevice(Device** pOutDevice, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, bool check) NN_NOEXCEPT
{
    if(check)
    {
        NN_RESULT_DO(CheckAwake());
        NN_RESULT_DO(CheckNfcEnabled(service));
        NN_RESULT_DO(CheckService(service));
    }

    for(auto& device : m_DeviceList)
    {
        if(!device)
        {
            continue;
        }

        if(nn::nfc::server::util::IsEqualDeviceHandle(device->GetHandle(), deviceHandle))
        {
            *pOutDevice = device.get();
            NN_RESULT_SUCCESS;
        }
    }

    return nn::nfc::ResultNfcDeviceNotFound();
}

nn::Result Manager::GetDevice(Device** pOutDevice, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    return GetDevice(pOutDevice, service, deviceHandle, true);
}

void Manager::LinkMonitoringThread(void* arg) NN_NOEXCEPT
{
    Manager* manager = reinterpret_cast<Manager*>(arg);
    bool retryFlag = false;

    while (NN_STATIC_CONDITION(true))
    {
        // デバイス情報の更新を待機
        nn::os::MultiWaitHolderType* pHolder;
        if(retryFlag)
        {
            pHolder = nn::os::TimedWaitAny(&manager->m_LinkMonitoringMultiWait, nn::TimeSpan::FromMilliSeconds(LinkMonitoringThreadRetryInterval));
        }
        else
        {
            pHolder = nn::os::WaitAny(&manager->m_LinkMonitoringMultiWait);
        }
        {
            if (manager->m_IsRunningThread == false || pHolder == &manager->m_TerminateLinkMonitoringHolder)
            {
                break;
            }
            else
            {
                if (pHolder == &manager->m_LinkHolder || pHolder == nullptr)
                {
                    if(pHolder == &manager->m_LinkHolder)
                    {
                        nn::os::ClearSystemEvent(&manager->m_LinkEvent);
                    }
                    if(!manager->m_ReaderWriterLock.try_lock())
                    {
                        retryFlag = true;
                        continue;
                    }
                    manager->UpdateDevices();
                    SignalAvailabilityChangeEvent();
                    manager->m_ReaderWriterLock.unlock();
                    retryFlag = false;
                }
                else
                {
                    for(auto i = 0; i < nn::nfc::DeviceCountMax; ++i)
                    {
                        if(manager->m_DeviceList[i])
                        {
                            manager->m_DeviceList[i]->LoopEventThreadFunction(pHolder);
                        }
                    }
                }
            }
        }
    }
}

nn::Result Manager::UpdateDevices() NN_NOEXCEPT
{
    int count = 0;
    std::unique_ptr<uint64_t[]> deviceUniqueId(new uint64_t[nn::nfc::DeviceCountMax]);
    std::unique_ptr<nn::nfc::DeviceHandle[]> buffer(new nn::nfc::DeviceHandle[nn::nfc::DeviceCountMax]);
    auto result = nn::nfc::server::core::Manager::GetInstance().ListDevices(deviceUniqueId.get(), buffer.get(), &count, nn::nfc::server::core::Service::NfcProcessService, nn::nfc::DeviceCountMax);
    if(result.IsFailure())
    {
        count = 0;
    }

    //不要なデバイスを削除
    for(auto i = 0; i < nn::nfc::DeviceCountMax; ++i)
    {
        if(!m_DeviceList[i])
        {
            continue;
        }

        bool find = false;
        for(auto j = 0; j < count; ++j)
        {
            if(deviceUniqueId[j] == m_DeviceList[i]->GetUniqueId()
               && nn::nfc::server::util::IsEqualDeviceHandle(buffer[j], m_DeviceList[i]->GetHandle()))
            {
                find = true;
                break;
            }
        }

        if(!find)
        {
            if(m_DeviceList[i])
            {
                m_DeviceList[i].reset(nullptr);
            }
        }
    }

    for(auto i = 0; i < count; ++i)
    {
        nn::nfc::DeviceHandle handle;
        handle = buffer[i];

        //既に見つかっているか
        bool find = false;
        for(auto j = 0; j < nn::nfc::DeviceCountMax; ++j)
        {
            if(!m_DeviceList[j])
            {
                continue;
            }

            if(deviceUniqueId[i] == m_DeviceList[j]->GetUniqueId()
               && nn::nfc::server::util::IsEqualDeviceHandle(handle, m_DeviceList[j]->GetHandle()))
            {
                find = true;
                break;
            }
        }

        //なければ追加
        if(!find)
        {
            for(auto j = 0; j < nn::nfc::DeviceCountMax; ++j)
            {
                //空いているところを探して
                if(!m_DeviceList[j])
                {
                    m_DeviceList[j].reset(new Device(deviceUniqueId[i], handle, &m_LinkMonitoringMultiWait));
                    if(IsInitialized())
                    {
                        m_DeviceList[j]->Initialize();
                    }
                    break;
                }
            }
        }
    }

    return result;
}

bool Manager::IsInitialized() NN_NOEXCEPT
{
    return 0 < m_InitializeCount;
}

nn::Result Manager::BeforeAccessDevice(Device** pOutDevice, nn::xcd::DeviceHandle* pOutXcdDeviceHandle, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    NN_RESULT_DO(GetDevice(pOutDevice, service, deviceHandle));
    return (*pOutDevice)->GetXcdDeviceHandle(pOutXcdDeviceHandle, service);
}

nn::Result Manager::AfterAccessDevice(nn::nfc::server::core::Service* service, Device* device, nn::xcd::DeviceHandle xcdDeviceHandle, nn::Result result) NN_NOEXCEPT
{
    if(result.IsFailure())
    {
        NN_RESULT_DO(CheckAwake());
        NN_RESULT_DO(CheckNfcEnabled(service));
        NN_RESULT_DO(CheckService(service));
        nn::xcd::DeviceHandle handle;
        NN_RESULT_DO(device->GetXcdDeviceHandle(&handle, service));
        if(handle != xcdDeviceHandle)
        {
            return nn::nfc::ResultNfcDeviceNotFound();
        }

        if(nn::nfc::ResultNfcDeviceError::Includes(result)
           || nn::nfc::ResultNeedUpdate::Includes(result)
           || nn::nfc::ResultNeedCharge::Includes(result))
        {
            device->Hide();
        }
    }

    return result;
}

}}}  // namespace nn::nfc::server
