﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nfc/server/core/nfc_CoreManager.h>
#include <nn/nfc/server/nfc_Device.h>
#include <nn/nfc/server/util/nfc_UtilUtil.h>
#include <nn/nfc/server/util/nfc_ScopedMutexLock.h>

namespace nn { namespace nfc {namespace server {

namespace
{
}

Device::Device(uint64_t uniqueId, const nn::nfc::DeviceHandle& deviceHandle, nn::os::MultiWaitType* eventMultiWait) NN_NOEXCEPT
:   m_State(nn::nfc::server::Device::State_Unexpected), m_UniqueId(uniqueId), m_Handle(deviceHandle), m_pEventMultiWait(eventMultiWait), m_IsRunningThread(false), m_InitializedCount(0), m_ActiveService(nullptr), m_Nfp(this), m_NfcMifare(this), m_NfcPt(this)
{
    nn::os::InitializeMutex(&m_Mutex, true, 0);

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    //イベントスレッド終了用イベント
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateSystemEvent(&m_EventTerminateEvent,
                                                             nn::os::EventClearMode_ManualClear,
                                                             false));
    nn::os::InitializeMultiWaitHolder(&m_EventTerminateHolder, &m_EventTerminateEvent);
    nn::os::LinkMultiWaitHolder(m_pEventMultiWait, &m_EventTerminateHolder);

    //タグ検出、喪失イベント
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateSystemEvent(&m_DetectEvent,
                                                             nn::os::EventClearMode_ManualClear,
                                                             false));
    nn::os::InitializeMultiWaitHolder(&m_DetectEventHolder, &m_DetectEvent);
    nn::os::LinkMultiWaitHolder(m_pEventMultiWait, &m_DetectEventHolder);

    //タグ検出・喪失以外のイベント
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateSystemEvent(&m_Event,
                                                             nn::os::EventClearMode_ManualClear,
                                                             false));
    nn::os::InitializeMultiWaitHolder(&m_EventHolder, &m_Event);
    nn::os::LinkMultiWaitHolder(m_pEventMultiWait, &m_EventHolder);

    //リセットイベント
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateSystemEvent(&m_ResetEvent,
                                                             nn::os::EventClearMode_ManualClear,
                                                             false));
    nn::os::InitializeMultiWaitHolder(&m_ResetEventHolder, &m_ResetEvent);
    nn::os::LinkMultiWaitHolder(m_pEventMultiWait, &m_ResetEventHolder);


    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateSystemEvent(&m_AccessFinishEvent,
                                                             nn::os::EventClearMode_ManualClear,
                                                             false));

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateSystemEvent(&m_AccessResetEvent,
                                                             nn::os::EventClearMode_ManualClear,
                                                             false));

    m_IsRunningThread = true;

    nn::nfc::server::core::Manager::GetInstance().SetEvent(&m_Event, &m_DetectEvent, &m_ResetEvent, m_Handle);

    SetState(nn::nfc::server::Device::State_Init);
}

Device::~Device() NN_NOEXCEPT
{
    if(m_State == nn::nfc::server::Device::State_Active
       || m_State == nn::nfc::server::Device::State_Mount
       || m_State == nn::nfc::server::Device::State_Keep)
    {
        Unmount();
        ReleasePassThroughSession();

        //TAGが喪失したら
        SignalDeactivateEvent();
    }

    SetState(nn::nfc::server::Device::State_Unexpected);
    m_ActiveService = nullptr;

    m_IsRunningThread = false;
    nn::os::SignalSystemEvent(&m_EventTerminateEvent);

    nn::nfc::server::core::Manager::GetInstance().SetEvent(nullptr, nullptr, nullptr, m_Handle);

    //イベントスレッド終了用イベント
    nn::os::UnlinkMultiWaitHolder(&m_EventTerminateHolder);
    nn::os::FinalizeMultiWaitHolder(&m_EventTerminateHolder);
    nn::os::DestroySystemEvent(&m_EventTerminateEvent);

    //タグ検出、喪失イベント
    nn::os::UnlinkMultiWaitHolder(&m_DetectEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_DetectEventHolder);
    nn::os::DestroySystemEvent(&m_DetectEvent);

    //タグ検出・喪失以外のイベント
    nn::os::UnlinkMultiWaitHolder(&m_EventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_EventHolder);
    nn::os::DestroySystemEvent(&m_Event);

    //リセットイベント
    nn::os::UnlinkMultiWaitHolder(&m_ResetEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_ResetEventHolder);
    nn::os::DestroySystemEvent(&m_ResetEvent);

    //
    nn::os::DestroySystemEvent(&m_AccessFinishEvent);
    nn::os::DestroySystemEvent(&m_AccessResetEvent);

    nn::os::FinalizeMutex(&m_Mutex);
}

bool Device::IsInitialized() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    return m_InitializedCount > 0;
}

void Device::Initialize() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    NN_ABORT_UNLESS(!IsInitialized());

    SetState(nn::nfc::server::Device::State_Init);
    ++m_InitializedCount;
}

void Device::Finalize() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    NN_ABORT_UNLESS(IsInitialized());

    --m_InitializedCount;

    if(m_State == nn::nfc::server::Device::State_Active
       || m_State == nn::nfc::server::Device::State_Mount
       || m_State == nn::nfc::server::Device::State_Keep)
    {
        Unmount();
        ReleasePassThroughSession();

        //TAGが喪失したら
        SignalDeactivateEvent();
    }

    SetState(nn::nfc::server::Device::State_Unexpected);
    m_ActiveService = nullptr;
}

void Device::Reset(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(!IsInitialized())
    {
        return;
    }

    if(m_State == nn::nfc::server::Device::State_Search
       || m_State == nn::nfc::server::Device::State_Active
       || m_State == nn::nfc::server::Device::State_Mount
       || m_State == nn::nfc::server::Device::State_Keep
       || m_State == nn::nfc::server::Device::State_Deactive)
    {
        if(m_ActiveService != nullptr && service != nullptr && m_ActiveService != service)
        {
            // 別のライブラリでデバイスを利用している場合はリセットしない
            return;
        }
    }

    switch(m_State)
    {
    case nn::nfc::server::Device::State_Active:
    case nn::nfc::server::Device::State_Mount:
    case nn::nfc::server::Device::State_Keep:
        {
            Unmount();
            ReleasePassThroughSession();

            SignalDeactivateEvent();
        }
        NN_FALL_THROUGH;
    case nn::nfc::server::Device::State_Search:
    case nn::nfc::server::Device::State_Deactive:
        {
            SetState(nn::nfc::server::Device::State_Init);
            m_ActiveService = nullptr;
        }
        break;
    default:
        break;
    }
}

nn::nfc::server::Device::State Device::GetState() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    return m_State;
}

void Device::SetState(nn::nfc::server::Device::State state) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    m_State = state;
}

void Device::SetSelectedTagId(const nn::nfc::TagId& tagId) NN_NOEXCEPT
{
    m_SelectedTagId = tagId;
}

void Device::SignalActivateEvent() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    nn::nfc::server::core::Manager::GetInstance().SignalEvent(m_ActiveService, m_Handle, true);
}

void Device::SignalDeactivateEvent() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    nn::nfc::server::core::Manager::GetInstance().SignalEvent(m_ActiveService, m_Handle, false);
}

void Device::ClearDetectEvent(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(service == nullptr)
    {
        return;
    }

    service->ClearDetectEvent(m_Handle);
}

nn::nfc::TagId Device::GetSelectedTagId() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    return m_SelectedTagId;
}

uint64_t Device::GetUniqueId() NN_NOEXCEPT
{
    return m_UniqueId;
}

nn::nfc::DeviceHandle Device::GetHandle() NN_NOEXCEPT
{
    return m_Handle;
}

nn::Result Device::StartDetection(nn::nfc::server::core::Service* service, nn::nfc::NfcProtocol protocolFilter) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    // StartDetection が呼ばれた場合、必ずイベントをクリアする(SIGLO-41450)
    ClearDetectEvent(service);

    switch(m_State)
    {
    case nn::nfc::server::Device::State_Init:
    case nn::nfc::server::Device::State_Deactive:
        {
            NN_RESULT_DO(nn::nfc::server::core::Manager::GetInstance().StartDiscovery(service, m_Handle, protocolFilter));
            m_ActiveService = service;
            SetState(nn::nfc::server::Device::State_Search);
        }
        NN_RESULT_SUCCESS;
    default:
        return nn::nfc::ResultInvalidDeviceState();
    }
}

nn::Result Device::StopDetection(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    switch(m_State)
    {
    case nn::nfc::server::Device::State_Active:
    case nn::nfc::server::Device::State_Mount:
    case nn::nfc::server::Device::State_Keep:
        {
            Unmount();
            ReleasePassThroughSession();

            SignalDeactivateEvent();
        }
        NN_FALL_THROUGH;
    case nn::nfc::server::Device::State_Search:
    case nn::nfc::server::Device::State_Deactive:
        {
            nn::Result result = nn::nfc::server::core::Manager::GetInstance().StopDiscovery(service, m_Handle);
            SetState(nn::nfc::server::Device::State_Init);
            m_ActiveService = nullptr;
            NN_RESULT_DO(result);
        }
        NN_RESULT_SUCCESS;
    default:
        return nn::nfc::ResultInvalidDeviceState();
    }
}

nn::Result Device::GetTagInfo(nn::nfc::TagInfo* pOutTagInfo, nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pOutTagInfo);

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    switch(m_State)
    {
    case nn::nfc::server::Device::State_Active:
    case nn::nfc::server::Device::State_Mount:
    case nn::nfc::server::Device::State_Keep:
        {
            NN_RESULT_DO(nn::nfc::server::core::Manager::GetInstance().GetTagInfo(pOutTagInfo, service, m_Handle, m_SelectedTagId));
        }
        NN_RESULT_SUCCESS;
    case nn::nfc::server::Device::State_Deactive:
        return nn::nfc::ResultTagNotFound();
    default:
        return nn::nfc::ResultInvalidDeviceState();
    }
}

nn::Result Device::GetNpadId(nn::hid::NpadIdType* pOutNpadId, nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pOutNpadId);

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    return nn::nfc::server::core::Manager::GetInstance().GetNpadId(pOutNpadId, service, m_Handle);
}

nn::Result Device::Mount(nn::nfc::server::core::Service* service, nn::nfp::ModelType modelType, nn::nfp::MountTarget mountTarget) NN_NOEXCEPT
{
    return m_Nfp.Mount(service, modelType, mountTarget);
}

nn::Result Device::Unmount() NN_NOEXCEPT
{
    return m_Nfp.Unmount();
}

nn::Result Device::OpenApplicationArea(nn::Bit32 accessId) NN_NOEXCEPT
{
    return m_Nfp.OpenApplicationArea(accessId);
}

nn::Result Device::GetApplicationArea(void* pOutBuffer, size_t* pOutSize, size_t bufferSize) NN_NOEXCEPT
{
    return m_Nfp.GetApplicationArea(pOutBuffer, pOutSize, bufferSize);
}

nn::Result Device::SetApplicationArea(const void* pData, size_t dataSize) NN_NOEXCEPT
{
    return m_Nfp.SetApplicationArea(pData, dataSize);
}

nn::Result Device::RecreateApplicationArea(const nn::nfp::ApplicationAreaCreateInfo& createInfo) NN_NOEXCEPT
{
    return m_Nfp.RecreateApplicationArea(createInfo);
}

nn::Result Device::Flush(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    return m_Nfp.Flush(service);
}

nn::Result Device::Restore(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    return m_Nfp.Restore(service);
}

nn::Result Device::CreateApplicationArea(nn::nfc::server::core::Service* service, const nn::nfp::ApplicationAreaCreateInfo& createInfo) NN_NOEXCEPT
{
    return m_Nfp.CreateApplicationArea(service, createInfo);
}

nn::Result Device::GetRegisterInfo(nn::nfp::RegisterInfo* pOutRegisterInfo) NN_NOEXCEPT
{
    return m_Nfp.GetRegisterInfo(pOutRegisterInfo);
}

nn::Result Device::GetCommonInfo(nn::nfp::CommonInfo* pOutCommonInfo) NN_NOEXCEPT
{
    return m_Nfp.GetCommonInfo(pOutCommonInfo);
}

nn::Result Device::GetModelInfo(nn::nfp::ModelInfo* pOutModelInfo) NN_NOEXCEPT
{
    return m_Nfp.GetModelInfo(pOutModelInfo);
}

nn::Result Device::Format(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    return m_Nfp.Format(service);
}

nn::Result Device::GetAdminInfo(nn::nfp::AdminInfo* pOutAdminInfo) NN_NOEXCEPT
{
    return m_Nfp.GetAdminInfo(pOutAdminInfo);
}

nn::Result Device::GetRegisterInfo(nn::nfp::RegisterInfoPrivate* pOutRegisterInfo) NN_NOEXCEPT
{
    return m_Nfp.GetRegisterInfo(pOutRegisterInfo);
}

nn::Result Device::SetRegisterInfo(const nn::nfp::RegisterInfoPrivate& regInfo) NN_NOEXCEPT
{
    return m_Nfp.SetRegisterInfo(regInfo);
}

nn::Result Device::DeleteRegisterInfo(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    return m_Nfp.DeleteRegisterInfo(service);
}

nn::Result Device::DeleteApplicationArea(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    return m_Nfp.DeleteApplicationArea(service);
}

nn::Result Device::ExistsApplicationArea(bool* outValue) NN_NOEXCEPT
{
    return m_Nfp.ExistsApplicationArea(outValue);
}

nn::Result Device::GetAll(nn::nfp::NfpData* pOutNfpData) NN_NOEXCEPT
{
    return m_Nfp.GetAll(pOutNfpData);
}

nn::Result Device::SetAll(const nn::nfp::NfpData& nfpData) NN_NOEXCEPT
{
    return m_Nfp.SetAll(nfpData);
}

nn::Result Device::FlushDebug(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    return m_Nfp.FlushDebug(service);
}

nn::Result Device::BreakTag(nn::nfc::server::core::Service* service, nn::nfp::BreakType breakType) NN_NOEXCEPT
{
    return m_Nfp.BreakTag(service, breakType);
}

nn::Result Device::WriteNtf(nn::nfc::server::core::Service* service, const void* pData, size_t dataSize, nn::nfp::NtfWriteType ntfWriteType) NN_NOEXCEPT
{
    return m_Nfp.WriteNtf(service, pData, dataSize, ntfWriteType);
}

size_t Device::GetApplicationAreaSize() NN_NOEXCEPT
{
    return m_Nfp.GetApplicationAreaSize();
}

nn::Result Device::ReadMifare(nn::nfc::MifareReadBlockData* pOutBlockData, nn::nfc::server::core::Service* service, const nn::nfc::MifareReadBlockParameter* pBlockParameter, size_t blockCount) NN_NOEXCEPT
{
    return m_NfcMifare.Read(pOutBlockData, service, pBlockParameter, blockCount);
}

nn::Result Device::WriteMifare(nn::nfc::server::core::Service* service, const nn::nfc::MifareWriteBlockParameter* pBlockParameter, size_t blockCount) NN_NOEXCEPT
{
    return m_NfcMifare.Write(service, pBlockParameter, blockCount);
}

nn::Result Device::SendCommandByPassThrough(void* pOutBuffer, size_t* pOutSize, nn::nfc::server::core::Service* service, const void* pData, size_t dataSize, size_t bufferSize, nn::TimeSpan timeout) NN_NOEXCEPT
{
    return m_NfcPt.SendCommand(pOutBuffer, pOutSize, service, pData, dataSize, bufferSize, timeout);
}

nn::Result Device::KeepPassThroughSession(nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    return m_NfcPt.KeepSession(service);
}

nn::Result Device::ReleasePassThroughSession() NN_NOEXCEPT
{
    return m_NfcPt.ReleaseSession();
}

nn::Result Device::GetXcdDeviceHandle(nn::xcd::DeviceHandle* pOutXcdDeviceHandle, nn::nfc::server::core::Service* service) NN_NOEXCEPT
{
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pOutXcdDeviceHandle);

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    return nn::nfc::server::core::Manager::GetInstance().GetXcdDeviceHandle(pOutXcdDeviceHandle, service, m_Handle);
}

nn::Result Device::CheckActive() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(m_State != nn::nfc::server::Device::State_Active)
    {
        if(m_State == nn::nfc::server::Device::State_Deactive)
        {
            return nn::nfc::ResultTagNotFound();
        }
        return nn::nfc::ResultInvalidDeviceState();
    }

    NN_RESULT_SUCCESS;
}

void Device::Hide() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    m_HideStartTick = nn::os::GetSystemTick();
}

bool Device::IsHidden() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    if(m_HideStartTick.GetInt64Value() == 0)
    {
        //Hide() が一度も呼ばれていない場合
        return false;
    }
    return (nn::os::ConvertToTimeSpan(nn::os::GetSystemTick() - m_HideStartTick).GetSeconds() < 60);
}

void Device::DispatchEventInEventThread(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    std::unique_ptr<nn::nfc::server::core::Info> info(new nn::nfc::server::core::Info());
    nn::Result result;

    if(pHolder == &m_DetectEventHolder)
    {
        nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
        nn::os::ClearSystemEvent(&m_DetectEvent);
        result = nn::nfc::server::core::Manager::GetInstance().GetDetectInfo(info.get(), nn::nfc::server::core::Service::NfcProcessService, m_Handle);
        if(result.IsFailure())
        {
            return;
        }
        switch(info->reason)
        {
        case nn::xcd::NfcEventReason_Detected:
            {
                if(m_State == nn::nfc::server::Device::State_Search)
                {
                    //見つけたTAG
                    SetState(nn::nfc::server::Device::State_Active);
                    SetSelectedTagId(info->ntagData.tagInfo.tagId);
                    SignalActivateEvent();
                }
            }
            break;
        case nn::xcd::NfcEventReason_Deactivated:
            {
                //TAGが喪失したら
                switch(m_State)
                {
                case nn::nfc::server::Device::State_Active:
                case nn::nfc::server::Device::State_Mount:
                case nn::nfc::server::Device::State_Keep:
                    {
                        Unmount();
                        ReleasePassThroughSession();

                        SignalDeactivateEvent();
                        SetState(nn::nfc::server::Device::State_Deactive);
                    }
                    break;
                default:
                    break;
                }
            }
            break;
        default:
            break;
        }
    }
    else if(pHolder == &m_EventHolder)
    {
        //こちらは排他不要(排他すると読み込みや書き込み処理結果として m_AccessFinishEvent を待っているスレッドとの間でデッドロックを起す。)

        nn::os::ClearSystemEvent(&m_Event);
        result = nn::nfc::server::core::Manager::GetInstance().GetInfo(info.get(), nn::nfc::server::core::Service::NfcProcessService, m_Handle);
        if(result.IsFailure())
        {
            return;
        }
        nn::os::SignalSystemEvent(&m_AccessFinishEvent);
    }
    else if(pHolder == &m_ResetEventHolder)
    {
        //こちらは排他不要(排他すると読み込みや書き込み処理結果として m_AccessResetEvent を待っているスレッドとの間でデッドロックを起す。)

        nn::os::ClearSystemEvent(&m_ResetEvent);

        nn::nfc::server::core::ResetInfo resetInfo;
        result = nn::nfc::server::core::Manager::GetInstance().GetResetInfo(&resetInfo, nn::nfc::server::core::Service::NfcProcessService, m_Handle);
        if(result.IsFailure())
        {
            return;
        }
        nn::os::SignalSystemEvent(&m_AccessResetEvent);

        Reset(nullptr);
    }
}

void Device::LoopEventThreadFunction(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    if(m_IsRunningThread == false || pHolder == &m_EventTerminateHolder)
    {
        nn::os::ClearSystemEvent(&m_EventTerminateEvent);
        return;
    }

    DispatchEventInEventThread(pHolder);
}

nn::Result Device::AfterAccessTag(nn::nfc::server::core::Service* service, nn::nfc::server::core::ScopedSession* session, nn::Result result) NN_NOEXCEPT
{
    if(result.IsFailure())
    {
        session->Release();
        NN_RESULT_DO(session->Keep());
        std::unique_ptr<nn::nfc::TagInfo> tagInfo(new nn::nfc::TagInfo);
        NN_RESULT_DO(GetTagInfo(tagInfo.get(), service));
    }

    return result;
}


}}}  // namespace nn::nfc::server
