﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/nfc/server/nfc_CommonImpl.h>
#include <nn/nfc/nfc_Types.h>
#include <nn/nfc/server/core/nfc_CoreManager.h>
#include <nn/nfc/server/nfc_Manager.h>
#include <nn/nfc/server/core/nfc_CoreUtil.h>
#include <nn/nfc/mifare/server/nfc_MifareResultOld.h>
#include <nn/nfc/mifare/server/nfc_MifarePrivateResultOld.h>

#if 0
#include <nn/nn_SdkLog.h>
#define NN_NFC_SERVER_COMMON_IMPL_LOG()  NN_SDK_LOG("[NFC] call %s(%d)\n", __FUNCTION__, __LINE__)
#else
#define NN_NFC_SERVER_COMMON_IMPL_LOG(...)
#endif

namespace nn { namespace nfc { namespace server {

namespace
{
}

CommonImpl::CommonImpl() NN_NOEXCEPT : m_IsInitialized(false), m_Service(nn::nfc::server::core::LibraryType_All)
{
}

CommonImpl::~CommonImpl() NN_NOEXCEPT
{
    Finalize();
}

nn::Result CommonImpl::InitializeNfc(nn::applet::AppletResourceUserId aruid, nn::Bit64 pid, const nn::sf::InArray<nn::xcd::McuVersionDataForNfc>& mcuVersionData) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    m_Service.SetLibraryType(nn::nfc::server::core::LibraryType_Nfc);
    m_Service.Initialize(aruid, pid, mcuVersionData);
    m_IsInitialized = true;
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::FinalizeNfc() NN_NOEXCEPT
{
    m_Service.Finalize();
    m_IsInitialized = false;
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::GetStateNfc(nn::sf::Out<nn::Bit32> state) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    *state = static_cast<nn::Bit32>(m_IsInitialized ? nn::nfc::State_Init : nn::nfc::State_None);
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::IsNfcEnabledNfc(nn::sf::Out<bool> isNfcEnabled) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return IsNfcEnabled(isNfcEnabled);
}

nn::Result CommonImpl::SetNfcEnabledNfc(bool isNfcEnabled) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return SetNfcEnabled(isNfcEnabled);
}

nn::Result CommonImpl::InitializeNfp(nn::applet::AppletResourceUserId aruid, nn::Bit64 pid, const nn::sf::InArray<nn::xcd::McuVersionDataForNfc>& mcuVersionData) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    m_Service.SetLibraryType(nn::nfc::server::core::LibraryType_Nfp);
    return InitializeCommon(aruid, pid, mcuVersionData);
}

nn::Result CommonImpl::FinalizeNfp() NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Finalize();
}

nn::Result CommonImpl::ListDevicesNfp(const nn::sf::OutArray<nn::nfc::DeviceHandle>& devices, nn::sf::Out<std::int32_t> count) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return ListDevices(devices, count);
}

nn::Result CommonImpl::StartDetectionNfp(nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return StartDetection(deviceHandle, static_cast<nn::Bit32>(nn::nfc::NfcProtocol_All));
}

nn::Result CommonImpl::StopDetectionNfp(nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return StopDetection(deviceHandle);
}

nn::Result CommonImpl::GetTagInfoNfp(nn::sf::Out<nn::nfc::TagInfo> tagInfo, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return GetTagInfo(tagInfo, deviceHandle);
}

nn::Result CommonImpl::GetActivateEventHandleNfp(nn::sf::Out<nn::sf::NativeHandle> handle, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return GetActivateEventHandle(handle, deviceHandle);
}

nn::Result CommonImpl::GetDeactivateEventHandleNfp(nn::sf::Out<nn::sf::NativeHandle> handle, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return GetDeactivateEventHandle(handle, deviceHandle);
}

nn::Result CommonImpl::GetStateNfp(nn::sf::Out<nn::Bit32> state) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return GetState(state);
}

nn::Result CommonImpl::GetDeviceStateNfp(nn::sf::Out<nn::Bit32> state, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    enum NfpDeviceState
    {
        NfpDeviceState_Init,
        NfpDeviceState_Search,
        NfpDeviceState_Active,
        NfpDeviceState_Deactive,
        NfpDeviceState_Mount,
        NfpDeviceState_Unexpected
    };

    nn::nfc::server::Device::State deviceState;
    deviceState = Manager::GetInstance().GetDeviceState(&m_Service, deviceHandle);
    switch(deviceState)
    {
    case nn::nfc::server::Device::State_Init:
        {
            *state = static_cast<nn::Bit32>(NfpDeviceState_Init);
        }
        break;
    case nn::nfc::server::Device::State_Search:
        {
            *state = static_cast<nn::Bit32>(NfpDeviceState_Search);
        }
        break;
    case nn::nfc::server::Device::State_Active:
    case nn::nfc::server::Device::State_Keep:
        {
            *state = static_cast<nn::Bit32>(NfpDeviceState_Active);
        }
        break;
    case nn::nfc::server::Device::State_Deactive:
        {
            *state = static_cast<nn::Bit32>(NfpDeviceState_Deactive);
        }
        break;
    case nn::nfc::server::Device::State_Mount:
        {
            *state = static_cast<nn::Bit32>(NfpDeviceState_Mount);
        }
        break;
    case nn::nfc::server::Device::State_Unexpected:
        {
            *state = static_cast<nn::Bit32>(NfpDeviceState_Unexpected);
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::GetNpadIdNfp(nn::sf::Out<nn::hid::NpadIdType> npadId, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return GetNpadId(npadId, deviceHandle);
}

nn::Result CommonImpl::GetAvailabilityChangeEventHandleNfp(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return GetAvailabilityChangeEventHandle(handle);
}

nn::Result CommonImpl::InitializeNfcMifare(nn::applet::AppletResourceUserId aruid, nn::Bit64 pid, const nn::sf::InArray<nn::xcd::McuVersionDataForNfc>& mcuVersionData) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    m_Service.SetLibraryType(nn::nfc::server::core::LibraryType_NfcMifare);
    return InitializeCommon(aruid, pid, mcuVersionData);
}

nn::Result CommonImpl::FinalizeNfcMifare() NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Finalize();
}

nn::Result CommonImpl::ListDevicesNfcMifare(const nn::sf::OutArray<nn::nfc::DeviceHandle>& devices, nn::sf::Out<std::int32_t> count) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return ListDevices(devices, count);
}

nn::Result CommonImpl::StartDetectionNfcMifare(nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return StartDetection(deviceHandle, static_cast<nn::Bit32>(nn::nfc::NfcProtocol_All));
}

nn::Result CommonImpl::StopDetectionNfcMifare(nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return StopDetection(deviceHandle);
}

nn::Result CommonImpl::ReadNfcMifare(const nn::sf::OutArray<nn::nfc::MifareReadBlockData>& blockData, nn::nfc::DeviceHandle deviceHandle, const nn::sf::InArray<nn::nfc::MifareReadBlockParameter>& blockParameter) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return ReadMifare(blockData, deviceHandle, blockParameter);
}

nn::Result CommonImpl::WriteNfcMifare(nn::nfc::DeviceHandle deviceHandle, const nn::sf::InArray<nn::nfc::MifareWriteBlockParameter>& blockParameter) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return WriteMifare(deviceHandle, blockParameter);
}

nn::Result CommonImpl::GetTagInfoNfcMifare(nn::sf::Out<nn::nfc::TagInfo> tagInfo, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return GetTagInfo(tagInfo, deviceHandle);
}

nn::Result CommonImpl::GetActivateEventHandleNfcMifare(nn::sf::Out<nn::sf::NativeHandle> handle, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return GetActivateEventHandle(handle, deviceHandle);
}

nn::Result CommonImpl::GetDeactivateEventHandleNfcMifare(nn::sf::Out<nn::sf::NativeHandle> handle, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return GetDeactivateEventHandle(handle, deviceHandle);
}

nn::Result CommonImpl::GetStateNfcMifare(nn::sf::Out<nn::Bit32> state) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return GetState(state);
}

nn::Result CommonImpl::GetDeviceStateNfcMifare(nn::sf::Out<nn::Bit32> state, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    enum NfcMifareDeviceState
    {
        NfcMifareDeviceState_Init,
        NfcMifareDeviceState_Search,
        NfcMifareDeviceState_Active,
        NfcMifareDeviceState_Deactive,
        NfcMifareDeviceState_Unexpected
    };

    nn::nfc::server::Device::State deviceState;
    deviceState = Manager::GetInstance().GetDeviceState(&m_Service, deviceHandle);
    switch(deviceState)
    {
    case nn::nfc::server::Device::State_Init:
        {
            *state = static_cast<nn::Bit32>(NfcMifareDeviceState_Init);
        }
        break;
    case nn::nfc::server::Device::State_Search:
        {
            *state = static_cast<nn::Bit32>(NfcMifareDeviceState_Search);
        }
        break;
    case nn::nfc::server::Device::State_Active:
    case nn::nfc::server::Device::State_Mount:
    case nn::nfc::server::Device::State_Keep:
        {
            *state = static_cast<nn::Bit32>(NfcMifareDeviceState_Active);
        }
        break;
    case nn::nfc::server::Device::State_Deactive:
        {
            *state = static_cast<nn::Bit32>(NfcMifareDeviceState_Deactive);
        }
        break;
    case nn::nfc::server::Device::State_Unexpected:
        {
            *state = static_cast<nn::Bit32>(NfcMifareDeviceState_Unexpected);
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::GetNpadIdNfcMifare(nn::sf::Out<nn::hid::NpadIdType> npadId, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return GetNpadId(npadId, deviceHandle);
}

nn::Result CommonImpl::GetAvailabilityChangeEventHandleNfcMifare(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return GetAvailabilityChangeEventHandle(handle);
}

nn::Result CommonImpl::Initialize(nn::applet::AppletResourceUserId aruid, nn::Bit64 pid, const nn::sf::InArray<nn::xcd::McuVersionDataForNfc>& mcuVersionData) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return InitializeCommon(aruid, pid, mcuVersionData);
}

nn::Result CommonImpl::Finalize() NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    if(m_IsInitialized)
    {

        if(m_Service.GetLibraryType() == nn::nfc::server::core::LibraryType_Nfc)
        {
            FinalizeNfc();
        }
        else
        {
            Manager::GetInstance().Finalize(&m_Service);
            m_Service.Finalize();
            m_IsInitialized = false;
        }
    }
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::GetState(nn::sf::Out<nn::Bit32> state) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    *state = static_cast<nn::Bit32>(Manager::GetInstance().GetState(&m_Service));
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::IsNfcEnabled(nn::sf::Out<bool> isNfcEnabled) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    bool outIsNfcEnabled;
    NN_RESULT_DO(nn::nfc::server::core::Manager::GetInstance().IsNfcEnabled(&outIsNfcEnabled, &m_Service));
    *isNfcEnabled = outIsNfcEnabled;
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::ListDevices(const nn::sf::OutArray<nn::nfc::DeviceHandle>& devices, nn::sf::Out<std::int32_t> count) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    int outCount;
    NN_RESULT_DO(Convert(Manager::GetInstance().ListDevices(devices.GetData(), &outCount, &m_Service, static_cast<int>(devices.GetLength()), true)));
    *count = static_cast<std::int32_t>(outCount);
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::GetDeviceState(nn::sf::Out<nn::Bit32> state, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();

    nn::nfc::server::Device::State deviceState;
    deviceState = Manager::GetInstance().GetDeviceState(&m_Service, deviceHandle);
    switch(deviceState)
    {
    case nn::nfc::server::Device::State_Init:
        {
            *state = static_cast<nn::Bit32>(nn::nfc::DeviceState_Init);
        }
        break;
    case nn::nfc::server::Device::State_Search:
        {
            *state = static_cast<nn::Bit32>(nn::nfc::DeviceState_Search);
        }
        break;
    case nn::nfc::server::Device::State_Active:
    case nn::nfc::server::Device::State_Mount:
        {
            *state = static_cast<nn::Bit32>(nn::nfc::DeviceState_Active);
        }
        break;
    case nn::nfc::server::Device::State_Deactive:
        {
            *state = static_cast<nn::Bit32>(nn::nfc::DeviceState_Deactive);
        }
        break;
    case nn::nfc::server::Device::State_Keep:
        {
            *state = static_cast<nn::Bit32>(nn::nfc::DeviceState_Keep);
        }
        break;
    case nn::nfc::server::Device::State_Unexpected:
        {
            *state = static_cast<nn::Bit32>(nn::nfc::DeviceState_Unexpected);
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::GetNpadId(nn::sf::Out<nn::hid::NpadIdType> npadId, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    nn::hid::NpadIdType value;
    NN_RESULT_DO(Convert(Manager::GetInstance().GetNpadId(&value, &m_Service, deviceHandle)));
    npadId.Set(value);
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::GetAvailabilityChangeEventHandle(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    nn::os::NativeHandle internalHandle;
    NN_RESULT_DO(Convert(Manager::GetInstance().GetAvailabilityChangeEventHandle(&internalHandle, &m_Service)));
    handle.Set(nn::sf::NativeHandle(internalHandle, false));
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::StartDetection(nn::nfc::DeviceHandle deviceHandle, nn::Bit32 protocolFilter) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Convert(Manager::GetInstance().StartDetection(&m_Service, deviceHandle, static_cast<nn::nfc::NfcProtocol>(protocolFilter)));
}

nn::Result CommonImpl::StopDetection(nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Convert(Manager::GetInstance().StopDetection(&m_Service, deviceHandle));
}

nn::Result CommonImpl::GetTagInfo(nn::sf::Out<nn::nfc::TagInfo> tagInfo, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    std::unique_ptr<nn::nfc::TagInfo> value(new nn::nfc::TagInfo());
    NN_RESULT_DO(Convert(Manager::GetInstance().GetTagInfo(value.get(), &m_Service, deviceHandle)));
    tagInfo.Set(*value);
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::GetActivateEventHandle(nn::sf::Out<nn::sf::NativeHandle> handle, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    nn::os::NativeHandle internalHandle;
    NN_RESULT_DO(Convert(Manager::GetInstance().GetActivateEventHandle(&internalHandle, &m_Service, deviceHandle)));
    handle.Set(nn::sf::NativeHandle(internalHandle, false));
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::GetDeactivateEventHandle(nn::sf::Out<nn::sf::NativeHandle> handle, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    nn::os::NativeHandle internalHandle;
    NN_RESULT_DO(Convert(Manager::GetInstance().GetDeactivateEventHandle(&internalHandle, &m_Service, deviceHandle)));
    handle.Set(nn::sf::NativeHandle(internalHandle, false));
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::SetNfcEnabled(bool isNfcEnabled) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return nn::nfc::server::core::Manager::GetInstance().SetNfcEnabled(&m_Service, isNfcEnabled);
}

nn::Result CommonImpl::MountNfp(nn::nfc::DeviceHandle deviceHandle, nn::Bit32 modelType, nn::Bit32 mountTarget) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Convert(Manager::GetInstance().Mount(&m_Service, deviceHandle, static_cast<nn::nfp::ModelType>(modelType), static_cast<nn::nfp::MountTarget>(mountTarget)));
}

nn::Result CommonImpl::UnmountNfp(nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Convert(Manager::GetInstance().Unmount(&m_Service, deviceHandle));
}

nn::Result CommonImpl::FlushNfp(nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Convert(Manager::GetInstance().Flush(&m_Service, deviceHandle));
}

nn::Result CommonImpl::RestoreNfp(nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Convert(Manager::GetInstance().Restore(&m_Service, deviceHandle));
}

nn::Result CommonImpl::GetRegisterInfoNfp(nn::sf::Out<nn::nfp::RegisterInfo> registerInfo, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    std::unique_ptr<nn::nfp::RegisterInfo> value(new nn::nfp::RegisterInfo());
    NN_RESULT_DO(Convert(Manager::GetInstance().GetRegisterInfo(value.get(), &m_Service, deviceHandle)));
    registerInfo.Set(*value);
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::GetCommonInfoNfp(nn::sf::Out<nn::nfp::CommonInfo> commonInfo, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    std::unique_ptr<nn::nfp::CommonInfo> value(new nn::nfp::CommonInfo());
    NN_RESULT_DO(Convert(Manager::GetInstance().GetCommonInfo(value.get(), &m_Service, deviceHandle)));
    commonInfo.Set(*value);
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::GetModelInfoNfp(nn::sf::Out<nn::nfp::ModelInfo> modelInfo, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    std::unique_ptr<nn::nfp::ModelInfo> value(new nn::nfp::ModelInfo());
    NN_RESULT_DO(Convert(Manager::GetInstance().GetModelInfo(value.get(), &m_Service, deviceHandle)));
    modelInfo.Set(*value);
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::GetApplicationAreaSizeNfp(nn::sf::Out<uint32_t> size, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    NN_UNUSED(deviceHandle);
    *size = static_cast<uint32_t>(Manager::GetInstance().GetApplicationAreaSize());
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::OpenApplicationAreaNfp(nn::nfc::DeviceHandle deviceHandle, nn::Bit32 accessId) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Convert(Manager::GetInstance().OpenApplicationArea(&m_Service, deviceHandle, accessId));
}

nn::Result CommonImpl::GetApplicationAreaNfp(const nn::sf::OutBuffer& buffer, nn::sf::Out<uint32_t> size, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    size_t outSize;
    NN_RESULT_DO(Convert(Manager::GetInstance().GetApplicationArea(buffer.GetPointerUnsafe(), &outSize, &m_Service, deviceHandle, buffer.GetSize())));
    *size = static_cast<uint32_t>(outSize);
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::SetApplicationAreaNfp(nn::nfc::DeviceHandle deviceHandle, const nn::sf::InBuffer& data) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Convert(Manager::GetInstance().SetApplicationArea(&m_Service, deviceHandle, data.GetPointerUnsafe(), data.GetSize()));
}

nn::Result CommonImpl::CreateApplicationAreaNfp(nn::nfc::DeviceHandle deviceHandle, nn::Bit32 accessId, const nn::sf::InBuffer& initialData) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    nn::nfp::ApplicationAreaCreateInfo createInfo;
    createInfo.accessId = accessId;
    createInfo.pInitialData = const_cast<char*>(initialData.GetPointerUnsafe());
    createInfo.initialDataSize = static_cast<uint32_t>(initialData.GetSize());
    return Convert(Manager::GetInstance().CreateApplicationArea(&m_Service, deviceHandle, createInfo));
}

nn::Result CommonImpl::RecreateApplicationAreaNfp(nn::nfc::DeviceHandle deviceHandle, nn::Bit32 accessId, const nn::sf::InBuffer& initialData) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    nn::nfp::ApplicationAreaCreateInfo createInfo;
    createInfo.accessId = accessId;
    createInfo.pInitialData = const_cast<char*>(initialData.GetPointerUnsafe());
    createInfo.initialDataSize = static_cast<uint32_t>(initialData.GetSize());
    return Convert(Manager::GetInstance().RecreateApplicationArea(&m_Service, deviceHandle, createInfo));
}

nn::Result CommonImpl::FormatNfp(nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Convert(Manager::GetInstance().Format(&m_Service, deviceHandle));
}

nn::Result CommonImpl::GetAdminInfoNfp(nn::sf::Out<nn::nfp::AdminInfo> adminInfo, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    std::unique_ptr<nn::nfp::AdminInfo> value(new nn::nfp::AdminInfo());
    NN_RESULT_DO(Convert(Manager::GetInstance().GetAdminInfo(value.get(), &m_Service, deviceHandle)));
    adminInfo.Set(*value);
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::GetRegisterInfoPrivateNfp(nn::sf::Out<nn::nfp::RegisterInfoPrivate> registerInfo, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    std::unique_ptr<nn::nfp::RegisterInfoPrivate> value(new nn::nfp::RegisterInfoPrivate());
    NN_RESULT_DO(Convert(Manager::GetInstance().GetRegisterInfo(value.get(), &m_Service, deviceHandle)));
    registerInfo.Set(*value);
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::SetRegisterInfoPrivateNfp(nn::nfc::DeviceHandle deviceHandle, const nn::nfp::RegisterInfoPrivate& registerInfo) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Convert(Manager::GetInstance().SetRegisterInfo(&m_Service, deviceHandle, registerInfo));
}

nn::Result CommonImpl::DeleteRegisterInfoNfp(nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Convert(Manager::GetInstance().DeleteRegisterInfo(&m_Service, deviceHandle));
}

nn::Result CommonImpl::DeleteApplicationAreaNfp(nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Convert(Manager::GetInstance().DeleteApplicationArea(&m_Service, deviceHandle));
}

nn::Result CommonImpl::ExistsApplicationAreaNfp(nn::sf::Out<bool> flag, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    bool value;
    NN_RESULT_DO(Convert(Manager::GetInstance().ExistsApplicationArea(&value, &m_Service, deviceHandle)));
    *flag = value;
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::GetAllNfp(nn::sf::Out<nn::nfp::NfpData> nfpData, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    std::unique_ptr<nn::nfp::NfpData> value(new nn::nfp::NfpData());
    NN_RESULT_DO(Convert(Manager::GetInstance().GetAll(value.get(), &m_Service, deviceHandle)));
    nfpData.Set(*value);
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::SetAllNfp(nn::nfc::DeviceHandle deviceHandle, const nn::nfp::NfpData& nfpData) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Convert(Manager::GetInstance().SetAll(&m_Service, deviceHandle, nfpData));
}

nn::Result CommonImpl::FlushDebugNfp(nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Convert(Manager::GetInstance().FlushDebug(&m_Service, deviceHandle));
}

nn::Result CommonImpl::BreakTagNfp(nn::nfc::DeviceHandle deviceHandle, nn::Bit32 breakType) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Convert(Manager::GetInstance().BreakTag(&m_Service, deviceHandle, static_cast<nn::nfp::BreakType>(breakType)));
}

nn::Result CommonImpl::ReadBackupDataNfp(const nn::sf::OutBuffer& buffer, nn::sf::Out<std::uint32_t> size) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    size_t outSize;
    NN_RESULT_DO(Convert(Manager::GetInstance().ReadBackupData(buffer.GetPointerUnsafe(), &outSize, &m_Service, buffer.GetSize())));
    *size = static_cast<uint32_t>(outSize);
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::WriteBackupDataNfp(const nn::sf::InBuffer& data) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Convert(Manager::GetInstance().WriteBackupData(&m_Service, data.GetPointerUnsafe(), data.GetSize()));
}

nn::Result CommonImpl::WriteNtfNfp(nn::nfc::DeviceHandle deviceHandle, const nn::sf::InBuffer& data, nn::Bit32 ntfWriteType) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Convert(Manager::GetInstance().WriteNtf(&m_Service, deviceHandle, data.GetPointerUnsafe(), data.GetSize(), static_cast<nn::nfp::NtfWriteType>(ntfWriteType)));
}

nn::Result CommonImpl::ReadMifare(const nn::sf::OutArray<nn::nfc::MifareReadBlockData>& blockData, nn::nfc::DeviceHandle deviceHandle, const nn::sf::InArray<nn::nfc::MifareReadBlockParameter>& blockParameter) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Convert(Manager::GetInstance().ReadMifare(blockData.GetData(), &m_Service, deviceHandle, &blockParameter[0], blockParameter.GetLength()));
}

nn::Result CommonImpl::WriteMifare(nn::nfc::DeviceHandle deviceHandle, const nn::sf::InArray<nn::nfc::MifareWriteBlockParameter>& blockParameter) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Convert(Manager::GetInstance().WriteMifare(&m_Service, deviceHandle, &blockParameter[0], blockParameter.GetLength()));
}

nn::Result CommonImpl::SendCommandByPassThrough(const nn::sf::OutBuffer& buffer, nn::sf::Out<std::uint32_t> size, nn::nfc::DeviceHandle deviceHandle, const nn::sf::InBuffer& data, nn::TimeSpan timeout) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    size_t outSize;
    NN_RESULT_DO(Manager::GetInstance().SendCommandByPassThrough(buffer.GetPointerUnsafe(), &outSize, &m_Service, deviceHandle, data.GetPointerUnsafe(), data.GetSize(), buffer.GetSize(), timeout));
    *size = static_cast<uint32_t>(outSize);
    NN_RESULT_SUCCESS;
}

nn::Result CommonImpl::KeepPassThroughSession(nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Manager::GetInstance().KeepPassThroughSession(&m_Service, deviceHandle);
}

nn::Result CommonImpl::ReleasePassThroughSession(nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    return Manager::GetInstance().ReleasePassThroughSession(&m_Service, deviceHandle);
}

namespace
{

nn::Result ConvertToNfpResult(nn::Result result) NN_NOEXCEPT
{
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(nn::nfc::ResultAccessOperationFailed)
    {
        NN_RESULT_THROW(nn::nfp::ResultUnknownError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultAccessTimeOutError)
    {
        NN_RESULT_THROW(nn::nfp::ResultUnknownError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultAccessError)
    {
        NN_RESULT_THROW(nn::nfp::ResultUnknownError());
    }
    NN_RESULT_CATCH(nn::nfc::server::core::ResultFsPathAlreadyExists)
    {
        NN_RESULT_THROW(nn::nfp::ResultUnknownError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNotUpdated)
    {
        NN_RESULT_THROW(nn::nfp::ResultNotUpdated());
    }
    NN_RESULT_CATCH(nn::nfc::ResultConflictFunction)
    {
        NN_RESULT_THROW(nn::nfp::ResultConflictFunction());
    }
    NN_RESULT_CATCH(nn::nfc::ResultMaxNfcDeviceActivated)
    {
        NN_RESULT_THROW(nn::nfp::ResultMaxNfcDeviceActivated());
    }
    NN_RESULT_CATCH(nn::nfc::ResultInvalidFormatVersion)
    {
        NN_RESULT_THROW(nn::nfp::ResultInvalidFormatVersion());
    }
    NN_RESULT_CATCH(nn::nfc::ResultInvalidRomArea)
    {
        NN_RESULT_THROW(nn::nfp::ResultInvalidRomArea());
    }
    NN_RESULT_CATCH(nn::nfc::ResultAuthenticationError)
    {
        NN_RESULT_THROW(nn::nfp::ResultAuthenticationError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultInvalidTag)
    {
        NN_RESULT_THROW(nn::nfp::ResultInvalidTag());
    }
    NN_RESULT_CATCH(nn::nfc::ResultInvalidFormat)
    {
        NN_RESULT_THROW(nn::nfp::ResultInvalidFormat());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNotSupported)
    {
        NN_RESULT_THROW(nn::nfp::ResultNotSupported());
    }
    NN_RESULT_CATCH(nn::nfc::ResultAlreadyCreated)
    {
        NN_RESULT_THROW(nn::nfp::ResultAlreadyCreated());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNotBroken)
    {
        NN_RESULT_THROW(nn::nfp::ResultNotBroken());
    }
    NN_RESULT_CATCH(nn::nfc::ResultAccessIdMisMatch)
    {
        NN_RESULT_THROW(nn::nfp::ResultAccessIdMisMatch());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNeedFormat)
    {
        NN_RESULT_THROW(nn::nfp::ResultNeedFormat());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNeedRestore)
    {
        NN_RESULT_THROW(nn::nfp::ResultNeedRestore());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNeedCreate)
    {
        NN_RESULT_THROW(nn::nfp::ResultNeedCreate());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNeedRegister)
    {
        NN_RESULT_THROW(nn::nfp::ResultNeedRegister());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNeedCharge)
    {
        NN_RESULT_THROW(nn::nfp::ResultNeedCharge());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNeedUpdate)
    {
        NN_RESULT_THROW(nn::nfp::ResultNeedUpdate());
    }
    NN_RESULT_CATCH(nn::nfc::ResultBackupSystemError)
    {
        NN_RESULT_THROW(nn::nfp::ResultBackupSystemError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNfcDeviceError)
    {
        NN_RESULT_THROW(nn::nfp::ResultNfcDeviceError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultUidMisMatch)
    {
        NN_RESULT_THROW(nn::nfp::ResultUidMisMatch());
    }
    NN_RESULT_CATCH(nn::nfc::ResultTagNotFound)
    {
        NN_RESULT_THROW(nn::nfp::ResultTagNotFound());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNeedRestart)
    {
        NN_RESULT_THROW(nn::nfp::ResultNeedRestart());
    }
    NN_RESULT_CATCH(nn::nfc::ResultOperationFailed)
    {
        NN_RESULT_THROW(nn::nfp::ResultOperationFailed());
    }
    NN_RESULT_CATCH(nn::nfc::ResultTimeOutError)
    {
        NN_RESULT_THROW(nn::nfp::ResultTimeOutError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNeedRetry)
    {
        NN_RESULT_THROW(nn::nfp::ResultNeedRetry());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNfcDisabled)
    {
        NN_RESULT_THROW(nn::nfp::ResultNfcDisabled());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNotForeground)
    {
        NN_RESULT_THROW(nn::nfp::ResultNotForeground());
    }
    NN_RESULT_CATCH(nn::nfc::ResultSleep)
    {
        NN_RESULT_THROW(nn::nfp::ResultSleep());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNotInFocus)
    {
        NN_RESULT_THROW(nn::nfp::ResultNotInFocus());
    }
    NN_RESULT_CATCH(nn::nfc::ResultUnknownError)
    {
        NN_RESULT_THROW(nn::nfp::ResultUnknownError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultInvalidDeviceState)
    {
        NN_RESULT_THROW(nn::nfp::ResultInvalidDeviceState());
    }
    NN_RESULT_CATCH(nn::nfc::ResultInvalidSize)
    {
        NN_RESULT_THROW(nn::nfp::ResultInvalidSize());
    }
    NN_RESULT_CATCH(nn::nfc::ResultInvalidMii)
    {
        NN_RESULT_THROW(nn::nfp::ResultInvalidMii());
    }
    NN_RESULT_CATCH(nn::nfc::ResultInvalidNickname)
    {
        NN_RESULT_THROW(nn::nfp::ResultInvalidNickname());
    }
    NN_RESULT_CATCH(nn::nfc::ResultBadRequest)
    {
        NN_RESULT_THROW(nn::nfp::ResultBadRequest());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNfcDeviceNotFound)
    {
        NN_RESULT_THROW(nn::nfp::ResultNfcDeviceNotFound());
    }
    NN_RESULT_CATCH(nn::nfc::ResultBackupCrcBroken)
    {
        NN_RESULT_THROW(nn::nfp::ResultBackupCrcBroken());
    }
    NN_RESULT_CATCH(nn::nfc::ResultBackupWriteFailed)
    {
        NN_RESULT_THROW(nn::nfp::ResultBackupWriteFailed());
    }
    NN_RESULT_CATCH(nn::nfc::ResultBackupReadFailed)
    {
        NN_RESULT_THROW(nn::nfp::ResultBackupReadFailed());
    }
    NN_RESULT_CATCH(nn::nfc::ResultBackupError)
    {
        NN_RESULT_THROW(nn::nfp::ResultBackupError());
    }
    NN_RESULT_CATCH_ALL
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

nn::Result ConvertToNfcMifareResult(nn::Result result) NN_NOEXCEPT
{
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(nn::nfc::ResultAccessOperationFailed)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultAccessOperationFailed());
    }
    NN_RESULT_CATCH(nn::nfc::ResultAccessTimeOutError)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultAccessTimeOutError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultAccessError)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultAccessError());
    }
    NN_RESULT_CATCH(nn::nfc::server::core::ResultFsPathAlreadyExists)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultUnknownError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNotUpdated)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultUnknownError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultConflictFunction)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultConflictFunction());
    }
    NN_RESULT_CATCH(nn::nfc::ResultMaxNfcDeviceActivated)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultMaxNfcDeviceActivated());
    }
    NN_RESULT_CATCH(nn::nfc::ResultInvalidFormatVersion)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultUnknownError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultInvalidRomArea)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultInvalidRomArea());
    }
    NN_RESULT_CATCH(nn::nfc::ResultAuthenticationError)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultAuthenticationError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultInvalidTag)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultInvalidTag());
    }
    NN_RESULT_CATCH(nn::nfc::ResultInvalidFormat)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultInvalidFormat());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNotSupported)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultNotSupported());
    }
    NN_RESULT_CATCH(nn::nfc::ResultAlreadyCreated)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultUnknownError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNotBroken)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultUnknownError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultAccessIdMisMatch)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultUnknownError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNeedFormat)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultUnknownError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNeedRestore)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultUnknownError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNeedCreate)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultUnknownError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNeedRegister)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultUnknownError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNeedCharge)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultNeedCharge());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNeedUpdate)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultNeedUpdate());
    }
    NN_RESULT_CATCH(nn::nfc::ResultBackupSystemError)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultBackupSystemError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNfcDeviceError)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultNfcDeviceError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultUidMisMatch)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultUidMisMatch());
    }
    NN_RESULT_CATCH(nn::nfc::ResultTagNotFound)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultTagNotFound());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNeedRestart)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultNeedRestart());
    }
    NN_RESULT_CATCH(nn::nfc::ResultOperationFailed)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultOperationFailed());
    }
    NN_RESULT_CATCH(nn::nfc::ResultTimeOutError)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultTimeOutError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNeedRetry)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultNeedRetry());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNfcDisabled)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultNfcDisabled());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNotForeground)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultNotForeground());
    }
    NN_RESULT_CATCH(nn::nfc::ResultSleep)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultSleep());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNotInFocus)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultNotInFocus());
    }
    NN_RESULT_CATCH(nn::nfc::ResultUnknownError)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultUnknownError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultInvalidDeviceState)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultInvalidDeviceState());
    }
    NN_RESULT_CATCH(nn::nfc::ResultInvalidSize)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultUnknownError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultInvalidMii)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultUnknownError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultInvalidNickname)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultUnknownError());
    }
    NN_RESULT_CATCH(nn::nfc::ResultBadRequest)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultBadRequest());
    }
    NN_RESULT_CATCH(nn::nfc::ResultNfcDeviceNotFound)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultNfcDeviceNotFound());
    }
    NN_RESULT_CATCH(nn::nfc::ResultBackupCrcBroken)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultBackupCrcBroken());
    }
    NN_RESULT_CATCH(nn::nfc::ResultBackupWriteFailed)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultBackupWriteFailed());
    }
    NN_RESULT_CATCH(nn::nfc::ResultBackupReadFailed)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultBackupReadFailed());
    }
    NN_RESULT_CATCH(nn::nfc::ResultBackupError)
    {
        NN_RESULT_THROW(nn::nfc::mifare::server::ResultBackupError());
    }
    NN_RESULT_CATCH_ALL
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

nn::Result ConvertToNfcResult(nn::Result result) NN_NOEXCEPT
{
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(nn::nfc::server::core::ResultFsPathAlreadyExists)
    {
        NN_RESULT_THROW(nn::nfc::ResultUnknownError());
    }
    NN_RESULT_CATCH_ALL
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

}
nn::Result CommonImpl::Convert(nn::Result result) NN_NOEXCEPT
{
    nn::nfc::server::core::LibraryType libraryType = m_Service.GetLibraryType();
    if(libraryType == nn::nfc::server::core::LibraryType_Nfp)
    {
        return ConvertToNfpResult(result);
    }
    else if(libraryType == nn::nfc::server::core::LibraryType_NfcMifare)
    {
        return ConvertToNfcMifareResult(result);
    }
    else
    {
        return ConvertToNfcResult(result);
    }
}

nn::Result CommonImpl::InitializeCommon(nn::applet::AppletResourceUserId aruid, nn::Bit64 pid, const nn::sf::InArray<nn::xcd::McuVersionDataForNfc>& mcuVersionData) NN_NOEXCEPT
{
    NN_NFC_SERVER_COMMON_IMPL_LOG();
    m_Service.Initialize(aruid, pid, mcuVersionData);
    nn::Result result = Manager::GetInstance().Initialize(&m_Service);
    if(result.IsFailure())
    {
        m_Service.Finalize();
        return result;
    }

    m_IsInitialized = true;
    NN_RESULT_SUCCESS;
}

}}}
