﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/nfc/server/core/nfc_SaveData.h>
#include "nfc_FileStream.h"
#include <nn/nfc/server/util/nfc_ScopedMutexLock.h>

#define NN_NFC_SERVER_SAVE_DATA_RETURN(x) return(ConvertToNfcResult(x))
#define NN_NFC_SERVER_SAVE_DATA_RESULT_DO(x) NN_RESULT_DO(ConvertToNfcResult(x))

namespace nn {
namespace nfc {
namespace server {
namespace core {

namespace {

const nn::fs::SystemSaveDataId SystemSaveDataId = 0x8000000000000020;
const char SystemSaveDataArchiveName[] = "data";

nn::Result ConvertToNfcResult(nn::Result fsResult)
{
    NN_RESULT_TRY(fsResult)
        NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
        {
            NN_RESULT_THROW(nn::nfc::server::core::ResultFsPathAlreadyExists());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_THROW(nn::nfc::ResultBackupSystemError());
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

nn::Result CreateAndMountSaveData() NN_NOEXCEPT
{
    nn::fs::DisableAutoSaveDataCreation();
    NN_RESULT_TRY(nn::fs::MountSystemSaveData(SystemSaveDataArchiveName, SystemSaveDataId))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            NN_NFC_SERVER_SAVE_DATA_RESULT_DO(nn::fs::CreateSystemSaveData(SystemSaveDataId, nn::nfc::server::core::SaveData::SystemSaveDataSize, nn::nfc::server::core::SaveData::SystemSaveDataJournalSize, 0));
            NN_NFC_SERVER_SAVE_DATA_RESULT_DO(nn::fs::MountSystemSaveData(SystemSaveDataArchiveName, SystemSaveDataId));
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_THROW(nn::nfc::ResultBackupSystemError());
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

}

const char* SaveData::FilePathNfpBackup = "data:/nfp_backup.dat";
const char* SaveData::FilePathNfcTerminalId = "data:/nfc_terminal_id.dat";
const char* SaveData::FilePathList[] = {
    FilePathNfpBackup,
    FilePathNfcTerminalId
};

/*
  セーブデータの各ファイルのサイズは、
  - 端末 ID ファイル(nfc_terminal_id.dat)          : 8バイト       sizeof(nn::nfc::server::core::Manager::m_TerminalId)
  - NFP バックアップデータファイル(nfp_backup.dat) : 2080032バイト nn::nfp::server::BackupDataStream::BackupTotalSize

  それぞれブロック数(16KB)に換算すると
  - 端末　ID ファイル              : 1 ブロック
  - NFP バックアップデータファイル : 127 ブロック

  従って nn::fs::CreateSystemSaveData() に指定する総利用ファイルサイズは
  1 ブロック + 127 ブロック + 2 ブロック(管理ブロック) = 130 ブロック = 130 * 16KB = 2080KB
 */

/*
  コミット実行までに変更されるサイズ一番大きくなるのは、1 エントリの NFP バックアップデータを保存するときです。
  1 エントリの NFP バックアップデータを保存するときに行われる変更は
  - バックアップデータヘッダの変更            : 32バイト   sizeof(nn::nfp::server::BackupDataHeader)
  - バックアップデータ索引 1 エントリ分の変更 : 32バイト   sizeof(nn::nfp::server::BackupTocItem)
  - バックアップデータ 1 エントリ分の変更     : 2048バイト nn::nfp::server::BackupDataStream::BackupEntrySize

  それぞれブロック数(16KB)に換算すると
  - バックアップデータヘッダの変更            : 1 ブロック(ブロックをまたぐことはありえない)
  - バックアップデータ索引 1 エントリ分の変更 : 1 ブロック(ブロックをまたぐことはありえない)
  - バックアップデータ 1 エントリ分の変更     : 2 ブロック(ブロックをまたぐ可能性がある)

  したがって、必要となるジャーナルサイズは、
  1 ブロック + 1 ブロック + 2 ブロック + デフォルトのエントリ管理領域(32KB) = 4 ブロック + 32KB = 4 * 16KB + 32KB = 96KB

  */

/*
  将来の拡張を見越して少し余裕を持たせたサイズにします。

  総利用ファイルサイズ + ジャーナルサイズが収まる区切りのよい 2.5MB を設定し、
  (総利用ファイルサイズ + reserved) + (ジャーナルサイズ + reserved) = 2.5MB
  となるように、

  総利用ファイルサイズ + reserved = 2432KB (> 2080KB)
  ジャーナルサイズ + reserved = 128KB (> 96KB)
　とします。
 */

const int64_t SaveData::SystemSaveDataSize = 2432 * 1024;
const int64_t SaveData::SystemSaveDataJournalSize = 128 * 1024;
const int64_t SaveData::SystemSaveDataCommitSize = SystemSaveDataJournalSize - 32 * 1024;

SaveData::SaveData() NN_NOEXCEPT
    : m_IsInitialized(false)
{
    nn::os::InitializeMutex(&m_Mutex, true, 0);
}

SaveData::~SaveData() NN_NOEXCEPT
{
    if(m_IsInitialized)
    {
        Finalize();
    }

    nn::os::FinalizeMutex(&m_Mutex);
}

bool SaveData::IsInitialized() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    return m_IsInitialized;
}

void SaveData::Initialize() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    NN_ABORT_UNLESS(!m_IsInitialized);

    // セーブデータをマウントします。存在しない場合は作成します。
    m_Result = CreateAndMountSaveData();

    m_IsInitialized = true;
}

void SaveData::Finalize() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    NN_ABORT_UNLESS(m_IsInitialized);

    for(auto i = 0; i < FileStreamCountMax; ++i)
    {
        if(m_FileStream[i])
        {
            m_FileStream[i].reset(nullptr);
        }
    }

    nn::fs::Unmount(SystemSaveDataArchiveName);

    m_IsInitialized = false;
}

nn::Result SaveData::Format() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    NN_ABORT_UNLESS(m_IsInitialized);
    NN_RESULT_DO(m_Result);

    // 一旦アンマウントします。
    Finalize();
    // 再度マウントします。
    Initialize();

    for(auto i = 0; i < FileCountMax; ++i)
    {
        NN_NFC_SERVER_SAVE_DATA_RESULT_DO(nn::fs::DeleteFile(FilePathList[i]));
    }

    NN_RESULT_SUCCESS;
}

nn::Result SaveData::Create(const char* path, int64_t size) NN_NOEXCEPT
{
    NN_NFC_SERVER_SAVE_DATA_RETURN(nn::fs::CreateFile(path, size));
}

nn::Result SaveData::Delete(const char* path) NN_NOEXCEPT
{
    NN_NFC_SERVER_SAVE_DATA_RETURN(nn::fs::DeleteFile(path));
}

nn::Result SaveData::Open(int* pOutFileHandle, const char* path, int mode) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    NN_ABORT_UNLESS(m_IsInitialized);
    NN_RESULT_DO(m_Result);

    bool find = false;

    for(auto i = 0; i < FileCountMax; ++i)
    {
        if(std::strcmp(path, FilePathList[i]) == 0)
        {
            find = true;
            break;
        }
    }

    NN_ABORT_UNLESS(find);

    for(auto i = 0; i < FileStreamCountMax; ++i)
    {
        if(!m_FileStream[i])
        {
            m_FileStream[i].reset(new FileStream);
            NN_NFC_SERVER_SAVE_DATA_RESULT_DO(m_FileStream[i]->Initialize(path, mode));
            *pOutFileHandle = i;
            NN_RESULT_SUCCESS;
        }
    }

    NN_ABORT();
}

nn::Result SaveData::SetPosition(int fileHandle, int64_t position) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    NN_ABORT_UNLESS(m_IsInitialized);
    NN_RESULT_DO(m_Result);
    NN_ABORT_UNLESS(m_FileStream[fileHandle]);

    NN_NFC_SERVER_SAVE_DATA_RETURN(m_FileStream[fileHandle]->SetPosition(position));
}

nn::Result SaveData::Read(size_t* pOutSize, void* pOutBuffer, int fileHandle, size_t bufferSize) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    NN_ABORT_UNLESS(m_IsInitialized);
    NN_RESULT_DO(m_Result);
    NN_ABORT_UNLESS(m_FileStream[fileHandle]);

    NN_NFC_SERVER_SAVE_DATA_RETURN(m_FileStream[fileHandle]->Read(pOutSize, pOutBuffer, bufferSize));
}

nn::Result SaveData::Read(void* pOutBuffer, int fileHandle, size_t readAndBufferSize) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    size_t readSize;
    auto result = Read(&readSize, pOutBuffer, fileHandle, readAndBufferSize);
    if(result.IsSuccess() && readSize == readAndBufferSize)
    {
        NN_RESULT_SUCCESS;
    }

    return nn::nfc::ResultBackupSystemError();
}

nn::Result SaveData::Write(int fileHandle, const void* pData, size_t dataSize, bool flush) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    NN_ABORT_UNLESS(m_IsInitialized);
    NN_RESULT_DO(m_Result);
    NN_ABORT_UNLESS(m_FileStream[fileHandle]);

    NN_NFC_SERVER_SAVE_DATA_RETURN(m_FileStream[fileHandle]->Write(pData, dataSize, flush));
}

nn::Result SaveData::GetSize(int64_t* pOutSize, int fileHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    NN_ABORT_UNLESS(m_IsInitialized);
    NN_RESULT_DO(m_Result);
    NN_ABORT_UNLESS(m_FileStream[fileHandle]);

    NN_NFC_SERVER_SAVE_DATA_RETURN(m_FileStream[fileHandle]->GetSize(pOutSize));
}

nn::Result SaveData::Flush(int fileHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    NN_ABORT_UNLESS(m_IsInitialized);
    NN_RESULT_DO(m_Result);

    NN_ABORT_UNLESS(m_FileStream[fileHandle]);

    NN_NFC_SERVER_SAVE_DATA_RETURN(m_FileStream[fileHandle]->Flush());
}

nn::Result SaveData::Close(int fileHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    NN_ABORT_UNLESS(m_IsInitialized);
    NN_RESULT_DO(m_Result);

    if(m_FileStream[fileHandle])
    {
        m_FileStream[fileHandle].reset(nullptr);
        NN_RESULT_SUCCESS;
    }

    NN_ABORT();
}

nn::Result SaveData::Commit() NN_NOEXCEPT
{
    NN_NFC_SERVER_SAVE_DATA_RETURN(nn::fs::CommitSaveData(SystemSaveDataArchiveName));
}

} // end of namespace core
} // end of namespace server
} // end of namespace nfc
} // end of namespace nn
