﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <memory>
#include <nn/result/result_HandlingUtility.h>
#include <nn/hid/system/hid_Result.h>
#include "nfc_LocalUtil.h"

namespace nn { namespace nfc { namespace server { namespace core {

namespace
{
}


//to Xcd
void ConvertToXcdNfcProtocol(nn::xcd::NfcProtocol* pOutDst, const nn::nfc::NfcProtocol& src) NN_NOEXCEPT
{
    switch(src)
    {
    case nn::nfc::NfcProtocol_TypeA:
        {
            *pOutDst = nn::xcd::NfcProtocol_TypeA;
        }
        break;
    case nn::nfc::NfcProtocol_TypeB:
        {
            *pOutDst = nn::xcd::NfcProtocol_TypeB;
        }
        break;
    case nn::nfc::NfcProtocol_TypeF:
        {
            *pOutDst = nn::xcd::NfcProtocol_TypeF;
        }
        break;
    case nn::nfc::NfcProtocol_Type15693:
        {
            *pOutDst = nn::xcd::NfcProtocol_Type15693;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void ConvertToXcdPollingMask(nn::xcd::NfcPollingMask* pOutDst, const nn::nfc::NfcProtocol& src) NN_NOEXCEPT
{
    if(src == nn::nfc::NfcProtocol_All)
    {
        *pOutDst = nn::xcd::NfcPollingMask_All;
        return;
    }

    std::memset(pOutDst, 0, sizeof(*pOutDst));

    if(src & nn::nfc::NfcProtocol_TypeA)
    {
        *pOutDst = static_cast<nn::xcd::NfcPollingMask>(*pOutDst | nn::xcd::NfcPollingMask_TechnologyA);
    }

    if(src & nn::nfc::NfcProtocol_TypeB)
    {
        *pOutDst = static_cast<nn::xcd::NfcPollingMask>(*pOutDst | nn::xcd::NfcPollingMask_TechnologyB);
    }

    if(src & nn::nfc::NfcProtocol_TypeF)
    {
        *pOutDst = static_cast<nn::xcd::NfcPollingMask>(*pOutDst | nn::xcd::NfcPollingMask_TechnologyF);
    }

    if(src & nn::nfc::NfcProtocol_Type15693)
    {
        *pOutDst = static_cast<nn::xcd::NfcPollingMask>(*pOutDst | nn::xcd::NfcPollingMask_TechnologyIso15693);
    }
}

void ConvertToXcdTagType(nn::xcd::NfcTagType* pOutDst, const nn::nfc::TagType& src) NN_NOEXCEPT
{
    *pOutDst = static_cast<nn::xcd::NfcTagType>(src);
}

void ConvertToXcdTagId(nn::xcd::NfcTagId* pOutDst, const nn::nfc::TagId& src) NN_NOEXCEPT
{
    std::memset(pOutDst, 0, sizeof(*pOutDst));
    pOutDst->length = src.length;
    std::memcpy(pOutDst->uid, src.uid, src.length);
}

void ConvertToXcdTagInfo(nn::xcd::NfcTagInfo* pOutDst, const nn::nfc::TagInfo& src) NN_NOEXCEPT
{
    std::memset(pOutDst, 0, sizeof(*pOutDst));

    ConvertToXcdTagId(&pOutDst->tagId, src.tagId);
    ConvertToXcdNfcProtocol(&pOutDst->protocol, static_cast<nn::nfc::NfcProtocol>(src.protocol));
    ConvertToXcdTagType(&pOutDst->tagType, static_cast<nn::nfc::TagType>(src.type));
}

void ConvertToXcdDiscoveryParameter(nn::xcd::NfcDiscoveryParameter* pOutDst, const nn::nfc::server::core::DiscoveryParameter& src) NN_NOEXCEPT
{
    ConvertToXcdPollingMask(&pOutDst->pollingMask, src.protocolFilter);
    pOutDst->activationTimeout = src.activationTimeout;
    pOutDst->discoveryPeriod = src.discoveryPeriod;
}

void ConvertToXcdNtagReadParameter(nn::xcd::NtagReadParameter* pOutDst, const nn::nfc::server::core::NtagReadParameter& src) NN_NOEXCEPT
{
    pOutDst->timeoutMsec = src.timeoutMsec;
    pOutDst->isPasswordRequired = src.isPasswordRequired;
    ConvertToXcdTagId(&pOutDst->tagId, src.tagId);
    pOutDst->blockCount = src.blockCount;
    std::memcpy(pOutDst->addresses, &src.addresses, sizeof(pOutDst->addresses));
}

void ConvertToXcdNtagWriteParameter(nn::xcd::NtagWriteParameter* pOutDst, const nn::nfc::server::core::NtagWriteParameter& src) NN_NOEXCEPT
{
    pOutDst->timeoutMsec = src.timeoutMsec;
    pOutDst->isPasswordRequired = src.isPasswordRequired;
    pOutDst->type2TagVersion = src.type2TagVersion;
    ConvertToXcdTagId(&pOutDst->tagId, src.tagId);
    std::memcpy(&pOutDst->ntagWriteData, &src.ntagWriteData, sizeof(pOutDst->ntagWriteData));
}

void ConvertToXcdPassThruParameter(nn::xcd::NfcPassThruParameter* pOutDst, const nn::nfc::server::core::PassThruParameter& src) NN_NOEXCEPT
{
    pOutDst->timeoutMsec = src.timeoutMsec;
    ConvertToXcdTagId(&pOutDst->tagId, src.tagId);
    pOutDst->sendDataSize = src.sendDataSize;
    std::memcpy(pOutDst->sendData, &src.sendData, sizeof(pOutDst->sendData));
}

void ConvertToXcdMifareReadParameter(nn::xcd::MifareReadParameter* pOutDst, const nn::nfc::server::core::MifareReadParameter& src) NN_NOEXCEPT
{
    pOutDst->timeoutMsec = src.timeoutMsec;
    pOutDst->keyFormat = static_cast<nn::xcd::MifareKeyValueFormat>(src.keyFormat);
    ConvertToXcdTagId(&pOutDst->tagId, src.tagId);
    pOutDst->blockCount = src.blockCount;
    std::memcpy(pOutDst->blocks, &src.blocks, sizeof(pOutDst->blocks));
}

void ConvertToXcdMifareWriteParameter(nn::xcd::MifareWriteParameter* pOutDst, const nn::nfc::server::core::MifareWriteParameter& src) NN_NOEXCEPT
{
    pOutDst->timeoutMsec = src.timeoutMsec;
    pOutDst->keyFormat = static_cast<nn::xcd::MifareKeyValueFormat>(src.keyFormat);
    ConvertToXcdTagId(&pOutDst->tagId, src.tagId);
    pOutDst->blockCount = src.blockCount;
    std::memcpy(pOutDst->blocks, &src.blocks, sizeof(pOutDst->blocks));
}

//to Nfc
void ConvertToNfcNfcProtocol(nn::nfc::NfcProtocol* pOutDst, const nn::xcd::NfcProtocol& src) NN_NOEXCEPT
{
    switch(src)
    {
    case nn::xcd::NfcProtocol_TypeA:
        {
            *pOutDst = nn::nfc::NfcProtocol_TypeA;
        }
        break;
    case nn::xcd::NfcProtocol_TypeB:
        {
            *pOutDst = nn::nfc::NfcProtocol_TypeB;
        }
        break;
    case nn::xcd::NfcProtocol_TypeF:
        {
            *pOutDst = nn::nfc::NfcProtocol_TypeF;
        }
        break;
    case nn::xcd::NfcProtocol_Type15693:
        {
            *pOutDst = nn::nfc::NfcProtocol_Type15693;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void ConvertToNfcTagType(nn::nfc::TagType* pOutDst, const nn::xcd::NfcTagType& src) NN_NOEXCEPT
{
    *pOutDst = static_cast<nn::nfc::TagType>(src);
}

void ConvertToNfcTagId(nn::nfc::TagId* pOutDst, const nn::xcd::NfcTagId& src) NN_NOEXCEPT
{
    std::memset(pOutDst, 0, sizeof(*pOutDst));
    pOutDst->length = static_cast<uint8_t>(src.length);
    std::memcpy(pOutDst->uid, src.uid, src.length);
}

void ConvertToNfcTagInfo(nn::nfc::TagInfo* pOutDst, const nn::xcd::NfcTagInfo& src) NN_NOEXCEPT
{
    std::memset(pOutDst, 0, sizeof(*pOutDst));
    ConvertToNfcTagId(&pOutDst->tagId, src.tagId);

    nn::nfc::NfcProtocol protocol;
    ConvertToNfcNfcProtocol(&protocol, src.protocol);
    pOutDst->protocol = static_cast<nn::Bit32>(protocol);

    nn::nfc::TagType type;
    ConvertToNfcTagType(&type, src.tagType);
    pOutDst->type = static_cast<nn::Bit32>(type);
}

void ConvertToNfcNtagData(nn::nfc::server::core::NtagData* pOutDst, const nn::xcd::NfcNtagData& src) NN_NOEXCEPT
{
    ConvertToNfcTagInfo(&pOutDst->tagInfo, src.tagInfo);
    pOutDst->type2TagVersion = src.type2TagVersion;
    std::memcpy(pOutDst->signature, src.signature, sizeof(pOutDst->signature));
    pOutDst->blockCount = src.blockCount;
    std::memcpy(pOutDst->readDataBlocks, src.readDataBlocks, sizeof(pOutDst->readDataBlocks));
}

void ConvertToNfcPassThruData(nn::nfc::server::core::PassThruData* pOutDst, const nn::xcd::NfcPassThruData& src) NN_NOEXCEPT
{
    ConvertToNfcTagInfo(&pOutDst->tagInfo, src.tagInfo);
    std::memcpy(pOutDst->responseData, src.responseData, sizeof(pOutDst->responseData));
    pOutDst->responseSize = src.responseSize;
}

void ConvertToNfcInfo(nn::nfc::server::core::Info* pOutDst, const nn::xcd::NfcInfo& src) NN_NOEXCEPT
{
    pOutDst->reason = src.reason;
    if(pOutDst->reason == nn::xcd::NfcEventReason_Detected)
    {
        ConvertToNfcTagInfo(&pOutDst->tagInfo, src.tagInfo);
    }
    else if(pOutDst->reason == nn::xcd::NfcEventReason_ReadFinish)
    {
        ConvertToNfcNtagData(&pOutDst->ntagData, src.ntagData);
    }
    else if(pOutDst->reason == nn::xcd::NfcEventReason_PassThruResult)
    {
        ConvertToNfcPassThruData(&pOutDst->passThruData, src.passThruData);
    }
    else
    {
        std::memcpy(pOutDst->_dummy, src._dummy, sizeof(pOutDst->_dummy));
    }
}

nn::Result ConvertToNfcResult(nn::Result result) NN_NOEXCEPT
{
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(nn::xcd::ResultNotConnected)
        {
            NN_RESULT_THROW(nn::nfc::ResultNfcDeviceNotFound());
        }
        NN_RESULT_CATCH(nn::xcd::ResultInvalidMcuState)
        {
            NN_RESULT_THROW(nn::nfc::ResultInvalidDeviceState());
        }
        NN_RESULT_CATCH(nn::xcd::ResultMcuBusy)
        {
            NN_RESULT_THROW(nn::nfc::ResultInvalidDeviceState());
        }
        NN_RESULT_CATCH(nn::xcd::ResultMcuVersionNotAvailable)
        {
            NN_RESULT_THROW(nn::nfc::ResultInvalidDeviceState());
        }
        NN_RESULT_CATCH(nn::xcd::ResultMcuFirmwareCorrupted)
        {
            NN_RESULT_THROW(nn::nfc::ResultNeedUpdate());
        }
        NN_RESULT_CATCH(nn::xcd::ResultNotSupported)
        {
            NN_RESULT_THROW(nn::nfc::ResultInvalidDeviceState());
        }
        NN_RESULT_CATCH(nn::xcd::ResultLowBattery)
        {
            NN_RESULT_THROW(nn::nfc::ResultNeedCharge());
        }
        NN_RESULT_CATCH(nn::xcd::ResultMcuHardwareError)
        {
            NN_RESULT_THROW(nn::nfc::ResultNfcDeviceError());
        }
        NN_RESULT_CATCH(nn::hid::system::ResultNoNfcDeviceFoundOnNpad)
        {
            NN_RESULT_THROW(nn::nfc::ResultNfcDeviceNotFound());
        }
        NN_RESULT_CATCH(nn::hid::system::ResultNfcActivatedOnOtherNpad)
        {
            NN_RESULT_THROW(nn::nfc::ResultMaxNfcDeviceActivated());
        }
        NN_RESULT_CATCH(nn::hid::system::ResultNfcActivateFailureNpadBusy)
        {
            NN_RESULT_THROW(nn::nfc::ResultConflictFunction());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_THROW(nn::nfc::ResultUnknownError());
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

bool GetRequiredMcuVersion(nn::xcd::McuVersionDataForNfc* pOutRequiredVersion, nn::xcd::DeviceType deviceType, const nn::nfc::server::core::McuVersionData& mcuVersionData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutRequiredVersion);

    // 渡された deviceType に対応する要求バージョン情報を取得
    for (decltype(mcuVersionData.count) i = 0; i < mcuVersionData.count; i++)
    {
        const auto& version = mcuVersionData.version[i];
        if (version.deviceType == deviceType)
        {
            *pOutRequiredVersion = version;
            return true;
        }
    }

    return false;
}

nn::Result SetDataFormatSync(nn::xcd::PeriodicDataFormat targetDataFormat, nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    nn::Result result;
    nn::xcd::PeriodicDataFormat currentFormat;
    int count;

    count = 0;
    do
    {
        if (0 < count)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(XcdRetryInterval));
        }
        result = nn::xcd::SetDataFormat(targetDataFormat, handle);
        ++count;
    } while (nn::xcd::ResultMcuBusy::Includes(result) && count < XcdRetryCountMax);

    NN_NFC_SERVER_CORE_RESULT_DO(result);


    count = 0;
    do
    {
        if (0 < count)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(XcdRetryInterval));
        }
        NN_NFC_SERVER_CORE_RESULT_DO(nn::xcd::GetDataFormat(&currentFormat, handle));
        ++count;

        // 接続台数が多い場合、Bluetooth の TSI 変更と hid の DataFormat 変更で 3 秒を超える場合があるため、
        // タイムアウトを長めにする
    } while (currentFormat != targetDataFormat && count < XcdSetDataFormatWaitCountMax);

    NN_RESULT_THROW_UNLESS(
        currentFormat == targetDataFormat,
        nn::nfc::ResultInvalidDeviceState());

    NN_RESULT_SUCCESS;
}

nn::Result SetMcuStateWithRetry(nn::xcd::McuState targetMcuState, nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    nn::Result result;
    int count;

    count = 0;
    do
    {
        if (0 < count)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(XcdRetryInterval));
        }
        result = nn::xcd::SetMcuState(targetMcuState, handle);
        ++count;
    } while (nn::xcd::ResultMcuBusy::Includes(result) && count < XcdRetryCountMax);

    NN_NFC_SERVER_CORE_RETURN(result);
}

nn::Result WaitMcuState(nn::xcd::McuState targetMcuState, nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    nn::Result result;
    nn::xcd::McuState currentMcuState;
    int count;

    count = 0;
    do
    {
        if (0 < count)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(XcdRetryInterval));
        }
        NN_NFC_SERVER_CORE_RESULT_DO(nn::xcd::GetMcuState(&currentMcuState, handle));
        ++count;
    } while (currentMcuState != targetMcuState && count < XcdSetMcuStateWaitCountMax);

    NN_RESULT_THROW_UNLESS(
        currentMcuState == targetMcuState,
        nn::nfc::ResultNfcDeviceError());

    NN_RESULT_SUCCESS;
}

nn::Result SetMcuStateSync(nn::xcd::McuState targetMcuState, nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    NN_RESULT_DO(SetMcuStateWithRetry(targetMcuState, handle));
    return WaitMcuState(targetMcuState, handle);
}

nn::Result StartDiscoveryWithRetry(const nn::nfc::server::core::DiscoveryParameter& discoveryParameter, nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    nn::xcd::NfcDiscoveryParameter xcdDiscoveryParameter;
    ConvertToXcdDiscoveryParameter(&xcdDiscoveryParameter, discoveryParameter);

    nn::Result result;
    int count = 0;
    do
    {
        if(0 < count)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(XcdRetryInterval));
        }
        result = nn::xcd::StartNfcDiscovery(xcdDiscoveryParameter, handle);
        ++count;
    } while(nn::xcd::ResultMcuBusy::Includes(result) && count < XcdRetryCountMax);

    NN_NFC_SERVER_CORE_RETURN(result);
}

nn::Result StopDiscoveryWithRetry(nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    nn::Result result;
    int count = 0;
    do
    {
        if(0 < count)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(XcdRetryInterval));
        }
        result = nn::xcd::StopNfcDiscovery(handle);
        ++count;
    } while(nn::xcd::ResultMcuBusy::Includes(result) && count < XcdRetryCountMax);

    NN_NFC_SERVER_CORE_RETURN(result);
}

nn::Result StartNtagReadWithRetry(const nn::nfc::server::core::NtagReadParameter& ntagReadParameter, nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    std::unique_ptr<nn::xcd::NtagReadParameter> xcdNtagReadParameter(new nn::xcd::NtagReadParameter());
    ConvertToXcdNtagReadParameter(xcdNtagReadParameter.get(), ntagReadParameter);

    nn::Result result;
    int count = 0;
    do
    {
        if(0 < count)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(XcdRetryInterval));
        }
        result = nn::xcd::StartNtagRead(*(xcdNtagReadParameter.get()), handle);
        ++count;
    } while(nn::xcd::ResultMcuBusy::Includes(result) && count < XcdRetryCountMax);

    NN_NFC_SERVER_CORE_RETURN(result);
}

nn::Result StartNtagWriteWithRetry(const nn::nfc::server::core::NtagWriteParameter& ntagWriteParameter, nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    std::unique_ptr<nn::xcd::NtagWriteParameter> xcdNtagWriteParameter(new nn::xcd::NtagWriteParameter());
    ConvertToXcdNtagWriteParameter(xcdNtagWriteParameter.get(), ntagWriteParameter);

    nn::Result result;
    int count = 0;
    do
    {
        if(0 < count)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(XcdRetryInterval));
        }
        result = nn::xcd::StartNtagWrite(*(xcdNtagWriteParameter.get()), handle);
        ++count;
    } while(nn::xcd::ResultMcuBusy::Includes(result) && count < XcdRetryCountMax);

    NN_NFC_SERVER_CORE_RETURN(result);
}

nn::Result SendRawDataWithRetry(const nn::nfc::server::core::PassThruParameter& passThruParameter, nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    std::unique_ptr<nn::xcd::NfcPassThruParameter> xcdPassThruParameter(new nn::xcd::NfcPassThruParameter());
    ConvertToXcdPassThruParameter(xcdPassThruParameter.get(), passThruParameter);

    nn::Result result;
    int count = 0;
    do
    {
        if(0 < count)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(XcdRetryInterval));
        }
        result = nn::xcd::SendNfcRawData(*(xcdPassThruParameter.get()), handle);
        ++count;
    } while(nn::xcd::ResultMcuBusy::Includes(result) && count < XcdRetryCountMax);

    NN_NFC_SERVER_CORE_RETURN(result);
}

nn::Result GetMcuVersionForNfcWithRetry(nn::xcd::McuVersionDataForNfc* pOutMcuVersionData, nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    nn::Result result;
    int count = 0;
    do
    {
        if(0 < count)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(XcdRetryInterval));
        }
        result = nn::xcd::GetMcuVersionForNfc(pOutMcuVersionData, handle);
        ++count;
    } while(nn::xcd::ResultMcuVersionNotAvailable::Includes(result) && count < 110); //XcdRetryInterval * count = 5.5秒となるように。MCU ハードウェア異常の判定に 5 秒程度かかるため。

    NN_NFC_SERVER_CORE_RETURN(result);
}

nn::Result StartMifareReadWithRetry(const nn::nfc::server::core::MifareReadParameter& mifareReadParameter, nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    std::unique_ptr<nn::xcd::MifareReadParameter> xcdMifareReadParameter(new nn::xcd::MifareReadParameter());
    ConvertToXcdMifareReadParameter(xcdMifareReadParameter.get(), mifareReadParameter);

    nn::Result result;
    int count = 0;
    do
    {
        if(0 < count)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(XcdRetryInterval));
        }
        result = nn::xcd::StartMifareRead(*(xcdMifareReadParameter.get()), handle);
        ++count;
    } while(nn::xcd::ResultMcuBusy::Includes(result) && count < XcdRetryCountMax);

    NN_NFC_SERVER_CORE_RETURN(result);
}

nn::Result StartMifareWriteWithRetry(const nn::nfc::server::core::MifareWriteParameter& mifareWriteParameter, nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    std::unique_ptr<nn::xcd::MifareWriteParameter> xcdMifareWriteParameter(new nn::xcd::MifareWriteParameter());
    ConvertToXcdMifareWriteParameter(xcdMifareWriteParameter.get(), mifareWriteParameter);

    nn::Result result;
    int count = 0;
    do
    {
        if(0 < count)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(XcdRetryInterval));
        }
        result = nn::xcd::StartMifareWrite(*(xcdMifareWriteParameter.get()), handle);
        ++count;
    } while(nn::xcd::ResultMcuBusy::Includes(result) && count < XcdRetryCountMax);

    NN_NFC_SERVER_CORE_RETURN(result);
}

}}}}  // namespace nn::nfc::server::core
