﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nfc/server/core/nfc_CoreService.h>
#include <nn/nfc/nfc_Result.h>
#include <nn/nfc/nfc_PrivateResult.h>
#include <nn/nfc/nfc_NfpResult.internal.h>
#include <nn/nfc/server/core/nfc_Result.h>
#include <nn/nfc/server/util/nfc_UtilUtil.h>

namespace nn { namespace nfc { namespace server { namespace core {

namespace
{

void GetMcuVersionDataFromInArray(McuVersionData* pOutVersionData, const McuVersionDataInArray versionDataInArray) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutVersionData);

    auto count = versionDataInArray.GetLength();
    NN_SDK_REQUIRES_MINMAX(count, 1ul, McuVersionDataCountMax);

    const auto* pData = versionDataInArray.GetData();

    for (decltype(count) i = 0; i < count; i++)
    {
        pOutVersionData->version[i] = pData[i];
    }
    pOutVersionData->count = count;
}

}

//
Service* Service::NfcProcessService = nullptr;

Service::Service() NN_NOEXCEPT
{
}

Service::Service(LibraryType libType) NN_NOEXCEPT : m_LibType(libType), m_Aruid(nn::applet::AppletResourceUserId::GetInvalidId()), m_Pid(nn::os::ProcessId::GetInvalidId())
{
    nn::os::CreateSystemEvent(&m_AvailabilityChangeEvent,
                              nn::os::EventClearMode_ManualClear,
                              true);
}

Service::~Service() NN_NOEXCEPT
{
    Finalize();
    nn::os::DestroySystemEvent(&m_AvailabilityChangeEvent);
}

void Service::SetLibraryType(LibraryType libType) NN_NOEXCEPT
{
    m_LibType = libType;
}

LibraryType Service::GetLibraryType() NN_NOEXCEPT
{
    return m_LibType;
}

nn::applet::AppletResourceUserId Service::GetAppletResourceUserId() NN_NOEXCEPT
{
    return m_Aruid;
}

nn::os::ProcessId Service::GetPid() NN_NOEXCEPT
{
    return m_Pid;
}

McuVersionData Service::GetMcuVersionData() NN_NOEXCEPT
{
    return m_McuVersionData;
}

void Service::Initialize(nn::applet::AppletResourceUserId aruid, nn::Bit64 pid, const McuVersionDataInArray& mcuVersionData) NN_NOEXCEPT
{
    m_Aruid = aruid;
    m_Pid.value = pid;
    GetMcuVersionDataFromInArray(&m_McuVersionData, mcuVersionData);
}

void Service::Finalize() NN_NOEXCEPT
{
    auto itr = m_DeviceEventList.begin();
    while(itr != m_DeviceEventList.end())
    {
        DeviceEvent* temporary = &(*itr);
        itr = m_DeviceEventList.erase(itr);
        delete(temporary);
    }
}

nn::Result Service::SetDeviceEvent(const nn::nfc::DeviceHandle& deviceHandle, nn::nfc::DeviceHandle* devices, int count) NN_NOEXCEPT
{
    bool find;

    //不要なデバイスのイベントをクリア
    for(auto& deviceEvent : m_DeviceEventList)
    {
        if(deviceEvent.IsInitialized())
        {
            find = false;
            for(auto j = 0; j < count; ++j)
            {
                if(deviceEvent.IsEqual(devices[j]))
                {
                    find = true;
                }
            }
            if(!find)
            {
                nn::os::ClearSystemEvent(&deviceEvent.GetActivateEvent());
                nn::os::ClearSystemEvent(&deviceEvent.GetDeactivateEvent());
            }
        }
    }

    //指定されたデバイスが存在するか
    find = false;
    for(auto i = 0; i < count; ++i)
    {
        if(nn::nfc::server::util::IsEqualDeviceHandle(deviceHandle, devices[i]))
        {
            find = true;
            break;
        }
    }
    if(!find)
    {
        return nn::nfc::ResultNfcDeviceNotFound();
    }

    //既にあるか
    for(auto& deviceEvent : m_DeviceEventList)
    {
        if(deviceEvent.IsInitialized())
        {
            if(deviceEvent.IsEqual(deviceHandle))
            {
                NN_RESULT_SUCCESS;
            }
        }
    }

    DeviceEvent* deviceEvent = new DeviceEvent();
    deviceEvent->Initialize(deviceHandle);
    m_DeviceEventList.push_back(*deviceEvent);

    NN_RESULT_SUCCESS;
}

nn::Result Service::GetActivateEventHandle(nn::os::NativeHandle* handle, const nn::nfc::DeviceHandle& deviceHandle, nn::nfc::DeviceHandle* devices, int count) NN_NOEXCEPT
{
    NN_RESULT_DO(SetDeviceEvent(deviceHandle, devices, count));

    for(auto& deviceEvent : m_DeviceEventList)
    {
        if(deviceEvent.IsInitialized())
        {
            if(deviceEvent.IsEqual(deviceHandle))
            {
                *handle = nn::os::GetReadableHandleOfSystemEvent(&deviceEvent.GetActivateEvent());
                NN_RESULT_SUCCESS;
            }
        }
    }

    return nn::nfc::ResultNfcDeviceNotFound();
}

nn::Result Service::GetDeactivateEventHandle(nn::os::NativeHandle* handle, const nn::nfc::DeviceHandle& deviceHandle, nn::nfc::DeviceHandle* devices, int count) NN_NOEXCEPT
{
    NN_RESULT_DO(SetDeviceEvent(deviceHandle, devices, count));

    for(auto& deviceEvent : m_DeviceEventList)
    {
        if(deviceEvent.IsInitialized())
        {
            if(deviceEvent.IsEqual(deviceHandle))
            {
                *handle = nn::os::GetReadableHandleOfSystemEvent(&deviceEvent.GetDeactivateEvent());
                NN_RESULT_SUCCESS;
            }
        }
    }

    return nn::nfc::ResultNfcDeviceNotFound();
}

nn::Result Service::GetAvailabilityChangeEventHandle(nn::os::NativeHandle* handle) NN_NOEXCEPT
{
    *handle = nn::os::GetReadableHandleOfSystemEvent(&m_AvailabilityChangeEvent);
    NN_RESULT_SUCCESS;
}

void Service::SignalActivateEvent(const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    for(auto& deviceEvent : m_DeviceEventList)
    {
        if(deviceEvent.IsInitialized())
        {
            if(deviceEvent.IsEqual(deviceHandle))
            {
                nn::os::SignalSystemEvent(&deviceEvent.GetActivateEvent());
                return;
            }
        }
    }
}

void Service::SignalDeactivateEvent(const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    for(auto& deviceEvent : m_DeviceEventList)
    {
        if(deviceEvent.IsInitialized())
        {
            if(deviceEvent.IsEqual(deviceHandle))
            {
                nn::os::SignalSystemEvent(&deviceEvent.GetDeactivateEvent());
                return;
            }
        }
    }
}

void Service::SignalAvailabilityChangeEvent() NN_NOEXCEPT
{
    nn::os::SignalSystemEvent(&m_AvailabilityChangeEvent);
}

void Service::ClearDetectEvent(const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    for(auto& deviceEvent : m_DeviceEventList)
    {
        if(deviceEvent.IsInitialized())
        {
            if(deviceEvent.IsEqual(deviceHandle))
            {
                nn::os::ClearSystemEvent(&deviceEvent.GetActivateEvent());
                nn::os::ClearSystemEvent(&deviceEvent.GetDeactivateEvent());
                return;
            }
        }
    }
}

}}}}  // namespace nn::nfc::server::core
