﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/xcd/xcd.h>
#include <nn/hid/system/hid_Nfc.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nfc/server/core/nfc_CoreManager.h>
#include "nfc_CoreDevice.h"
#include <nn/nfc/server/util/nfc_UtilUtil.h>
#include "nfc_LocalUtil.h"
#include <nn/nfc/server/util/nfc_ScopedMutexLock.h>

namespace nn { namespace nfc { namespace server { namespace core {

namespace
{
const uint16_t s_ActivationTimeout = 1000;//TEMP
const size_t s_EventMessageQueueCapacity = 16;
const size_t s_AccessEventMessageQueueCapacity = 4;

void CancelTimer(nn::os::TimerEventType* timerEventType)
{
    nn::os::StopTimerEvent(timerEventType);
    nn::os::ClearTimerEvent(timerEventType);
}

//タグ喪失扱いにするイベントかどうか
bool IsDeactivated(const Info& info) NN_NOEXCEPT
{
    if(info.reason == nn::xcd::NfcEventReason_ControllerDisconnected)
    {
        return true;
    }
    else if(info.reason == nn::xcd::NfcEventReason_Error
            && (info.errorInfo.resultCode == nn::xcd::NfcResultCode_FunctionError
                || info.errorInfo.resultCode == nn::xcd::NfcResultCode_ResetRequired
                || info.errorInfo.resultCode == nn::xcd::NfcResultCode_Timeout
                || info.errorInfo.resultCode == nn::xcd::NfcResultCode_CommandTimeout
                || info.errorInfo.resultCode == nn::xcd::NfcResultCode_SequenceError))
    {
        return true;
    }

    return false;
}

}

NN_DISABLE_WARNING_ARRAY_DEFAULT_INITIALIZATION_IN_CONSTRUCTOR;

Device::Device(uint64_t uniqueId, nn::hid::NpadIdType npadId, nn::os::MultiWaitType* eventMultiWait) NN_NOEXCEPT
    : m_State(Device::State_None)
    , m_UniqueId(uniqueId)
    , m_Handle()
    , m_XcdDeviceHandle()
    , m_pEventMultiWait(eventMultiWait)
    , m_EventMessageQueue(s_EventMessageQueueCapacity)
    , m_EventMessageQueueHolder()
    , m_DetectEvent()
    , m_DetectEventHolder()
    , m_pDetectEventForNfcUser(nullptr)
    , m_Event()
    , m_EventHolder()
    , m_pEventForNfcUser(nullptr)
    , m_IntermittentLowTerminateEvent()
    , m_IntermittentLowTerminateHolder()
    , m_AccessEventMessageQueue(s_AccessEventMessageQueueCapacity)
    , m_NormalDiscoveryEvent()
    , m_NormalDiscoveryResult()
    , m_NfcActivateEvent()
    , m_NfcActivateCancelEvent()
    , m_DetectedTag()
    , m_Mutex()
    , m_DriverEventMutex()
    , m_Info()
    , m_DetectInfo()
    , m_ResetInfo()
    , m_IsRunningThread(false)
    , m_EventThreadState(EventThreadState_NotInitialized)
    , m_DriverEventState(DriverEventState_NotInitialized)
    , m_DriverEventRequest(DriverEventRequestType_None)
    , m_DiscoveryParameter()
    , m_McuVersionData()
    , m_IsEnableKeepSession(false)
    , m_IsFunctionError(false)
    , m_InitializedCount(0)
    , m_IsInitializedDriver(false)
    , m_pResetEventForNfcUser(nullptr)
    , m_ActiveService(nullptr)
{
    nn::os::InitializeMutex(&m_Mutex, true, 0);
    m_DriverEventMutex.Initialize();

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    std::memset(&m_Handle, 0, sizeof(m_Handle));
    m_Handle._id = static_cast<nn::Bit32>(npadId);

    //イベントスレッド状態初期化
    SetEventThreadState(Device::EventThreadState_NotInitialized);

    //イベントスレッド終了用イベント
    //タグアクセス開始要求
    //タグアクセス終了要求
    //通常 Discover 開始要求
    //通常 Discover 終了要求
    //イベントスレッドリセットイベント
    //以上を扱うメッセージキュー
    //初期化リストで初期化済み
    nn::os::InitializeMultiWaitHolder(&m_EventMessageQueueHolder, m_EventMessageQueue.GetBase(), nn::os::MessageQueueWaitType_WaitForNotEmpty);
    nn::os::LinkMultiWaitHolder(m_pEventMultiWait, &m_EventMessageQueueHolder);

    //間欠動作 Low 期間終了イベント
    nn::os::InitializeTimerEvent(&m_IntermittentLowTerminateEvent, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&m_IntermittentLowTerminateHolder, &m_IntermittentLowTerminateEvent);
    nn::os::LinkMultiWaitHolder(m_pEventMultiWait, &m_IntermittentLowTerminateHolder);

    //アクセス対象タグ発見イベント
    //アクセス対象タグ喪失イベント
    //以上を扱うメッセージキュー
    //初期化リストで初期化済み


    //通常 Discover 開始・終了受領イベント
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateSystemEvent(&m_NormalDiscoveryEvent,
                                                             nn::os::EventClearMode_ManualClear,
                                                             false));

    //Activate Nfc イベント
    nn::hid::system::BindNfcActivateEvent(npadId, &m_NfcActivateEvent, nn::os::EventClearMode_ManualClear);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateSystemEvent(&m_NfcActivateCancelEvent,
                                                             nn::os::EventClearMode_ManualClear,
                                                             false));

    m_IsRunningThread = true;
}

Device::~Device() NN_NOEXCEPT
{
    if(IsInitialized())
    {
        m_InitializedCount = 1;
        Finalize();
    }

    m_IsRunningThread = false;
    m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_Terminate));

    //Activate Nfc イベント
    nn::os::DestroySystemEvent(&m_NfcActivateEvent);
    nn::os::DestroySystemEvent(&m_NfcActivateCancelEvent);

    //通常 Discover 開始・終了受領イベント
    nn::os::DestroySystemEvent(&m_NormalDiscoveryEvent);

    //イベントスレッド終了用イベント
    //タグアクセス開始イベント
    //タグアクセス終了イベント
    //通常 Discover 開始要求
    //通常 Discover 終了要求
    //デストラクタで終了処理される
    nn::os::UnlinkMultiWaitHolder(&m_EventMessageQueueHolder);
    nn::os::FinalizeMultiWaitHolder(&m_EventMessageQueueHolder);

    //間欠動作 Low 期間終了イベント
    nn::os::UnlinkMultiWaitHolder(&m_IntermittentLowTerminateHolder);
    nn::os::FinalizeMultiWaitHolder(&m_IntermittentLowTerminateHolder);
    nn::os::FinalizeTimerEvent(&m_IntermittentLowTerminateEvent);

    nn::os::FinalizeMutex(&m_Mutex);
}

bool Device::IsInitialized() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    return m_InitializedCount > 0;
}

void Device::Initialize() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    NN_ABORT_UNLESS(!IsInitialized());

    SetEventThreadState(Device::EventThreadState_NotStartedDiscovery);
    m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_Reset));

    SetState(Device::State_Init);
    ++m_InitializedCount;
}

void Device::Finalize() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    NN_ABORT_UNLESS(IsInitialized());

    --m_InitializedCount;

    FinalizeDriver();
    FinalizeNpad();

    m_DetectedTag.reset(nullptr);

    SetState(Device::State_None);
    m_ActiveService = nullptr;
    SetEventThreadState(Device::EventThreadState_NotInitialized);
    m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_Reset));
}

void Device::Reset(ResetReason reason, Service* service) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(!IsInitialized())
    {
        return;
    }

    if(m_State == Device::State_Search
       || m_State == Device::State_Active
       || m_State == Device::State_Deactive)
    {
        if(m_ActiveService != nullptr && service != nullptr && m_ActiveService != service)
        {
            // 別のライブラリでデバイスを利用している場合はリセットしない
            return;
        }
    }

    FinalizeDriver();
    FinalizeNpad();

    SetState(Device::State_Init);

    m_ResetInfo.reason = reason;
    SignalResetEvent();
    m_ActiveService = nullptr;
    SetEventThreadState(Device::EventThreadState_NotStartedDiscovery);
    m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_Reset));

    m_NormalDiscoveryResult = nn::nfc::ResultInvalidDeviceState();
    nn::os::SignalSystemEvent(&m_NormalDiscoveryEvent);

    m_AccessEventMessageQueue.Clear();
    m_AccessEventMessageQueue.Send(static_cast<nn::Bit8>(Device::AccessEventType_Cancel));

    nn::os::SignalSystemEvent(&m_NfcActivateCancelEvent);
}

nn::Result Device::StartDiscovery(Service* service, const DiscoveryParameter& discoveryParameter, const McuVersionData& mcuVersionData) NN_NOEXCEPT
{
    nn::Result result;
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    NN_RESULT_DO(CheckFunctionError());

    if(m_State != Device::State_Init && m_State != Device::State_Deactive)
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    //リセットの後始末
    nn::os::ClearSystemEvent(&m_NormalDiscoveryEvent);
    m_AccessEventMessageQueue.Clear();
    nn::os::ClearSystemEvent(&m_NfcActivateCancelEvent);

    NN_RESULT_DO(InitializeNpad());

    result = InitializeDriver(mcuVersionData);
    if(result.IsFailure())
    {
        FinalizeNpad();
        return result;
    }

    m_DiscoveryParameter = discoveryParameter;
    m_McuVersionData = mcuVersionData;

    SetEventThreadState(Device::EventThreadState_NotStartedDiscovery);
    m_ActiveService = service;
    m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_NormalDiscoveryStart));

    lock.Unlock();

    if(!nn::os::TimedWaitSystemEvent(&m_NormalDiscoveryEvent, nn::TimeSpan::FromMilliSeconds(3000)))
    {
        FinalizeDriver();
        FinalizeNpad();
        m_ActiveService = nullptr;
        SetEventThreadState(Device::EventThreadState_NotStartedDiscovery);
        m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_Reset));
        return nn::nfc::ResultInvalidDeviceState();
    }

    lock.Lock();

    nn::os::ClearSystemEvent(&m_NormalDiscoveryEvent);

    if(m_NormalDiscoveryResult.IsSuccess())
    {
        SetState(Device::State_Search);
    }
    else
    {
        FinalizeDriver();
        FinalizeNpad();
        m_ActiveService = nullptr;
        SetEventThreadState(Device::EventThreadState_NotStartedDiscovery);
        m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_Reset));
    }

    return m_NormalDiscoveryResult;
}

nn::Result Device::StopDiscovery() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    NN_RESULT_DO(CheckFunctionError());

    if(m_State == Device::State_None || m_State == Device::State_Init || m_State == Device::State_Unexpected)
    {
        return nn::nfc::ResultInvalidDeviceState();
    }

    m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_NormalDiscoveryStop));

    lock.Unlock();

    bool isReceived;
    isReceived = nn::os::TimedWaitSystemEvent(&m_NormalDiscoveryEvent, nn::TimeSpan::FromMilliSeconds(3000));

    lock.Lock();

    if(!isReceived)
    {
        m_NormalDiscoveryResult = nn::nfc::ResultInvalidDeviceState();
    }

    nn::os::ClearSystemEvent(&m_NormalDiscoveryEvent);

    {
        FinalizeDriver();
        FinalizeNpad();

        SetState(Device::State_Init);
        m_ActiveService = nullptr;
        SetEventThreadState(Device::EventThreadState_NotStartedDiscovery);
        m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_Reset));
    }

    return m_NormalDiscoveryResult;
}

nn::Result Device::KeepSession() NN_NOEXCEPT
{
    {
        nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

        NN_RESULT_DO(CheckFunctionError());
        NN_RESULT_DO(CheckActive());

        if(m_IsEnableKeepSession)
        {
            //KeepSessionが既に呼び出し済み
            return nn::nfc::ResultInvalidDeviceState();
        }

        //DetectEventThread へアクセス開始を通知する
        m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_AccessStart));
    }

    //タグが見つかるのを待つ
    Device::AccessEventType accessEventType;
    bool isReceived = m_AccessEventMessageQueue.TimedReceive(reinterpret_cast<nn::Bit8*>(&accessEventType), nn::TimeSpan::FromMilliSeconds(1500));

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    nn::Result result;
    if(isReceived && accessEventType == Device::AccessEventType_Deteceted)
    {
        //タグが見つかった場合

        result = CheckActive();
        if(result.IsSuccess())
        {
            m_IsEnableKeepSession = true;
        }
    }
    else
    {
        //タグが見つからなかった場合

        result = CheckActive();
        if(result.IsSuccess())
        {
            if(!isReceived || accessEventType == Device::AccessEventType_Cancel)
            {
                //本来 ResultCode_Timeout を受けたときに行う処理(ワークアラウンド)
                DeactivateTag();
            }

            result = nn::nfc::ResultTagNotFound();
        }
    }

    if(result.IsFailure())
    {
        //失敗した場合は DetectEventThread へアクセス終了を通知する
        m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_AccessTerminate));
    }

    return result;
}

nn::Result Device::ReleaseSession() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(!m_IsEnableKeepSession)
    {
        //KeepSession を呼び出していない
        return nn::nfc::ResultInvalidDeviceState();
    }

    m_IsEnableKeepSession = false;

    //DetectEventThread へアクセス終了を通知する
    m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_AccessTerminate));

    NN_RESULT_SUCCESS;
}

void Device::SetEvent(nn::os::SystemEventType* pEvent, nn::os::SystemEventType* pDetectEvent, nn::os::SystemEventType* pResetEvent) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    m_pEventForNfcUser = pEvent;
    m_pDetectEventForNfcUser = pDetectEvent;
    m_pResetEventForNfcUser = pResetEvent;
}

void Device::SignalEvent() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(m_ActiveService == nullptr)
    {
        return;
    }

    if(m_pEventForNfcUser)
    {
        nn::os::SignalSystemEvent(m_pEventForNfcUser);
    }
}

void Device::SignalDetectEvent() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(m_ActiveService == nullptr)
    {
        return;
    }

    if(m_pDetectEventForNfcUser)
    {
        nn::os::SignalSystemEvent(m_pDetectEventForNfcUser);
    }
}

void Device::SignalResetEvent() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(m_ActiveService == nullptr)
    {
        return;
    }

    if(m_pResetEventForNfcUser)
    {
        nn::os::SignalSystemEvent(m_pResetEventForNfcUser);
    }
}

Info& Device::GetInfo() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    return m_Info;
}

Info& Device::GetDetectInfo() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    return m_DetectInfo;
}

ResetInfo& Device::GetResetInfo() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    return m_ResetInfo;
}

Tag* Device::GetTag(const nn::nfc::TagId& tagId) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(m_State != Device::State_Active)
    {
        return nullptr;
    }

    if(m_DetectedTag && nn::nfc::server::util::IsEqualTagId(m_DetectedTag->GetId(), tagId))
    {
        return m_DetectedTag.get();
    }

    return nullptr;
}

Ntag* Device::GetNtag(const nn::nfc::TagId& tagId) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    if(m_State != Device::State_Active)
    {
        return nullptr;
    }

    Tag* tag = GetTag(tagId);

    if(tag != nullptr && tag->GetProtocol() == nn::nfc::NfcProtocol_TypeA && tag->GetType() == nn::nfc::TagType_Type2)
    {
        return static_cast<Ntag*>(tag);
    }

    return nullptr;
}

nn::Result Device::CheckFunctionError() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(m_IsFunctionError)
    {
        return nn::nfc::ResultNfcDeviceError();
    }

    NN_RESULT_SUCCESS;
}

nn::Result Device::CheckActive() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(m_State != Device::State_Active)
    {
        if(m_State == Device::State_Deactive)
        {
            return nn::nfc::ResultTagNotFound();
        }
        return nn::nfc::ResultInvalidDeviceState();
    }

    NN_RESULT_SUCCESS;
}

Mifare* Device::GetMifare(const nn::nfc::TagId& tagId) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    if(m_State != Device::State_Active)
    {
        return nullptr;
    }

    Tag* tag = GetTag(tagId);

    if(tag != nullptr && tag->GetProtocol() == nn::nfc::NfcProtocol_TypeA && tag->GetType() == nn::nfc::TagType_Mifare)
    {
        return static_cast<Mifare*>(tag);
    }

    return nullptr;
}

uint64_t Device::GetUniqueId() NN_NOEXCEPT
{
    return m_UniqueId;
}

nn::nfc::DeviceHandle Device::GetHandle() NN_NOEXCEPT
{
    return m_Handle;
}

nn::Result Device::StartNtagReadCommon(const nn::nfc::TagId& tagId, const NtagReadParameter& ntagReadParameter) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    Ntag* ntag = GetNtag(tagId);
    if(ntag == nullptr)
    {
        return nn::nfc::ResultTagNotFound();
    }

    return ntag->StartRead(ntagReadParameter);
}

nn::Result Device::StartNtagRead(const nn::nfc::TagId& tagId, const NtagReadParameter& ntagReadParameter) NN_NOEXCEPT
{
    {
        nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

        NN_RESULT_DO(CheckFunctionError());
        NN_RESULT_DO(CheckXcdDeviceHandle());
        NN_RESULT_DO(CheckActive());

        if(m_IsEnableKeepSession)
        {
            //KeepSession にて、既にアクセスできる状態
            return StartNtagReadCommon(tagId, ntagReadParameter);
        }

        //DetectEventThread へアクセス開始を通知する
        m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_AccessStart));
    }

    //タグが見つかるのを待つ
    Device::AccessEventType accessEventType;
    bool isReceived = m_AccessEventMessageQueue.TimedReceive(reinterpret_cast<nn::Bit8*>(&accessEventType), nn::TimeSpan::FromMilliSeconds(1500));

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    nn::Result result;
    if(isReceived && accessEventType == Device::AccessEventType_Deteceted)
    {
        //タグが見つかった場合

        result = CheckActive();
        if(result.IsSuccess())
        {
            result = StartNtagReadCommon(tagId, ntagReadParameter);
        }
    }
    else
    {
        //タグが見つからなかった場合

        result = CheckActive();
        if(result.IsSuccess())
        {
            if(!isReceived || accessEventType == Device::AccessEventType_Cancel)
            {
                //本来 ResultCode_Timeout を受けたときに行う処理(ワークアラウンド)
                DeactivateTag();
            }

            result = nn::nfc::ResultTagNotFound();
        }
    }

    if(result.IsFailure())
    {
        //失敗した場合は DetectEventThread へアクセス終了を通知する
        m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_AccessTerminate));
    }

    return result;
}

nn::Result Device::StartNtagWriteCommon(const nn::nfc::TagId& tagId, const NtagWriteParameter& ntagWriteParameter) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    Ntag* ntag = GetNtag(tagId);
    if(ntag == nullptr)
    {
        return nn::nfc::ResultTagNotFound();
    }

    return ntag->StartWrite(ntagWriteParameter);
}

nn::Result Device::StartNtagWrite(const nn::nfc::TagId& tagId, const NtagWriteParameter& ntagWriteParameter) NN_NOEXCEPT
{
    {
        nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

        NN_RESULT_DO(CheckFunctionError());
        NN_RESULT_DO(CheckXcdDeviceHandle());
        NN_RESULT_DO(CheckActive());

        if(m_IsEnableKeepSession)
        {
            //KeepSession にて、既にアクセスできる状態
            return StartNtagWriteCommon(tagId, ntagWriteParameter);
        }

        //DetectEventThread へアクセス開始を通知する
        m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_AccessStart));
    }

    //タグが見つかるのを待つ
    Device::AccessEventType accessEventType;
    bool isReceived = m_AccessEventMessageQueue.TimedReceive(reinterpret_cast<nn::Bit8*>(&accessEventType), nn::TimeSpan::FromMilliSeconds(1500));

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    nn::Result result;
    if(isReceived && accessEventType == Device::AccessEventType_Deteceted)
    {
        //タグが見つかった場合

        result = CheckActive();
        if(result.IsSuccess())
        {
            result = StartNtagWriteCommon(tagId, ntagWriteParameter);
        }
    }
    else
    {
        //タグが見つからなかった場合

        result = CheckActive();
        if(result.IsSuccess())
        {
            if(!isReceived || accessEventType == Device::AccessEventType_Cancel)
            {
                //本来 ResultCode_Timeout を受けたときに行う処理(ワークアラウンド)
                DeactivateTag();
            }

            result = nn::nfc::ResultTagNotFound();
        }
    }

    if(result.IsFailure())
    {
        //失敗した場合は DetectEventThread へアクセス終了を通知する
        m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_AccessTerminate));
    }

    return result;
}

nn::Result Device::SendRawDataCommon(const nn::nfc::TagId& tagId, const PassThruParameter& passThruParameter) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    Tag* tag = Device::GetTag(tagId);
    if(tag == nullptr)
    {
        return nn::nfc::ResultTagNotFound();
    }

    return tag->SendRawData(passThruParameter);
}

nn::Result Device::SendRawData(const nn::nfc::TagId& tagId, const PassThruParameter& passThruParameter) NN_NOEXCEPT
{
    {
        nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

        NN_RESULT_DO(CheckFunctionError());
        NN_RESULT_DO(CheckXcdDeviceHandle());
        NN_RESULT_DO(CheckActive());

        if(m_IsEnableKeepSession)
        {
            //KeepSession にて、既にアクセスできる状態
            return SendRawDataCommon(tagId, passThruParameter);
        }

        //DetectEventThread へアクセス開始を通知する
        m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_AccessStart));
    }

    //タグが見つかるのを待つ
    Device::AccessEventType accessEventType;
    bool isReceived = m_AccessEventMessageQueue.TimedReceive(reinterpret_cast<nn::Bit8*>(&accessEventType), nn::TimeSpan::FromMilliSeconds(1500));

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    nn::Result result;
    if(isReceived && accessEventType == Device::AccessEventType_Deteceted)
    {
        //タグが見つかった場合
        result = CheckActive();
        if(result.IsSuccess())
        {
            result = SendRawDataCommon(tagId, passThruParameter);
        }
    }
    else
    {
        //タグが見つからなかった場合

        result = CheckActive();
        if(result.IsSuccess())
        {
            if(!isReceived || accessEventType == Device::AccessEventType_Cancel)
            {
                //本来 ResultCode_Timeout を受けたときに行う処理(ワークアラウンド)
                DeactivateTag();
            }

            result = nn::nfc::ResultTagNotFound();
        }
    }

    if(result.IsFailure())
    {
        //失敗した場合は DetectEventThread へアクセス終了を通知する
        m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_AccessTerminate));
    }

    return result;
}

nn::Result Device::GetTagInfo(nn::nfc::TagInfo* tagInfo, const nn::nfc::TagId& tagId) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    NN_RESULT_DO(CheckActive());

    Tag* tag = GetTag(tagId);
    if(tag == nullptr)
    {
        return nn::nfc::ResultTagNotFound();
    }

    tagInfo->tagId = tag->GetId();
    tagInfo->protocol = tag->GetProtocol();
    tagInfo->type = tag->GetType();

    NN_RESULT_SUCCESS;
}

nn::Result Device::GetNpadId(nn::hid::NpadIdType* pOutNpadId) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    *pOutNpadId = static_cast<nn::hid::NpadIdType>(m_Handle._id);
    NN_RESULT_SUCCESS;
}

nn::Result Device::GetXcdDeviceHandle(nn::xcd::DeviceHandle* pOutXcdDeviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    NN_NFC_SERVER_CORE_RETURN(nn::hid::system::GetXcdHandleForNpadWithNfc(pOutXcdDeviceHandle, static_cast<nn::hid::NpadIdType>(m_Handle._id)));
}

nn::Result Device::StartMifareReadCommon(const nn::nfc::TagId& tagId, const MifareReadParameter& mifareReadParameter) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    Mifare* mifare = GetMifare(tagId);
    if(mifare == nullptr)
    {
        return nn::nfc::ResultTagNotFound();
    }

    return mifare->StartRead(mifareReadParameter);
}

nn::Result Device::StartMifareRead(const nn::nfc::TagId& tagId, const MifareReadParameter& mifareReadParameter) NN_NOEXCEPT
{
    {
        nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

        if(m_IsFunctionError)
        {
            return nn::nfc::ResultNfcDeviceError();
        }

        if(m_State != Device::State_Active)
        {
            return nn::nfc::ResultInvalidDeviceState();
        }

        NN_RESULT_DO(CheckXcdDeviceHandle());

        if(m_IsEnableKeepSession)
        {
            //KeepSession にて、既にアクセスできる状態
            return StartMifareReadCommon(tagId, mifareReadParameter);
        }

        //DetectEventThread へアクセス開始を通知する
        m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_AccessStart));
    }

    //タグが見つかるのを待つ
    Device::AccessEventType accessEventType;
    bool isReceived = m_AccessEventMessageQueue.TimedReceive(reinterpret_cast<nn::Bit8*>(&accessEventType), nn::TimeSpan::FromMilliSeconds(1500));

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    nn::Result result;
    if(isReceived && accessEventType == Device::AccessEventType_Deteceted)
    {
        //タグが見つかった場合
        if(m_State != Device::State_Active)
        {
            result = nn::nfc::ResultInvalidDeviceState();
        }
        else
        {
            result = StartMifareReadCommon(tagId, mifareReadParameter);
        }
    }
    else
    {
        //タグが見つからなかった場合

        if(!isReceived)
        {
            //本来 ResultCode_Timeout を受けたときに行う処理(ワークアラウンド)
            DeactivateTag();
        }

        if(m_State != Device::State_Active)
        {
            result = nn::nfc::ResultInvalidDeviceState();
        }
        else
        {
            result = nn::nfc::ResultTagNotFound();
        }
    }

    if(result.IsFailure())
    {
        //失敗した場合は DetectEventThread へアクセス終了を通知する
        m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_AccessTerminate));
    }

    return result;
}

nn::Result Device::StartMifareWriteCommon(const nn::nfc::TagId& tagId, const MifareWriteParameter& mifareWriteParameter) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    Mifare* mifare = GetMifare(tagId);
    if(mifare == nullptr)
    {
        return nn::nfc::ResultTagNotFound();
    }

    return mifare->StartWrite(mifareWriteParameter);
}

nn::Result Device::StartMifareWrite(const nn::nfc::TagId& tagId, const MifareWriteParameter& mifareWriteParameter) NN_NOEXCEPT
{
    {
        nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

        if(m_IsFunctionError)
        {
            return nn::nfc::ResultNfcDeviceError();
        }

        if(m_State != Device::State_Active)
        {
            return nn::nfc::ResultInvalidDeviceState();
        }

        NN_RESULT_DO(CheckXcdDeviceHandle());

        if(m_IsEnableKeepSession)
        {
            //KeepSession にて、既にアクセスできる状態
            return StartMifareWriteCommon(tagId, mifareWriteParameter);
        }

        //DetectEventThread へアクセス開始を通知する
        m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_AccessStart));
    }

    //タグが見つかるのを待つ
    Device::AccessEventType accessEventType;
    bool isReceived = m_AccessEventMessageQueue.TimedReceive(reinterpret_cast<nn::Bit8*>(&accessEventType), nn::TimeSpan::FromMilliSeconds(1500));

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    nn::Result result;
    if(isReceived && accessEventType == Device::AccessEventType_Deteceted)
    {
        //タグが見つかった場合
        if(m_State != Device::State_Active)
        {
            result = nn::nfc::ResultInvalidDeviceState();
        }
        else
        {
            result = StartMifareWriteCommon(tagId, mifareWriteParameter);
        }
    }
    else
    {
        //タグが見つからなかった場合

        if(!isReceived)
        {
            //本来 ResultCode_Timeout を受けたときに行う処理(ワークアラウンド)
            DeactivateTag();
        }

        if(m_State != Device::State_Active)
        {
            result = nn::nfc::ResultInvalidDeviceState();
        }
        else
        {
            result = nn::nfc::ResultTagNotFound();
        }
    }

    if(result.IsFailure())
    {
        //失敗した場合は DetectEventThread へアクセス終了を通知する
        m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_AccessTerminate));
    }

    return result;
}

bool Device::IsNtag(const nn::nfc::TagId& tagId) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    Ntag* ntag = GetNtag(tagId);
    return (ntag != nullptr);
}

bool Device::IsMifare(const nn::nfc::TagId& tagId) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    Mifare* mifare = GetMifare(tagId);
    return (mifare != nullptr);
}

void Device::SetState(Device::State state) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    m_State = state;
}

nn::Result Device::InitializeNpad() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    nn::Result result;
    nn::hid::NpadIdType npadId;
    GetNpadId(&npadId);
    if(!nn::hid::system::IsNfcActivated(npadId))
    {
        result = nn::hid::system::ActivateNfc(npadId);
        if(result.IsFailure())
        {
            NN_NFC_SERVER_CORE_RETURN(result);
        }

        nn::os::MultiWaitType multiWait;
        nn::os::MultiWaitHolderType nfcActivateEventHolder;
        nn::os::MultiWaitHolderType nfcActivateCancelEventHolder;
        nn::os::InitializeMultiWait(&multiWait);
        nn::os::InitializeMultiWaitHolder(&nfcActivateEventHolder, &m_NfcActivateEvent);
        nn::os::LinkMultiWaitHolder(&multiWait, &nfcActivateEventHolder);
        nn::os::InitializeMultiWaitHolder(&nfcActivateCancelEventHolder, &m_NfcActivateCancelEvent);
        nn::os::LinkMultiWaitHolder(&multiWait, &nfcActivateCancelEventHolder);
        lock.Unlock();
        auto pHolder = nn::os::TimedWaitAny(&multiWait, nn::TimeSpan::FromMilliSeconds(1000));
        lock.Lock();
        nn::os::UnlinkAllMultiWaitHolder(&multiWait);
        nn::os::FinalizeMultiWait(&multiWait);
        nn::os::FinalizeMultiWaitHolder(&nfcActivateEventHolder);
        nn::os::FinalizeMultiWaitHolder(&nfcActivateCancelEventHolder);

        if(pHolder == nullptr){
            return nn::nfc::ResultNfcDeviceNotFound();
        }

        nn::os::ClearSystemEvent(&m_NfcActivateEvent);
        nn::os::ClearSystemEvent(&m_NfcActivateCancelEvent);

        if(!nn::hid::system::IsNfcActivated(npadId))
        {
            return nn::nfc::ResultNfcDeviceNotFound();
        }
    }

    result = nn::hid::system::GetXcdHandleForNpadWithNfc(&m_XcdDeviceHandle, npadId);
    if(result.IsFailure())
    {
        nn::hid::system::DeactivateNfc(npadId);
    }

    NN_NFC_SERVER_CORE_RETURN(result);
}

void Device::FinalizeNpad() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    nn::hid::NpadIdType npadId;
    GetNpadId(&npadId);
    if(!nn::hid::system::IsNfcActivated(npadId))
    {
        return;
    }

    nn::hid::system::DeactivateNfc(npadId);
}

nn::Result Device::InitializeDriver(const McuVersionData& mcuVersionData) NN_NOEXCEPT
{
    nn::Result result;
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(m_IsInitializedDriver)
    {
        NN_RESULT_SUCCESS;
    }

    // 既にイベント初期化済みの場合は処理順がおかしいのでエラー
    if (IsDriverEventInitialized())
    {
        // ListDevices からやり直す必要あり
        NN_RESULT_THROW(nn::nfc::ResultUnknownError());
    }

    //SIGLO-39860 McuState_Initializing から McuState_Nfc への遷移失敗に対するワークアラウンド
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(40));

    NN_RESULT_DO(SetDataFormatSync(nn::xcd::PeriodicDataFormat_MCU, m_XcdDeviceHandle));
    result = nn::xcd::SetNfcEvent(&m_Event, &m_DetectEvent, m_XcdDeviceHandle);
    if(result.IsFailure())
    {
        SetDataFormatSync(nn::xcd::PeriodicDataFormat_Basic, m_XcdDeviceHandle);
        NN_NFC_SERVER_CORE_RETURN(result);
    }

    nn::Result setMcuStateResult = SetMcuStateWithRetry(nn::xcd::McuState_Nfc, m_XcdDeviceHandle);
    {
        nn::xcd::McuVersionDataForNfc mcuVersionDataOfDevice;
        result = GetMcuVersionForNfcWithRetry(&mcuVersionDataOfDevice, m_XcdDeviceHandle);
        if(result.IsSuccess())
        {
            nn::xcd::McuVersionDataForNfc requiredMcuVersion;
            if (GetRequiredMcuVersion(&requiredMcuVersion, mcuVersionDataOfDevice.deviceType, mcuVersionData))
            {
                if (mcuVersionDataOfDevice.major < requiredMcuVersion.major
                    || (mcuVersionDataOfDevice.major == requiredMcuVersion.major
                        && mcuVersionDataOfDevice.minor < requiredMcuVersion.minor))
                {
                    result = nn::nfc::ResultNeedUpdate();
                }
                else
                {
                    result = nn::nfc::server::core::ConvertToNfcResult(nn::xcd::CheckNfcDevicePower(m_XcdDeviceHandle));
                    if(result.IsSuccess())
                    {
                        // ファームウェアは正常
                        if(setMcuStateResult.IsFailure())
                        {
                            result = setMcuStateResult;
                        }
                        else
                        {
                            result = WaitMcuState(nn::xcd::McuState_Nfc, m_XcdDeviceHandle);
                        }
                    }
                }
            }
            else
            {
                // 要求バージョンが定義されていないデバイスは使用しない
                result = nn::nfc::ResultNfcDeviceNotFound();
            }
        }
    }

    if(result.IsFailure())
    {
        //ドライバからのタグ検出、喪失イベント
        nn::os::DestroySystemEvent(&m_DetectEvent);
        //タグ検出・喪失以外のドライバからのイベント
        nn::os::DestroySystemEvent(&m_Event);

        SetMcuStateSync(nn::xcd::McuState_Standby, m_XcdDeviceHandle);
        SetDataFormatSync(nn::xcd::PeriodicDataFormat_Basic, m_XcdDeviceHandle);
        return result;
    }

    RequestLinkDriverEvent();

    m_IsInitializedDriver = true;
    NN_RESULT_SUCCESS;
}

void Device::FinalizeDriver() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    if(!m_IsInitializedDriver)
    {
        return;
    }

    m_IsFunctionError = false;
    m_IsEnableKeepSession = false;

    {
        SetMcuStateSync(nn::xcd::McuState_Standby, m_XcdDeviceHandle);
        SetDataFormatSync(nn::xcd::PeriodicDataFormat_Basic, m_XcdDeviceHandle);
    }

    RequestUnlinkDriverEvent();

    m_IsInitializedDriver = false;
}

bool Device::IsDriverEventInitialized() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_DriverEventMutex)> lock(m_DriverEventMutex);

    return m_DriverEventState != DriverEventState_NotInitialized;
}

void Device::RequestLinkDriverEvent() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_DriverEventMutex)> lock(m_DriverEventMutex);

    NN_SDK_ASSERT_EQUAL(m_DriverEventState, DriverEventState_NotInitialized);
    m_DriverEventState   = DriverEventState_Initialized;
    m_DriverEventRequest = DriverEventRequestType_Link;

    if (Manager::GetInstance().IsInLinkMonitoringThread())
    {
        ProcessDriverEventRequest();
    }
    else
    {
        // DetectEventThread にイベントの登録解除要求を出す
        m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_ManageDriverEvent));
    }
}

void Device::RequestUnlinkDriverEvent() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_DriverEventMutex)> lock(m_DriverEventMutex);

    m_DriverEventRequest = DriverEventRequestType_Unlink;

    if (Manager::GetInstance().IsInLinkMonitoringThread())
    {
        ProcessDriverEventRequest();
    }
    else
    {
        // DetectEventThread にイベントの登録解除要求を出す
        m_EventMessageQueue.Send(static_cast<nn::Bit8>(Device::EventType_ManageDriverEvent));
    }
}

void Device::ProcessDriverEventRequest() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_DriverEventMutex)> lock(m_DriverEventMutex);

    switch (m_DriverEventRequest)
    {
    case DriverEventRequestType_None:
        // 要求が出ていない
        break;

    case DriverEventRequestType_Link:
        {
            nn::os::InitializeMultiWaitHolder(&m_DetectEventHolder, &m_DetectEvent);
            nn::os::LinkMultiWaitHolder(m_pEventMultiWait, &m_DetectEventHolder);

            nn::os::InitializeMultiWaitHolder(&m_EventHolder, &m_Event);
            nn::os::LinkMultiWaitHolder(m_pEventMultiWait, &m_EventHolder);

            m_DriverEventState = DriverEventState_Linked;
        }
        break;

    case DriverEventRequestType_Unlink:
        {
            // 先にリンクを解除
            if (m_DriverEventState == DriverEventState_Linked)
            {
                nn::os::UnlinkMultiWaitHolder(&m_DetectEventHolder);
                nn::os::FinalizeMultiWaitHolder(&m_DetectEventHolder);
                nn::os::UnlinkMultiWaitHolder(&m_EventHolder);
                nn::os::FinalizeMultiWaitHolder(&m_EventHolder);

                m_DriverEventState = DriverEventState_Initialized;
            }

            // 初期化済みのイベントを破棄
            if (m_DriverEventState == DriverEventState_Initialized)
            {
                nn::os::DestroySystemEvent(&m_DetectEvent);
                nn::os::DestroySystemEvent(&m_Event);

                m_DriverEventState = DriverEventState_NotInitialized;
            }

            // 必ず破棄まで完了している
            NN_ABORT_UNLESS_EQUAL(m_DriverEventState, DriverEventState_NotInitialized);
        }
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // 要求処理済み
    m_DriverEventRequest = DriverEventRequestType_None;
}

nn::Result Device::CheckXcdDeviceHandle() NN_NOEXCEPT
{
    nn::hid::NpadIdType npadId;
    GetNpadId(&npadId);

    if(!nn::hid::system::IsNfcActivated(npadId))
    {
        return nn::nfc::ResultNfcDeviceNotFound();
    }

    nn::xcd::DeviceHandle xcdDeviceHandle;
    NN_NFC_SERVER_CORE_RESULT_DO(nn::hid::system::GetXcdHandleForNpadWithNfc(&xcdDeviceHandle, npadId));

    if(xcdDeviceHandle != m_XcdDeviceHandle)
    {
        return nn::nfc::ResultNfcDeviceNotFound();
    }

    NN_RESULT_SUCCESS;
}

void Device::SetEventThreadState(Device::EventThreadState state) NN_NOEXCEPT
{
    m_EventThreadState = state;
}

nn::Result Device::StartDiscoveryInEventThread(uint16_t activationTimeout) NN_NOEXCEPT
{
    nn::nfc::server::core::DiscoveryParameter discoverParamater = m_DiscoveryParameter;
    discoverParamater.activationTimeout = activationTimeout;
    NN_RESULT_DO(CheckXcdDeviceHandle());
    return StartDiscoveryWithRetry(discoverParamater, m_XcdDeviceHandle);
}

nn::Result Device::StopDiscoveryInEventThread() NN_NOEXCEPT
{
    NN_RESULT_DO(CheckXcdDeviceHandle());
    return StopDiscoveryWithRetry(m_XcdDeviceHandle);
}

void Device::DeactivateTag() NN_NOEXCEPT
{
    if(m_DetectedTag){
        m_DetectedTag.reset(nullptr);
    }

    StopDiscoveryInEventThread();

    SetState(Device::State_Deactive);
    SetEventThreadState(Device::EventThreadState_NotStartedDiscovery);

    m_DetectInfo.reason = nn::xcd::NfcEventReason_Deactivated;
    SignalDetectEvent();
}

void Device::ProcessIntermittentLow() NN_NOEXCEPT
{
    const int64_t lowTime = 1000;//[msec] //TEMP

    //間欠動作のLow期間を開始する
    nn::Result result;
    result = StopDiscoveryInEventThread();
    if(result.IsFailure())
    {
        DeactivateTag();
        return;
    }

    SetEventThreadState(Device::EventThreadState_IntermittentLow);
    nn::os::StartOneShotTimerEvent(&m_IntermittentLowTerminateEvent, nn::TimeSpan::FromMilliSeconds(lowTime));
}

void Device::ProcessIntermittentHigh() NN_NOEXCEPT
{
    //間欠動作における StartDiscovery を実行
    nn::Result result = StartDiscoveryInEventThread(s_ActivationTimeout);
    if(result.IsFailure())
    {
        DeactivateTag();
        return;
    }

    SetEventThreadState(Device::EventThreadState_IntermittentHigh);
}

void Device::PrintEvent(nn::os::MultiWaitHolderType* pHolder, Device::EventType eventType, const Info& info, bool unexpected) NN_NOEXCEPT
{
#if 0
    if(unexpected)
    {
        NN_SDK_LOG("[UNEXPECTED] ", m_EventThreadState);
    }

    NN_SDK_LOG("m_EventThreadState %d ", m_EventThreadState);

    if(&m_EventMessageQueueHolder == pHolder)
    {
        if(Device::EventType_NormalDiscoveryStart == eventType)
        {
            NN_SDK_LOG("received NormalDiscoveryStart\n");
        }
        else if(Device::EventType_NormalDiscoveryStop == eventType)
        {
            NN_SDK_LOG("received NormalDiscoveryStop\n");
        }
        else if(Device::EventType_AccessStart == eventType)
        {
            NN_SDK_LOG("received AccessStart\n");
        }
        else if(Device::EventType_AccessTerminate == eventType)
        {
            NN_SDK_LOG("received AccessTerminate\n");
        }
        else if(Device::EventType_Reset == eventType)
        {
            NN_SDK_LOG("received Reset\n");
        }
        else
        {
            NN_SDK_LOG("received unknown.\n");
        }
    }
    else if(&m_IntermittentLowTerminateHolder == pHolder)
    {
        NN_SDK_LOG("received m_IntermittentLowTerminateHolder\n");
    }
    else if(&m_DetectEventHolder == pHolder)
    {
        NN_SDK_LOG("received m_DetectEventHolder reason %d result_code %d\n", info.reason, info.errorInfo.resultCode);
    }
    else if(&m_EventHolder == pHolder)
    {
        NN_SDK_LOG("received m_EventHolder reason %d result_code %d\n", info.reason, info.errorInfo.resultCode);
    }
    else
    {
        NN_SDK_LOG("received unknown\n");
    }
#else
    NN_UNUSED(pHolder);
    NN_UNUSED(eventType);
    NN_UNUSED(info);
    NN_UNUSED(unexpected);
#endif
}

void Device::DispatchEventInNotInitialized(nn::os::MultiWaitHolderType* pHolder, Device::EventType eventType, const Info& info) NN_NOEXCEPT
{
    NN_UNUSED(pHolder);
    NN_UNUSED(eventType);
    NN_UNUSED(info);
}

void Device::DispatchEventInNotStartedDiscovery(nn::os::MultiWaitHolderType* pHolder, Device::EventType eventType, const Info& info) NN_NOEXCEPT
{
    if(pHolder == &m_EventMessageQueueHolder)
    {
        if(eventType == Device::EventType_NormalDiscoveryStart)
        {
            m_NormalDiscoveryResult = StartDiscoveryInEventThread(0);
            nn::os::SignalSystemEvent(&m_NormalDiscoveryEvent);
            if(m_NormalDiscoveryResult.IsSuccess())
            {
                SetEventThreadState(Device::EventThreadState_NormalStartDiscovery);
            }
        }
        else if(eventType == Device::EventType_NormalDiscoveryStop)
        {
            //タグを喪失した場合
            m_NormalDiscoveryResult = StopDiscoveryInEventThread();
            nn::os::SignalSystemEvent(&m_NormalDiscoveryEvent);
        }
        else
        {
            PrintEvent(pHolder, eventType, info, true);
        }
    }
    else
    {
        PrintEvent(pHolder, eventType, info, true);
    }
}

void Device::DispatchEventInNormalStartDiscovery(nn::os::MultiWaitHolderType* pHolder, Device::EventType eventType, const Info& info) NN_NOEXCEPT
{
    if(pHolder == &m_EventMessageQueueHolder)
    {
        if(eventType == Device::EventType_NormalDiscoveryStop)
        {
            m_NormalDiscoveryResult = StopDiscoveryInEventThread();
            SetEventThreadState(Device::EventThreadState_NotStartedDiscovery);
            nn::os::SignalSystemEvent(&m_NormalDiscoveryEvent);
        }
        else
        {
            PrintEvent(pHolder, eventType, info, true);
        }
    }
    else if(pHolder == &m_DetectEventHolder)
    {
        //このスレッド状態では、nn::xcd::NfcEventReason_Detectedしかありえない
        if(info.reason == nn::xcd::NfcEventReason_Detected)
        {
            m_DetectInfo = info;

            //タグを検出した場合
            if(m_DetectInfo.tagInfo.protocol == nn::nfc::NfcProtocol_TypeA
               && m_DetectInfo.tagInfo.type == nn::nfc::TagType_Mifare)
            {
                m_DetectedTag.reset(new Mifare(m_XcdDeviceHandle, m_DetectInfo.tagInfo.tagId));
            }
            else if(m_DetectInfo.tagInfo.protocol == nn::nfc::NfcProtocol_TypeA
               && m_DetectInfo.tagInfo.type == nn::nfc::TagType_Type2)
            {
                m_DetectedTag.reset(new Ntag(m_XcdDeviceHandle, m_DetectInfo.tagInfo.tagId));
            }
            else
            {
                m_DetectedTag.reset(new Tag(m_XcdDeviceHandle, m_DetectInfo.tagInfo.tagId, static_cast<nn::nfc::NfcProtocol>(m_DetectInfo.tagInfo.protocol), static_cast<nn::nfc::TagType>(m_DetectInfo.tagInfo.type)));
            }

            SetState(Device::State_Active);

            ProcessIntermittentLow();

            //発生したイベントは上位に通知
            SignalDetectEvent();
        }
    }
    else if(pHolder == &m_EventHolder)
    {
        if(info.reason == nn::xcd::NfcEventReason_Error
           && (info.errorInfo.resultCode == nn::xcd::NfcResultCode_SequenceError
               || info.errorInfo.resultCode == nn::xcd::NfcResultCode_Timeout
               || info.errorInfo.resultCode == nn::xcd::NfcResultCode_CommandTimeout))
        {
            StopDiscoveryInEventThread();
            StartDiscoveryInEventThread(0);
        }
        else if(info.reason == nn::xcd::NfcEventReason_Error
                && info.errorInfo.resultCode == nn::xcd::NfcResultCode_ResetRequired)
        {
            nn::Result result;
            result = InitializeNpad();
            if(result.IsFailure())
            {
                return;
            }
            result = InitializeDriver(m_McuVersionData);
            if(result.IsFailure())
            {
                FinalizeNpad();
                return;
            }
            result = StartDiscoveryInEventThread(0);
            if(result.IsFailure())
            {
                FinalizeDriver();
                FinalizeNpad();
                return;
            }
        }
    }
    else
    {
        PrintEvent(pHolder, eventType, info, true);
    }
}

void Device::DispatchEventInIntermittentLow(nn::os::MultiWaitHolderType* pHolder, Device::EventType eventType, const Info& info) NN_NOEXCEPT
{
    if(pHolder == &m_EventMessageQueueHolder)
    {
        if(eventType == Device::EventType_NormalDiscoveryStop)
        {
            CancelTimer(&m_IntermittentLowTerminateEvent);

            SetEventThreadState(Device::EventThreadState_NotStartedDiscovery);
            nn::os::SignalSystemEvent(&m_NormalDiscoveryEvent);
        }
        else if(eventType == Device::EventType_AccessStart)
        {
            CancelTimer(&m_IntermittentLowTerminateEvent);

            nn::Result result = StartDiscoveryInEventThread(s_ActivationTimeout);
            if(result.IsFailure())
            {
                DeactivateTag();
                m_AccessEventMessageQueue.Send(static_cast<nn::Bit8>(Device::AccessEventType_Deactivated));
                return;
            }

            SetEventThreadState(Device::EventThreadState_ReadyAccessing);
        }
        else
        {
            PrintEvent(pHolder, eventType, info, true);
        }
    }
    else if(pHolder == &m_IntermittentLowTerminateHolder)
    {
        ProcessIntermittentHigh();
    }
    else if(pHolder == &m_EventHolder)
    {
        if(IsDeactivated(info))
        {
            DeactivateTag();
        }
    }
    else
    {
        PrintEvent(pHolder, eventType, info, true);
    }
}

void Device::DispatchEventInIntermittentHigh(nn::os::MultiWaitHolderType* pHolder, Device::EventType eventType, const Info& info) NN_NOEXCEPT
{
    if(pHolder == &m_EventMessageQueueHolder)
    {
        if(eventType == Device::EventType_NormalDiscoveryStop)
        {
            m_NormalDiscoveryResult = CheckXcdDeviceHandle();
            if(m_NormalDiscoveryResult.IsSuccess())
            {
                m_NormalDiscoveryResult = StopDiscoveryWithRetry(m_XcdDeviceHandle);
            }
            SetEventThreadState(Device::EventThreadState_NotStartedDiscovery);
            nn::os::SignalSystemEvent(&m_NormalDiscoveryEvent);
        }
        else if(eventType == Device::EventType_AccessStart)
        {
            SetEventThreadState(Device::EventThreadState_ReadyAccessing);
        }
        else
        {
            PrintEvent(pHolder, eventType, info, true);
        }
    }
    else if(pHolder == &m_DetectEventHolder)
    {
        //このスレッド状態では、nn::xcd::NfcEventReason_Detectedしかありえない
        if(info.reason == nn::xcd::NfcEventReason_Detected)
        {
            if(nn::nfc::server::util::IsEqualTagId(m_DetectInfo.tagInfo.tagId, info.tagInfo.tagId))
            {
                ProcessIntermittentLow();
            }
            else
            {
                //別のタグに代わっている場合は、前のタグの喪失処理をする
                DeactivateTag();
            }
        }
    }
    else if(pHolder == &m_EventHolder)
    {
        if(IsDeactivated(info))
        {
            DeactivateTag();
        }
    }
    else
    {
        PrintEvent(pHolder, eventType, info, true);
    }
}

void Device::DispatchEventInReadyAccessing(nn::os::MultiWaitHolderType* pHolder, Device::EventType eventType, const Info& info) NN_NOEXCEPT
{
    if(pHolder == &m_EventMessageQueueHolder)
    {
        if(eventType == Device::EventType_NormalDiscoveryStop)
        {
            m_NormalDiscoveryResult = CheckXcdDeviceHandle();
            if(m_NormalDiscoveryResult.IsSuccess())
            {
                m_NormalDiscoveryResult = StopDiscoveryWithRetry(m_XcdDeviceHandle);
            }
            SetEventThreadState(Device::EventThreadState_NotStartedDiscovery);
            nn::os::SignalSystemEvent(&m_NormalDiscoveryEvent);
        }
        else if(eventType == Device::EventType_AccessTerminate)
        {
            ProcessIntermittentLow();
        }
        else
        {
            PrintEvent(pHolder, eventType, info, true);
        }
    }
    else if(pHolder == &m_DetectEventHolder)
    {
        //このスレッド状態では、nn::xcd::NfcEventReason_Detectedしかありえない
        if(info.reason == nn::xcd::NfcEventReason_Detected)
        {
            if(nn::nfc::server::util::IsEqualTagId(m_DetectInfo.tagInfo.tagId, info.tagInfo.tagId))
            {
                //タグアクセス開始を要求元にタグが見つかったことを通知
                m_AccessEventMessageQueue.Send(static_cast<nn::Bit8>(Device::AccessEventType_Deteceted));

                //タグアクセス中へ遷移
                SetEventThreadState(Device::EventThreadState_Accessing);
            }
            else
            {
                //別のタグに代わっている場合は、前のタグの喪失処理をする
                DeactivateTag();

                m_AccessEventMessageQueue.Send(static_cast<nn::Bit8>(Device::AccessEventType_Deactivated));
            }
        }
    }
    else if(pHolder == &m_EventHolder)
    {
        if(IsDeactivated(info))
        {
            DeactivateTag();

            m_AccessEventMessageQueue.Send(static_cast<nn::Bit8>(Device::AccessEventType_Deactivated));
        }
    }
    else
    {
        PrintEvent(pHolder, eventType, info, true);
    }
}

void Device::DispatchEventInAccessing(nn::os::MultiWaitHolderType* pHolder, Device::EventType eventType, const Info& info) NN_NOEXCEPT
{
    if(pHolder == &m_EventMessageQueueHolder)
    {
        if(eventType == Device::EventType_NormalDiscoveryStop)
        {
            m_NormalDiscoveryResult = CheckXcdDeviceHandle();
            if(m_NormalDiscoveryResult.IsSuccess())
            {
                m_NormalDiscoveryResult = StopDiscoveryWithRetry(m_XcdDeviceHandle);
            }
            SetEventThreadState(Device::EventThreadState_NotStartedDiscovery);
            nn::os::SignalSystemEvent(&m_NormalDiscoveryEvent);
        }
        else if(eventType == Device::EventType_AccessTerminate)
        {
            ProcessIntermittentLow();
        }
        else
        {
            PrintEvent(pHolder, eventType, info, true);
        }
    }
    else if(pHolder == &m_EventHolder)
    {
        if(IsDeactivated(info))
        {
            // タグアクセス中に発生した FunctionError はアクセスした API の戻りにエラーを返す
            // タグ喪失通知を送る前に行う必要あり(まずアクセスが完了しないと喪失通知が届かないため)
            if(info.reason == nn::xcd::NfcEventReason_Error
               && info.errorInfo.resultCode == nn::xcd::NfcResultCode_FunctionError)
            {
                m_Info = info;
                SignalEvent();
            }

            DeactivateTag();
        }
        else
        {
            m_Info = info;

            if(!m_IsEnableKeepSession)
            {
                ProcessIntermittentLow();
            }

            SignalEvent();
        }
    }
    else
    {
        PrintEvent(pHolder, eventType, info, true);
    }
}

bool Device::PreProcessEvent(Device::EventType* pOutEventType, Info* pOutInfo, nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    if(&m_EventMessageQueueHolder == pHolder)
    {
        m_EventMessageQueue.Receive(reinterpret_cast<nn::Bit8*>(pOutEventType));

        // Manager へのイベント出し入れ要求は優先して処理する
        if (*pOutEventType == Device::EventType_ManageDriverEvent)
        {
            ProcessDriverEventRequest();
            return false;
        }
    }
    else if(pHolder == &m_IntermittentLowTerminateHolder)
    {
        nn::os::ClearTimerEvent(&m_IntermittentLowTerminateEvent);
    }
    else if(pHolder == &m_DetectEventHolder)
    {
        if(m_IsInitializedDriver)
        {
            nn::os::ClearSystemEvent(&m_DetectEvent);
        }
        else
        {
            return false;
        }
    }
    else if(pHolder == &m_EventHolder)
    {
        if(m_IsInitializedDriver)
        {
            nn::os::ClearSystemEvent(&m_Event);
        }
        else
        {
            return false;
        }
    }
    else
    {
        return false;
    }

    if(pHolder == &m_DetectEventHolder || pHolder == &m_EventHolder)
    {
        std::unique_ptr<nn::xcd::NfcInfo> nfcInfo(new nn::xcd::NfcInfo());
        nn::Result result = nn::xcd::GetNfcInfo(nfcInfo.get(), m_XcdDeviceHandle);
        if(result.IsFailure())
        {
            return false;
        }

        if(nfcInfo->reason == nn::xcd::NfcEventReason_Error)
        {
            if(nfcInfo->errorInfo.resultCode == nn::xcd::NfcResultCode_FunctionError)
            {
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
                NN_ABORT("[nfc] nn::xcd::NfcResultCode_FunctionError\n");
#endif
                m_IsFunctionError = true;
            }
            else if(nfcInfo->errorInfo.resultCode == nn::xcd::NfcResultCode_ResetRequired)
            {
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
                NN_SDK_LOG("[nfc] nn::xcd::NfcResultCode_ResetRequired\n");
#endif
                //Reset に失敗する場合もある
                //ここでのエラーは後に API を使うことでユーザに伝わる
                FinalizeDriver();
                FinalizeNpad();
            }
        }

        ConvertToNfcInfo(pOutInfo, *nfcInfo);
    }

    return true;
}

void Device::DispatchEventInEventThread(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    std::unique_ptr<Info> info(new Info());
    Device::EventType eventType;
    if(!PreProcessEvent(&eventType, info.get(), pHolder))
    {
        return;
    }
    PrintEvent(pHolder, eventType, *info, false);

    switch(m_EventThreadState)
    {
    case Device::EventThreadState_NotInitialized:
        {
            DispatchEventInNotInitialized(pHolder, eventType, *info);
        }
        break;
    case Device::EventThreadState_NotStartedDiscovery:
        {
            DispatchEventInNotStartedDiscovery(pHolder, eventType, *info);
        }
        break;
    case Device::EventThreadState_NormalStartDiscovery:
        {
            DispatchEventInNormalStartDiscovery(pHolder, eventType, *info);
        }
        break;
    case Device::EventThreadState_IntermittentLow:
        {
            DispatchEventInIntermittentLow(pHolder, eventType, *info);
        }
        break;
    case Device::EventThreadState_IntermittentHigh:
        {
            DispatchEventInIntermittentHigh(pHolder, eventType, *info);
        }
        break;
    case Device::EventThreadState_ReadyAccessing:
        {
            DispatchEventInReadyAccessing(pHolder, eventType, *info);
        }
        break;
    case Device::EventThreadState_Accessing:
        {
            DispatchEventInAccessing(pHolder, eventType, *info);
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void Device::LoopEventThreadFunction(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    if(m_IsRunningThread == false)
    {
        return;
    }

    if(pHolder != &m_EventMessageQueueHolder
       && pHolder != &m_IntermittentLowTerminateHolder
       && pHolder != &m_DetectEventHolder
       && pHolder != &m_EventHolder)
    {
        return;
    }

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    DispatchEventInEventThread(pHolder);
}

}}}}  // namespace nn::nfc::server::core
