﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/news_Data.h>
#include <nn/news/news_Result.h>
#include <nn/news/detail/news_ShimLibraryGlobal.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace news {

Data::Data() NN_NOEXCEPT :
    m_Handle(nullptr)
{

}

Data::~Data() NN_NOEXCEPT
{
    Close();
}

nn::Result Data::Open(const char* name) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(name);

    NN_RESULT_THROW_UNLESS(!m_Handle, ResultAlreadyOpened());

    size_t length = static_cast<size_t>(nn::util::Strnlen(name, DataNameLengthMax + 1));

    NN_RESULT_THROW_UNLESS(length > 0 && length <= DataNameLengthMax, ResultInvalidArgument());

    detail::ipc::INewsDataService* handle = nullptr;

    NN_RESULT_DO(detail::ShimLibraryGlobal::GetInstance().CreateNewsDataService(&handle));

    NN_UTIL_SCOPE_EXIT
    {
        if (handle)
        {
            nn::sf::ReleaseSharedObject(handle);
        }
    };

    NN_RESULT_DO(handle->Open(nn::sf::InArray<char>(name, length + 1)));

    m_Handle = handle;
    handle = nullptr;

    NN_RESULT_SUCCESS;
}

nn::Result Data::Open(const NewsRecord& record) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_Handle, ResultAlreadyOpened());

    detail::ipc::INewsDataService* handle = nullptr;

    NN_RESULT_DO(detail::ShimLibraryGlobal::GetInstance().CreateNewsDataService(&handle));

    NN_UTIL_SCOPE_EXIT
    {
        if (handle)
        {
            nn::sf::ReleaseSharedObject(handle);
        }
    };

    NN_RESULT_DO(handle->OpenWithNewsRecord(record));

    m_Handle = handle;
    handle = nullptr;

    NN_RESULT_SUCCESS;
}

void Data::Close() NN_NOEXCEPT
{
    if (m_Handle)
    {
        nn::sf::ReleaseSharedObject(m_Handle);
        m_Handle = nullptr;
    }
}

nn::Result Data::Read(size_t* outSize, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSize);
    NN_SDK_REQUIRES(offset >= 0);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(size > 0);

    NN_RESULT_THROW_UNLESS(m_Handle, ResultNotOpened());

    uint64_t actualSize = 0;

    NN_RESULT_DO(m_Handle->Read(&actualSize, offset, nn::sf::OutBuffer(static_cast<char*>(buffer), size)));

    *outSize = static_cast<size_t>(actualSize);

    NN_RESULT_SUCCESS;
}

nn::Result Data::GetSize(int64_t* outSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSize);

    NN_RESULT_THROW_UNLESS(m_Handle, ResultNotOpened());

    NN_RESULT_DO(m_Handle->GetSize(outSize));

    NN_RESULT_SUCCESS;
}

}}
