﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/service/util/news_Sqlite.h>
#include <nn/mem.h>
#include <nn/util/util_Utf8StringUtil.h>

namespace nn { namespace news { namespace detail { namespace service { namespace util {

namespace
{
    // N_IF(expr, expr_if_true, expr_if_false)
     void N_IF(sqlite3_context* pContext, int argc, sqlite3_value** argv) NN_NOEXCEPT
    {
        if (argc != 3)
        {
            sqlite3_result_null(pContext);
            return;
        }

        sqlite3_result_value(pContext, sqlite3_value_int(argv[0]) ? argv[1] : argv[2]);
    }

    // N_SWITCH(expr, case1, expr_if_equal_case1, [..., caseN, expr_if_equal_caseN,] default)
    void N_SWITCH(sqlite3_context* pContext, int argc, sqlite3_value** argv) NN_NOEXCEPT
    {
        if (argc < 4 || (argc % 2) != 0)
        {
            sqlite3_result_null(pContext);
            return;
        }

        int baseValueType = sqlite3_value_type(argv[0]);

        for (int i = 1; i < argc - 1; i += 2)
        {
            if (baseValueType == sqlite3_value_type(argv[i]))
            {
                switch (baseValueType)
                {
                case SQLITE_INTEGER:
                    {
                        if (sqlite3_value_int64(argv[0]) == sqlite3_value_int64(argv[i]))
                        {
                            sqlite3_result_value(pContext, argv[i + 1]);
                            return;
                        }
                    }
                    break;
                case SQLITE_TEXT:
                    {
                        const char* value1 = reinterpret_cast<const char*>(sqlite3_value_text(argv[0]));
                        const char* value2 = reinterpret_cast<const char*>(sqlite3_value_text(argv[i]));

                        if (nn::util::Strncmp(value1, value2, 4096) == 0)
                        {
                            sqlite3_result_value(pContext, argv[i + 1]);
                            return;
                        }
                    }
                    break;
                default:
                    break;
                }
            }
        }

        // デフォルト値を返す。
        sqlite3_result_value(pContext, argv[argc - 1]);
    }

    // N_RAND(begin, end)
    void N_RAND(sqlite3_context* pContext, int argc, sqlite3_value** argv) NN_NOEXCEPT
    {
        if (argc != 2)
        {
            sqlite3_result_null(pContext);
            return;
        }
        if (sqlite3_value_type(argv[0]) != SQLITE_INTEGER || sqlite3_value_type(argv[1]) != SQLITE_INTEGER)
        {
            sqlite3_result_null(pContext);
            return;
        }

        int32_t begin = static_cast<int32_t>(sqlite3_value_int(argv[0]));
        int32_t end = static_cast<int32_t>(sqlite3_value_int(argv[1]));

        if (begin >= end)
        {
            sqlite3_result_null(pContext);
            return;
        }

        int32_t value = static_cast<int32_t>(detail::service::util::GetRandom(begin, end));

        sqlite3_result_int(pContext, value);
    }

    // N_PMATCH(value, partial)
    void N_PMATCH(sqlite3_context* pContext, int argc, sqlite3_value** argv) NN_NOEXCEPT
    {
        if (argc != 2)
        {
            sqlite3_result_null(pContext);
            return;
        }
        if (sqlite3_value_type(argv[0]) != SQLITE_TEXT || sqlite3_value_type(argv[1]) != SQLITE_TEXT)
        {
            sqlite3_result_null(pContext);
            return;
        }

        const char* pValue = reinterpret_cast<const char*>(sqlite3_value_text(argv[0]));
        const char* pPartial = reinterpret_cast<const char*>(sqlite3_value_text(argv[1]));

        if (std::strstr(pValue, pPartial))
        {
            sqlite3_result_int(pContext, 1);
        }
        else
        {
            sqlite3_result_int(pContext, 0);
        }
    }

    int GetPageSizeCallback(void* pParam, int argc, char** argv, char** colNames) NN_NOEXCEPT
    {
        int* pPageSize = reinterpret_cast<int*>(pParam);

        for (int i = 0; i < argc; i++)
        {
            if (nn::util::Strncmp(colNames[i], "page_size", sizeof ("page_size")) == 0)
            {
                *pPageSize = static_cast<int>(std::strtol(argv[i], nullptr, 10));
                break;
            }
        }

        return SQLITE_OK;
    }
}

namespace
{
    nn::mem::StandardAllocator g_Allocator;
    nn::Bit64 g_Heap[SqliteHeapSize / sizeof (nn::Bit64)];
}

namespace
{
    int SqliteInitializeMemory(void* pParam) NN_NOEXCEPT
    {
        NN_UNUSED(pParam);

        g_Allocator.Initialize(g_Heap, sizeof (g_Heap));

        return SQLITE_OK;
    }

    void SqliteShutdownMemory(void* pParam) NN_NOEXCEPT
    {
        NN_UNUSED(pParam);

        g_Allocator.Finalize();
    }

    void* SqliteMalloc(int size) NN_NOEXCEPT
    {
        return g_Allocator.Allocate(static_cast<size_t>(size));
    }

    void SqliteFree(void* p) NN_NOEXCEPT
    {
        if (p)
        {
            g_Allocator.Free(p);
        }
    }

    void* SqliteRealloc(void* p, int size) NN_NOEXCEPT
    {
        return g_Allocator.Reallocate(p, static_cast<size_t>(size));
    }

    int SqliteMemSize(void* p) NN_NOEXCEPT
    {
        return static_cast<int>(g_Allocator.GetSizeOf(p));
    }

    int SqliteRoundup(int n) NN_NOEXCEPT
    {
        return ((n + 7) & ~7);
    }
}

nn::Result Sqlite::SetupAllocator() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(sqlite3_mem_methods, s_MemoryMethods, =
    {
        SqliteMalloc,           // xMalloc
        SqliteFree,             // xFree
        SqliteRealloc,          // xRealloc
        SqliteMemSize,          // xSize
        SqliteRoundup,          // xRoundup
        SqliteInitializeMemory, // xInit
        SqliteShutdownMemory,   // xShutdown
        nullptr                 // pAppData
    });

    SQLITE_DO(sqlite3_config(SQLITE_CONFIG_MALLOC, &s_MemoryMethods));

    NN_RESULT_SUCCESS;
}

nn::Result Sqlite::SetupGlobalHeapAllocator() NN_NOEXCEPT
{
    // xMalloc に nullptr を指定することで、デフォルトのメモリアロケータが利用される。
    NN_FUNCTION_LOCAL_STATIC(sqlite3_mem_methods, s_MemoryMethods, =
    {
        nullptr, // xMalloc
        nullptr, // xFree
        nullptr, // xRealloc
        nullptr, // xSize
        nullptr, // xRoundup
        nullptr, // xInit
        nullptr, // xShutdown
        nullptr  // pAppData
    });

    SQLITE_DO(sqlite3_config(SQLITE_CONFIG_MALLOC, &s_MemoryMethods));

    NN_RESULT_SUCCESS;
}

nn::Result Sqlite::SetPerformanceImprovementParameters(sqlite3* pHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHandle);

    // ジャーナル処理を無効化することで、パフォーマンスを向上させる。
    SQLITE_DO(sqlite3_exec(pHandle, "PRAGMA journal_mode = OFF", nullptr, nullptr, nullptr));
    // 追加削除の繰り返しによる断片化を防ぎ、パフォーマンスの劣化を防ぐ。
    SQLITE_DO(sqlite3_exec(pHandle, "PRAGMA auto_vacuum = FULL", nullptr, nullptr, nullptr));

    NN_RESULT_SUCCESS;
}

nn::Result Sqlite::RegisterCustomFunctions(sqlite3* pHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHandle);

    SQLITE_DO(sqlite3_create_function(pHandle, "N_IF", 3, SQLITE_UTF8, nullptr, N_IF, nullptr, nullptr));
    SQLITE_DO(sqlite3_create_function(pHandle, "N_SWITCH", -1, SQLITE_UTF8, nullptr, N_SWITCH, nullptr, nullptr));
    SQLITE_DO(sqlite3_create_function(pHandle, "N_RAND", 2, SQLITE_UTF8, nullptr, N_RAND, nullptr, nullptr));
    SQLITE_DO(sqlite3_create_function(pHandle, "N_PMATCH", 2, SQLITE_UTF8, nullptr, N_PMATCH, nullptr, nullptr));

    NN_RESULT_SUCCESS;
}

nn::Result Sqlite::GetPageSize(int* outPageSize, sqlite3* pHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outPageSize);
    NN_SDK_REQUIRES_NOT_NULL(pHandle);

    SQLITE_DO(sqlite3_exec(pHandle, "PRAGMA page_size;", GetPageSizeCallback, outPageSize, nullptr));

    NN_RESULT_SUCCESS;
}

nn::Result Sqlite::AddColumn(sqlite3* pHandle,
    const char* tableName, const char* columnName, int64_t defaultValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHandle);
    NN_SDK_REQUIRES_NOT_NULL(tableName);
    NN_SDK_REQUIRES_NOT_NULL(columnName);

    char escapedTableName[EscapedStringSize] = {};
    NN_RESULT_THROW_UNLESS(EscapeString(escapedTableName, sizeof (escapedTableName), tableName), ResultInvalidArgument());

    char escapedColumnName[EscapedStringSize] = {};
    NN_RESULT_THROW_UNLESS(EscapeString(escapedColumnName, sizeof (escapedColumnName), columnName), ResultInvalidArgument());

    sqlite3_stmt* pStatement = nullptr;

    NN_UTIL_SCOPE_EXIT
    {
        if (pStatement)
        {
            sqlite3_finalize(pStatement);
        }
    };

    char query[256] = {};

    int length = nn::util::SNPrintf(query, sizeof (query),
        "ALTER TABLE %s ADD %s INTEGER DEFAULT %lld", escapedTableName, escapedColumnName, defaultValue);

    NN_SDK_ASSERT_LESS(static_cast<size_t>(length), sizeof (query));

    NN_DETAIL_NEWS_INFO("[news] Query: %s\n", query);

    SQLITE_DO(sqlite3_prepare_v2(pHandle, query, length, &pStatement, nullptr));
    SQLITE_DO(sqlite3_step(pStatement));

    NN_RESULT_SUCCESS;
}

nn::Result Sqlite::AddColumn(sqlite3* pHandle,
    const char* tableName, const char* columnName, const char* defaultValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHandle);
    NN_SDK_REQUIRES_NOT_NULL(tableName);
    NN_SDK_REQUIRES_NOT_NULL(columnName);
    NN_SDK_REQUIRES_NOT_NULL(defaultValue);

    char escapedTableName[EscapedStringSize] = {};
    NN_RESULT_THROW_UNLESS(EscapeString(escapedTableName, sizeof (escapedTableName), tableName), ResultInvalidArgument());

    char escapedColumnName[EscapedStringSize] = {};
    NN_RESULT_THROW_UNLESS(EscapeString(escapedColumnName, sizeof (escapedColumnName), columnName), ResultInvalidArgument());

    char escapedDefaultValue[EscapedStringSize] = {};
    NN_RESULT_THROW_UNLESS(EscapeString(escapedDefaultValue, sizeof (escapedDefaultValue), defaultValue), ResultInvalidArgument());

    sqlite3_stmt* pStatement = nullptr;

    NN_UTIL_SCOPE_EXIT
    {
        if (pStatement)
        {
            sqlite3_finalize(pStatement);
        }
    };

    char query[256] = {};

    int length = nn::util::SNPrintf(query, sizeof (query),
        "ALTER TABLE %s ADD %s TEXT DEFAULT '%s'", escapedTableName, escapedColumnName, escapedDefaultValue);

    NN_SDK_ASSERT_LESS(static_cast<size_t>(length), sizeof (query));

    NN_DETAIL_NEWS_INFO("[news] Query: %s\n", query);

    SQLITE_DO(sqlite3_prepare_v2(pHandle, query, length, &pStatement, nullptr));
    SQLITE_DO(sqlite3_step(pStatement));

    NN_RESULT_SUCCESS;
}

nn::Result Sqlite::HandleError(int result) NN_NOEXCEPT
{
    if (result == SQLITE_OK)
    {
        NN_RESULT_SUCCESS;
    }

    switch (result)
    {
    case SQLITE_ERROR:
        return ResultSqliteErrorError();
    case SQLITE_INTERNAL:
        return ResultSqliteErrorInternal();
    case SQLITE_PERM:
        return ResultSqliteErrorPerm();
    case SQLITE_ABORT:
        return ResultSqliteErrorAbort();
    case SQLITE_BUSY:
        return ResultSqliteErrorBusy();
    case SQLITE_LOCKED:
        return ResultSqliteErrorLocked();
    case SQLITE_NOMEM:
        return ResultSqliteErrorNoMem();
    case SQLITE_READONLY:
        return ResultSqliteErrorReadOnly();
    case SQLITE_INTERRUPT:
        return ResultSqliteErrorInterrupt();
    case SQLITE_IOERR:
        return ResultSqliteErrorIoErr();
    case SQLITE_CORRUPT:
        return ResultSqliteErrorCorrupt();
    case SQLITE_NOTFOUND:
        return ResultSqliteErrorNotFound();
    case SQLITE_FULL:
        return ResultSqliteErrorFull();
    case SQLITE_CANTOPEN:
        return ResultSqliteErrorCantOpen();
    case SQLITE_PROTOCOL:
        return ResultSqliteErrorProtocol();
    case SQLITE_EMPTY:
        return ResultSqliteErrorEmpty();
    case SQLITE_SCHEMA:
        return ResultSqliteErrorSchema();
    case SQLITE_TOOBIG:
        return ResultSqliteErrorTooBig();
    case SQLITE_CONSTRAINT:
        return ResultSqliteErrorConstraint();
    case SQLITE_MISMATCH:
        return ResultSqliteErrorMismatch();
    case SQLITE_MISUSE:
        return ResultSqliteErrorMisuse();
    case SQLITE_NOLFS:
        return ResultSqliteErrorNoLfs();
    case SQLITE_AUTH:
        return ResultSqliteErrorAuth();
    case SQLITE_FORMAT:
        return ResultSqliteErrorFormat();
    case SQLITE_RANGE:
        return ResultSqliteErrorRange();
    case SQLITE_NOTADB:
        return ResultSqliteErrorNotADb();
    case SQLITE_NOTICE:
        return ResultSqliteErrorNotice();
    case SQLITE_WARNING:
        return ResultSqliteErrorWarning();
    default:
        return ResultUnexpectedSqliteError();
    }
}

bool Sqlite::EscapeString(char* output, size_t outputSize, const char* input, bool isSingleQuote) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(output);
    NN_SDK_REQUIRES_GREATER(outputSize, 0u);
    NN_SDK_REQUIRES_NOT_NULL(input);

    int inputLength = nn::util::Strnlen(input, static_cast<int>(outputSize));

    if (inputLength >= static_cast<int>(outputSize))
    {
        return false;
    }
    if (inputLength > 0 && !nn::util::VerifyUtf8String(input, static_cast<size_t>(inputLength)))
    {
        return false;
    }

    const char quote = isSingleQuote ? '\'' : '"';

    const char* p = input;

    int escapedLength = inputLength;

    while (*p != '\0')
    {
        if (*p++ == quote)
        {
            escapedLength++;
        }
    }
    if (escapedLength + 1 > static_cast<int>(outputSize))
    {
        return false;
    }

    while (*input != '\0')
    {
        if (*input == quote)
        {
            *output++ = quote;
        }

        *output++ = *input++;
    }

    *output = '\0';

    return true;
}

}}}}}
