﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/service/news_ServiceCreator.h>
#include <nn/news/detail/service/news_NewsService.h>
#include <nn/news/detail/service/news_NewlyArrivedEventHolder.h>
#include <nn/news/detail/service/news_NewsDataService.h>
#include <nn/news/detail/service/news_NewsDatabaseService.h>
#include <nn/news/detail/service/news_OverwriteEventHolder.h>
#include <nn/news/detail/service/news_Common.h>
#include <nn/news/detail/service/news_ServiceMemoryManager.h>
#include <nn/sf/sf_ObjectFactory.h>

#define CHECK_CAPABILITY(flag) \
    NN_RESULT_THROW_UNLESS(m_Capability.IsPermitted(flag), ResultNotPermitted())

namespace nn { namespace news { namespace detail { namespace service {

ServiceCreator::ServiceCreator(const char* serviceName, const Capability& capability) NN_NOEXCEPT :
    m_ServiceName(serviceName),
    m_Capability(capability)
{
    NN_UNUSED(m_ServiceName);
}

ServiceCreator::~ServiceCreator() NN_NOEXCEPT
{
}

nn::Result ServiceCreator::CreateNewsService(nn::sf::Out<nn::sf::SharedPointer<nn::news::detail::ipc::INewsService>> outService) NN_NOEXCEPT
{
    auto& allocator = detail::service::ServiceMemoryManager::GetInstance().GetNewsServiceAllocator();

    auto p = nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::
        CreateSharedEmplaced<detail::ipc::INewsService, NewsService>(&allocator, m_Capability);

    // NN_DETAIL_NEWS_INFO("[news] Create news service. (%s)\n", m_ServiceName ? m_ServiceName : "");

    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionResource());

    outService.Set(std::move(p));

    NN_RESULT_SUCCESS;
}

nn::Result ServiceCreator::CreateNewlyArrivedEventHolder(nn::sf::Out<nn::sf::SharedPointer<nn::news::detail::ipc::INewlyArrivedEventHolder>> outHolder) NN_NOEXCEPT
{
    auto& allocator = detail::service::ServiceMemoryManager::GetInstance().GetNewlyArrivedEventHolderAllocator();

    auto p = nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::
        CreateSharedEmplaced<detail::ipc::INewlyArrivedEventHolder, NewlyArrivedEventHolder>(&allocator);

    // NN_DETAIL_NEWS_INFO("[news] Create newly arrived event holder. (%s)\n", m_ServiceName ? m_ServiceName : "");

    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionResource());

    outHolder.Set(std::move(p));

    NN_RESULT_SUCCESS;
}

nn::Result ServiceCreator::CreateOverwriteEventHolder(nn::sf::Out<nn::sf::SharedPointer<nn::news::detail::ipc::IOverwriteEventHolder>> outHolder) NN_NOEXCEPT
{
    auto& allocator = detail::service::ServiceMemoryManager::GetInstance().GetOverwriteEventHolderAllocator();

    auto p = nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::
        CreateSharedEmplaced<detail::ipc::IOverwriteEventHolder, OverwriteEventHolder>(&allocator);

    // NN_DETAIL_NEWS_INFO("[news] Create overwrite event holder. (%s)\n", m_ServiceName ? m_ServiceName : "");

    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionResource());

    outHolder.Set(std::move(p));

    NN_RESULT_SUCCESS;
}

nn::Result ServiceCreator::CreateNewsDataService(nn::sf::Out<nn::sf::SharedPointer<nn::news::detail::ipc::INewsDataService>> outService) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);

    auto& allocator = detail::service::ServiceMemoryManager::GetInstance().GetNewsDataServiceAllocator();

    auto p = nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::
        CreateSharedEmplaced<detail::ipc::INewsDataService, NewsDataService>(&allocator, m_Capability);

    // NN_DETAIL_NEWS_INFO("[news] Create news data service. (%s)\n", m_ServiceName ? m_ServiceName : "");

    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionResource());

    outService.Set(std::move(p));

    NN_RESULT_SUCCESS;
}

nn::Result ServiceCreator::CreateNewsDatabaseService(nn::sf::Out<nn::sf::SharedPointer<nn::news::detail::ipc::INewsDatabaseService>> outService) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);

    auto& allocator = detail::service::ServiceMemoryManager::GetInstance().GetNewsDatabaseServiceAllocator();

    auto p = nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::
        CreateSharedEmplaced<detail::ipc::INewsDatabaseService, NewsDatabaseService>(&allocator, m_Capability);

    // NN_DETAIL_NEWS_INFO("[news] Create news database service. (%s)\n", m_ServiceName ? m_ServiceName : "");

    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionResource());

    outService.Set(std::move(p));

    NN_RESULT_SUCCESS;
}

}}}}
