﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/service/news_NewsDatabaseService.h>
#include <nn/news/detail/service/news_Common.h>
#include <nn/news/detail/service/core/news_NewsDatabase.h>
#include <nn/news/detail/service/core/news_ForegroundStorageLockManager.h>

#define CHECK_CAPABILITY(flag) \
    NN_RESULT_THROW_UNLESS(m_Capability.IsPermitted(flag), ResultNotPermitted())

#define CHECK_IPC_STRING(value, maxSize) \
    do                                                              \
    {                                                               \
        size_t size = value.GetLength();                            \
        if (size == 0 || size > maxSize || value[size - 1] != '\0') \
        {                                                           \
            NN_RESULT_THROW(ResultInvalidArgument());               \
        }                                                           \
    }                                                               \
    while (NN_STATIC_CONDITION(0))

namespace nn { namespace news { namespace detail { namespace service {

NewsDatabaseService::NewsDatabaseService(const Capability& capability) NN_NOEXCEPT :
    m_Capability(capability)
{
    detail::service::core::ForegroundStorageLockManager::GetInstance().AcquireDataAccessorLock();
}

NewsDatabaseService::~NewsDatabaseService() NN_NOEXCEPT
{
    detail::service::core::ForegroundStorageLockManager::GetInstance().ReleaseDataAccessorLock();
}

nn::Result NewsDatabaseService::GetList(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::news::NewsRecord>& outRecords,
    const nn::sf::InArray<char>& wherePhrase, const nn::sf::InArray<char>& orderByPhrase, std::int32_t offset) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);

    CHECK_IPC_STRING(wherePhrase, WherePhraseLengthMax + 1);
    CHECK_IPC_STRING(orderByPhrase, OrderByPhraseLengthMax + 1);

    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

    int actualCount = 0;

    NN_RESULT_DO(detail::service::core::NewsDatabase::GetInstance().GetList(&actualCount, outRecords.GetData(),
        wherePhrase.GetData(), orderByPhrase.GetData(), offset, static_cast<int>(outRecords.GetLength())));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result NewsDatabaseService::GetListV1(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::news::NewsRecordV1>& outRecords,
    const nn::sf::InArray<char>& wherePhrase, const nn::sf::InArray<char>& orderByPhrase, std::int32_t offset) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);

    CHECK_IPC_STRING(wherePhrase, WherePhraseLengthMax + 1);
    CHECK_IPC_STRING(orderByPhrase, OrderByPhraseLengthMax + 1);

    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

    int actualCount = 0;

    NN_RESULT_DO(detail::service::core::NewsDatabase::GetInstance().GetList(&actualCount, outRecords.GetData(),
        wherePhrase.GetData(), orderByPhrase.GetData(), offset, static_cast<int>(outRecords.GetLength())));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result NewsDatabaseService::Count(nn::sf::Out<std::int32_t> outCount,
    const nn::sf::InArray<char>& wherePhrase) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);

    CHECK_IPC_STRING(wherePhrase, WherePhraseLengthMax + 1);

    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

    int actualCount = 0;

    NN_RESULT_DO(detail::service::core::NewsDatabase::GetInstance().Count(&actualCount, wherePhrase.GetData()));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result NewsDatabaseService::CountWithKey(nn::sf::Out<std::int32_t> outCount,
    bool distinct, const nn::sf::InArray<char>& key, const nn::sf::InArray<char>& wherePhrase) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);

    CHECK_IPC_STRING(key, KeyLengthMax + 1);
    CHECK_IPC_STRING(wherePhrase, WherePhraseLengthMax + 1);

    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

    int actualCount = 0;

    NN_RESULT_DO(detail::service::core::NewsDatabase::GetInstance().Count(&actualCount, distinct, key.GetData(), wherePhrase.GetData()));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result NewsDatabaseService::UpdateIntegerValue(const nn::sf::InArray<char>& key, std::int32_t value, const nn::sf::InArray<char>& wherePhrase) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);

    CHECK_IPC_STRING(key, KeyLengthMax + 1);
    CHECK_IPC_STRING(wherePhrase, WherePhraseLengthMax + 1);

    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

    NN_RESULT_DO(detail::service::core::NewsDatabase::GetInstance().Update(key.GetData(), value, wherePhrase.GetData()));

    detail::service::core::FileSystem::Commit(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME);

    NN_RESULT_SUCCESS;
}

nn::Result NewsDatabaseService::UpdateIntegerValueWithAddition(const nn::sf::InArray<char>& key, std::int32_t value, const nn::sf::InArray<char>& wherePhrase) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);

    CHECK_IPC_STRING(key, KeyLengthMax + 1);
    CHECK_IPC_STRING(wherePhrase, WherePhraseLengthMax + 1);

    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

    NN_RESULT_DO(detail::service::core::NewsDatabase::GetInstance().UpdateWithAddition(key.GetData(), value, wherePhrase.GetData()));

    detail::service::core::FileSystem::Commit(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME);

    NN_RESULT_SUCCESS;
}

nn::Result NewsDatabaseService::UpdateStringValue(const nn::sf::InArray<char>& key, const nn::sf::InArray<char>& value, const nn::sf::InArray<char>& wherePhrase) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);

    CHECK_IPC_STRING(key, KeyLengthMax + 1);
    CHECK_IPC_STRING(value, StringValueLengthMax + 1);
    CHECK_IPC_STRING(wherePhrase, WherePhraseLengthMax + 1);

    NN_RESULT_THROW_UNLESS(detail::service::core::StringVerifier::VerifyDatabaseStringValue(value.GetData()), ResultInvalidArgument());

    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

    NN_RESULT_DO(detail::service::core::NewsDatabase::GetInstance().Update(key.GetData(), value.GetData(), wherePhrase.GetData()));

    detail::service::core::FileSystem::Commit(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME);

    NN_RESULT_SUCCESS;
}

}}}}
