﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/service/news_NewsDataService.h>
#include <nn/news/detail/service/news_Common.h>
#include <nn/news/detail/service/core/news_ForegroundStorageLockManager.h>

#define CHECK_CAPABILITY(flag) \
    NN_RESULT_THROW_UNLESS(m_Capability.IsPermitted(flag), ResultNotPermitted())

#define CHECK_IPC_STRING(value, maxSize) \
    do                                                              \
    {                                                               \
        size_t size = value.GetLength();                            \
        if (size == 0 || size > maxSize || value[size - 1] != '\0') \
        {                                                           \
            NN_RESULT_THROW(ResultInvalidArgument());               \
        }                                                           \
    }                                                               \
    while (NN_STATIC_CONDITION(0))

namespace nn { namespace news { namespace detail { namespace service {

NewsDataService::NewsDataService(const Capability& capability) NN_NOEXCEPT :
    m_Capability(capability),
    m_IsOpened(false)
{
    detail::service::core::ForegroundStorageLockManager::GetInstance().AcquireDataAccessorLock();
}

NewsDataService::~NewsDataService() NN_NOEXCEPT
{
    if (m_IsOpened)
    {
        nn::fs::CloseFile(m_Handle);
    }

    detail::service::core::ForegroundStorageLockManager::GetInstance().ReleaseDataAccessorLock();
}

nn::Result NewsDataService::Open(const nn::sf::InArray<char>& name) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);

    CHECK_IPC_STRING(name, DataNameLengthMax + 1);

    NN_RESULT_THROW_UNLESS(!m_IsOpened, ResultAlreadyOpened());
    NN_RESULT_THROW_UNLESS(detail::service::core::StringVerifier::VerifyDataName(name.GetData()), ResultInvalidArgument());

    char path[detail::service::core::Path::NewsDataPathLengthMax + 1] = {};
    detail::service::core::Path::MakeNewsDataPath(path, sizeof (path), name.GetData());

    NN_RESULT_DO(nn::fs::OpenFile(&m_Handle, path, nn::fs::OpenMode_Read));

    m_IsOpened = true;

    NN_RESULT_SUCCESS;
}

nn::Result NewsDataService::OpenWithNewsRecord(const nn::news::NewsRecord& record) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);

    NN_RESULT_THROW_UNLESS(!m_IsOpened, ResultAlreadyOpened());

    char path[detail::service::core::Path::NewsDataPathLengthMax + 1] = {};
    detail::service::core::Path::MakeNewsDataPath(path, sizeof (path), record);

    NN_RESULT_DO(nn::fs::OpenFile(&m_Handle, path, nn::fs::OpenMode_Read));

    m_IsOpened = true;

    NN_RESULT_SUCCESS;
}

nn::Result NewsDataService::OpenWithNewsRecordV1(const nn::news::NewsRecordV1& record) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);

    NN_RESULT_THROW_UNLESS(!m_IsOpened, ResultAlreadyOpened());

    char path[detail::service::core::Path::NewsDataPathLengthMax + 1] = {};
    detail::service::core::Path::MakeNewsDataPath(path, sizeof (path), record);

    NN_RESULT_DO(nn::fs::OpenFile(&m_Handle, path, nn::fs::OpenMode_Read));

    m_IsOpened = true;

    NN_RESULT_SUCCESS;
}

nn::Result NewsDataService::Read(nn::sf::Out<std::uint64_t> outSize,
    std::int64_t offset, const nn::sf::OutBuffer& buffer) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);

    NN_RESULT_THROW_UNLESS(m_IsOpened, ResultNotOpened());

    size_t readSize = 0;

    NN_RESULT_DO(nn::fs::ReadFile(&readSize, m_Handle, offset, buffer.GetPointerUnsafe(), buffer.GetSize()));

    *outSize = readSize;

    NN_RESULT_SUCCESS;
}

nn::Result NewsDataService::GetSize(nn::sf::Out<std::int64_t> outSize) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);

    NN_RESULT_THROW_UNLESS(m_IsOpened, ResultNotOpened());

    NN_RESULT_DO(nn::fs::GetFileSize(outSize.GetPointer(), m_Handle));

    NN_RESULT_SUCCESS;
}

}}}}
