﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/service/core/news_Version.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

namespace
{
    const char* g_FilePath = "news-sys:/sup.version";
}

namespace
{
    int32_t GetSystemVersion() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(int32_t, s_Version, = 0);

        if (s_Version != 0)
        {
            return s_Version;
        }

        int32_t version = 0;

        if (nn::settings::fwdbg::GetSettingsItemValue(&version, sizeof (version), "news", "system_version") == sizeof (version))
        {
            s_Version = version;
        }
        else
        {
            // 初期値は 1
            s_Version = 1;
        }

        return s_Version;
    }

    int32_t GetSupportedSemanticsVersion(int formatVersion) NN_NOEXCEPT
    {
        // システムやニュースビューアがサポートするバージョンを増やす度に更新する。
        // ここを更新する際、ニュースシステムバージョン (news.settings.fwdbg.yml) も必ず更新すること。
        switch (GetSystemVersion())
        {
        case 1: // システムバージョン 1 は、ニュースバージョン 1.1 のみ対応。
            {
                if (formatVersion == 1)
                {
                    return 1;
                }
            }
            break;
        case 2: // システムバージョン 2 は、ニュースバージョン 1.2 まで対応。
            {
                if (formatVersion == 1)
                {
                    return 2;
                }
            }
            break;
        case 3: // システムバージョン 3 は、ニュースバージョン 1.3 まで対応。
            {
                if (formatVersion == 1)
                {
                    return 3;
                }
            }
            break;
        case 4: // システムバージョン 4 以降は、ニュースバージョン 1.4 まで対応。
            NN_FALL_THROUGH;
        default:
            {
                if (formatVersion == 1)
                {
                    return 4;
                }
            }
            break;
        }

        return -1;
    }
}

Version::Version() NN_NOEXCEPT :
    m_Mutex(true),
    m_Version(0)
{
}

bool Version::IsSupported(const NewsVersion& version) NN_NOEXCEPT
{
    return GetSupportedSemanticsVersion(version.format) >= version.semantics;
}

nn::Result Version::Clear() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

    NN_RESULT_TRY(nn::fs::DeleteFile(g_FilePath))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    FileSystem::Commit(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME);

    m_Version = 0;

    NN_RESULT_SUCCESS;
}

nn::Result Version::RequireSystemUpdate() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (m_Version == GetSystemVersion())
    {
        NN_RESULT_SUCCESS;
    }

    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();
    {
        NN_RESULT_DO(FileSystem::CreateFile(g_FilePath, 4));

        nn::fs::FileHandle handle = {};
        NN_RESULT_DO(nn::fs::OpenFile(&handle, g_FilePath, nn::fs::OpenMode_Write));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        int32_t version = GetSystemVersion();

        NN_RESULT_DO(nn::fs::WriteFile(handle, 0, &version, 4,
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }
    FileSystem::Commit(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME);

    m_Version = GetSystemVersion();

    NN_RESULT_SUCCESS;
}

nn::Result Version::IsSystemUpdateRequired(bool* outIsRequired) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIsRequired);

    *outIsRequired = false;

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (m_Version == 0)
    {
        NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();
        {
            nn::fs::FileHandle handle = {};

            NN_RESULT_TRY(nn::fs::OpenFile(&handle, g_FilePath, nn::fs::OpenMode_Read))
                NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
                {
                    NN_RESULT_SUCCESS;
                }
            NN_RESULT_END_TRY;

            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handle);
            };

            int32_t version = 0;
            NN_RESULT_DO(nn::fs::ReadFile(handle, 0, &version, 4));

            m_Version = version;
        }

        // 最新の状態になったことが判明したので、比較用ファイルは不要となる。
        if (m_Version < GetSystemVersion())
        {
            nn::fs::DeleteFile(g_FilePath);
            FileSystem::Commit(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME);
        }
    }

    // セーブデータに保存されているバージョンが現在のバージョンと同等以上なら、システム更新すべき。
    *outIsRequired = (m_Version >= GetSystemVersion());

    NN_RESULT_SUCCESS;
}

}}}}}
