﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/service/core/news_TopicListDownloader.h>
#include <nn/news/detail/service/core/news_TopicListReader.h>
#include <nn/news/detail/service/core/news_AccountCountryChecker.h>
#include <nn/news/detail/service/util/news_HttpErrorHandler.h>
#include <nn/nn_Version.h>
#include <nn/bcat/service/bcat_ApiService.h>
#include <nn/bcat/bcat_Result.h>
#include <nn/settings/system/settings_Language.h>
#include <nn/settings/system/settings_SerialNumber.h>
#include <nn/ssl/ssl_Context.h>
#include <curl/curl.h>

#if defined (NN_BUILD_CONFIG_SPEC_GENERIC)

    // Generic Windows
    #define UA_PLATFORM "5797bc97-d32c-48ad-b47e-3847bda0db7c"

#elif defined(NN_BUILD_CONFIG_SPEC_NX)

    #if defined (NN_BUILD_CONFIG_OS_WIN)
        // NX Windows
        #define UA_PLATFORM "159b8d57-d7af-4ce6-823e-c38d1d661918"
    #elif defined(NN_BUILD_CONFIG_OS_HORIZON)
        // NX Horizon
        #define UA_PLATFORM "789f928b-138e-4b2f-afeb-1acae821d897"
    #else
        #error "Unsupported OS"
    #endif

#else

    #error "Unsupported Spec"

#endif

#define CURL_EASY_DO(exp) \
    do                                                              \
    {                                                               \
        CURLcode code_ = (exp);                                     \
                                                                    \
        if (code_ != CURLE_OK)                                      \
        {                                                           \
            return detail::service::util::HandleHttpError(code_);   \
        }                                                           \
    }                                                               \
    while (NN_STATIC_CONDITION(0))

#define CURL_SLIST_APPEND(headers, header) \
    do                                                                  \
    {                                                                   \
        curl_slist* headers_ = curl_slist_append(headers, header);      \
                                                                        \
        NN_RESULT_THROW_UNLESS(headers_, ResultHttpErrorOutOfMemory()); \
        headers = headers_;                                             \
    }                                                                   \
    while (NN_STATIC_CONDITION(0))

namespace nn { namespace news { namespace detail { namespace service { namespace core {

namespace
{
    const char* g_Fqdn = "bcat-topics-%.cdn.nintendo.net";
}

namespace
{
    struct DownloadParam
    {
        Bit8* buffer;
        size_t bufferSize;
        size_t downloadedSize;
        nn::os::Event* pCancelEvent;
        nn::nifm::NetworkConnection* pConnection;
    };
}

namespace
{
    void GetUserAgent(char* buffer, size_t size) NN_NOEXCEPT
    {
        nn::util::SNPrintf(buffer, size, "libcurl (nnNews; %s; SDK %d.%d.%d.%d)",
            UA_PLATFORM, NN_SDK_VERSION_MAJOR, NN_SDK_VERSION_MINOR, NN_SDK_VERSION_MICRO, NN_SDK_VERSION_RELSTEP);
    }
}

TopicListDownloader::TopicListDownloader() NN_NOEXCEPT :
    m_pCancelEvent(nullptr),
    m_ArchiveSize(0),
    m_DecodedSize(0)
{
}

void TopicListDownloader::SetCancelEvent(nn::os::Event* pCancelEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pCancelEvent);

    m_pCancelEvent = pCancelEvent;
}

nn::Result TopicListDownloader::Download(int* outCount, TopicId* outTopicIds, int count,
    nn::ApplicationId targetAppId, nn::ApplicationId newsAppId, const char* newsPassphrase,
    nn::nifm::NetworkConnection* pConnection) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outTopicIds);
    NN_SDK_REQUIRES_GREATER(count, 0);
    NN_SDK_REQUIRES_NOT_EQUAL(targetAppId, nn::ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_EQUAL(newsAppId, nn::ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(newsPassphrase);
    NN_SDK_REQUIRES_NOT_NULL(pConnection);

    NN_DETAIL_NEWS_INFO("[news] TopicListDownloader::Download(%016llx) ...\n", targetAppId.value);

    NN_RESULT_TRY(StepDownload(targetAppId, pConnection))
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceled());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceled());

    NN_RESULT_DO(StepDecode(newsAppId, newsPassphrase));

    NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceled());

    NN_RESULT_DO(StepRead(outCount, outTopicIds, count));

    NN_RESULT_SUCCESS;
}

nn::Result TopicListDownloader::StepDownload(nn::ApplicationId appId, nn::nifm::NetworkConnection* pConnection) NN_NOEXCEPT
{
    NN_DETAIL_NEWS_INFO("[news] - TopicListDownloader::StepDownload() ...\n");

    nn::settings::LanguageCode language;
    nn::settings::GetLanguageCode(&language);

    AccountCountryChecker::QueryString queryString;
    AccountCountryChecker::GetInstance().GetQueryString(&queryString);

    char url[256] = {};
    nn::util::SNPrintf(url, sizeof (url), "https://%s/api/nx/v1/titles/%016llx/topics?l=%s%s%s",
        g_Fqdn, appId.value, language.string, queryString.value[0] ? "&" : "", queryString.value);

    NN_DETAIL_NEWS_INFO("[news] Url = %s\n", url);

    CURL* curl = curl_easy_init();

    NN_RESULT_THROW_UNLESS(curl, ResultHttpErrorFailedInit());

    curl_slist* headers = nullptr;

    NN_UTIL_SCOPE_EXIT
    {
        curl_easy_cleanup(curl);

        if (headers)
        {
            curl_slist_free_all(headers);
        }
    };

    DownloadParam downloadParam = {m_ArchiveData, sizeof (m_ArchiveData), 0, m_pCancelEvent, pConnection};

    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_URL, url));

    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_SSL_CTX_FUNCTION, SslCtxFunction));

    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, HttpWriteFunction));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_WRITEDATA, &downloadParam));

    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_XFERINFOFUNCTION, HttpProgressFunction));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_XFERINFODATA, &downloadParam));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0));

    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_MAXREDIRS, 1));

    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_LOW_SPEED_LIMIT, 1));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_LOW_SPEED_TIME, 30));

    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT, 30));

    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_SSL_VERIFYDATE, 0));

    char ua[128] = {};
    GetUserAgent(ua, sizeof (ua));

    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_USERAGENT, ua));

    {
        nn::settings::system::SerialNumber serialNumber = {};
        nn::settings::system::GetSerialNumber(&serialNumber);

        char header[128] = {};
        nn::util::SNPrintf(header, sizeof (header), "X-Nintendo-Serial-Number: %s", serialNumber.string);

        CURL_SLIST_APPEND(headers, header);
    }

    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_HTTPHEADER, headers));

    // CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_VERBOSE, 1));

    CURL_EASY_DO(curl_easy_perform(curl));

    long statusCode = 0;
    CURL_EASY_DO(curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &statusCode));

    NN_RESULT_DO(detail::service::util::HandleHttpStatusCode(statusCode));

    m_ArchiveSize = downloadParam.downloadedSize;

    NN_RESULT_SUCCESS;
}

nn::Result TopicListDownloader::StepDecode(nn::ApplicationId appId, const char* passphrase) NN_NOEXCEPT
{
    NN_DETAIL_NEWS_INFO("[news] - TopicListDownloader::StepDecode() ...\n");

    NN_RESULT_TRY(nn::bcat::service::DecodeArchiveFile(&m_DecodedSize,
        m_DecodedData, sizeof (m_DecodedData), m_ArchiveData, m_ArchiveSize, appId, passphrase))
        NN_RESULT_CATCH(nn::bcat::ResultDataVerificationFailed)
        {
            NN_RESULT_THROW(ResultDataVerificationFailed());
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result TopicListDownloader::StepRead(int* outCount, TopicId* outTopicIds, int count) NN_NOEXCEPT
{
    NN_DETAIL_NEWS_INFO("[news] - TopicListDownloader::StepRead() ...\n");

    nne::nlib::MemoryInputStream stream;

    stream.Init(m_DecodedData, m_DecodedSize);

    TopicListReader reader;

    NN_RESULT_DO(reader.Read(outCount, outTopicIds, count, stream));

    NN_RESULT_SUCCESS;
}

bool TopicListDownloader::IsCanceled() NN_NOEXCEPT
{
    return (m_pCancelEvent && m_pCancelEvent->TryWait());
}

CURLcode TopicListDownloader::SslCtxFunction(CURL* curl, void* ssl, void* param) NN_NOEXCEPT
{
    NN_UNUSED(param);

    nn::ssl::Context* context = reinterpret_cast<nn::ssl::Context*>(ssl);

    if (context->Create(nn::ssl::Context::SslVersion_Auto).IsFailure())
    {
        return CURLE_ABORTED_BY_CALLBACK;
    }

    return CURLE_OK;
}

size_t TopicListDownloader::HttpWriteFunction(char* buffer, size_t size, size_t count, void* param) NN_NOEXCEPT
{
    DownloadParam* pParam = static_cast<DownloadParam*>(param);

    size_t bufferSize = size * count;

    if (pParam->downloadedSize + bufferSize < pParam->bufferSize)
    {
        std::memcpy(&pParam->buffer[pParam->downloadedSize], buffer, bufferSize);
        pParam->downloadedSize += bufferSize;
    }
    else
    {
        return 0;
    }

    return bufferSize;
}

int TopicListDownloader::HttpProgressFunction(void* param, curl_off_t dlTotal, curl_off_t dlNow, curl_off_t ulTotal, curl_off_t ulNow) NN_NOEXCEPT
{
    NN_UNUSED(dlTotal);
    NN_UNUSED(dlNow);
    NN_UNUSED(ulTotal);
    NN_UNUSED(ulNow);

    DownloadParam* pParam = static_cast<DownloadParam*>(param);

    return ((pParam->pCancelEvent && pParam->pCancelEvent->TryWait()) || !pParam->pConnection->IsAvailable());
}

}}}}}
