﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/service/core/news_StorageManager.h>
#include <nn/news/detail/service/core/news_NewsDatabase.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

namespace
{
    // ディレクトリエントリーの 1 回の読み込みバッファのサイズ。
    const int DirectoryEntryCountPerRead = 50;
}

namespace
{
    nn::Result DeleteMissingRecords(bool* outIsCommitRequired) NN_NOEXCEPT
    {
        *outIsCommitRequired = false;

        NewsRecord* records = new NewsRecord[DataCountMax];
        int recordCount;

        NN_RESULT_THROW_UNLESS(records, ResultOutOfMemory());

        NN_UTIL_SCOPE_EXIT
        {
            delete[] records;
        };

        NN_RESULT_DO(NewsDatabase::GetInstance().GetList(&recordCount, records, "", "", 0, DataCountMax));

        if (recordCount == 0)
        {
            NN_RESULT_SUCCESS;
        }

        nn::fs::DirectoryHandle handle = {};

        NN_RESULT_TRY(nn::fs::OpenDirectory(&handle, "news:/data", nn::fs::OpenDirectoryMode_File))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
                NN_RESULT_DO(NewsDatabase::GetInstance().DeleteAll());
                *outIsCommitRequired = true;

                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseDirectory(handle);
        };

        int64_t entryCount;
        NN_RESULT_DO(nn::fs::GetDirectoryEntryCount(&entryCount, handle));

        if (entryCount == recordCount)
        {
            NN_RESULT_SUCCESS;
        }

        // 784 * DirectoryEntryCountPerRead
        nn::fs::DirectoryEntry* entries = new nn::fs::DirectoryEntry[DirectoryEntryCountPerRead];

        NN_RESULT_THROW_UNLESS(entries, ResultOutOfMemory());

        NN_UTIL_SCOPE_EXIT
        {
            delete[] entries;
        };

        while (NN_STATIC_CONDITION(1))
        {
            NN_RESULT_DO(nn::fs::ReadDirectory(&entryCount, entries, handle, DirectoryEntryCountPerRead));

            if (entryCount == 0)
            {
                break;
            }

            for (int i = 0; i < entryCount; i++)
            {
                UserId userId;
                NewsId newsId;

                int count = std::sscanf(entries[i].name, "%21s_%22s.msgpack", userId.value, newsId.value);

                if (count == 2)
                {
                    if (userId.value[0] == 'D')
                    {
                        userId.value[0] = '\0';
                    }

                    for (int r = 0; r < recordCount; r++)
                    {
                        if (nn::util::Strncmp(records[r].userId.value, userId.value, sizeof (userId.value)) == 0 &&
                            nn::util::Strncmp(records[r].newsId.value, newsId.value, sizeof (newsId.value)) == 0)
                        {
                            // データが存在するニュースのレコードをクリアする。
                            records[r].newsId.value[0] = '\0';
                            break;
                        }
                    }
                }
            }
        }

        for (int i = 0; i < recordCount; i++)
        {
            if (records[i].newsId.value[0])
            {
                char wherePhrase[64] = {};
                nn::util::SNPrintf(wherePhrase, sizeof (wherePhrase),
                    "user_id = '%s' AND news_id = '%s'", records[i].userId.value, records[i].newsId.value);

                NN_RESULT_DO(NewsDatabase::GetInstance().Delete(wherePhrase));
                *outIsCommitRequired = true;
            }
        }

        NN_RESULT_SUCCESS;
    } // NOLINT(impl/function_size)
}

StorageManager::StorageManager() NN_NOEXCEPT :
    m_Mutex(false)
{
}

nn::Result StorageManager::Recover() NN_NOEXCEPT
{
    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

    bool isCommitRequired1;
    NN_RESULT_DO(NewsDatabase::GetInstance().Shrink(&isCommitRequired1));

    bool isCommitRequired2;
    NN_RESULT_DO(DeleteMissingRecords(&isCommitRequired2));

    if (isCommitRequired1 || isCommitRequired2)
    {
        FileSystem::Commit(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME);
    }

    NN_RESULT_SUCCESS;
}

void StorageManager::Lock() NN_NOEXCEPT
{
    m_Mutex.Lock();
}

void StorageManager::Unlock() NN_NOEXCEPT
{
    m_Mutex.Unlock();
}

}}}}}
