﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/service/core/news_ReceivedHistoryManager.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

namespace
{
    const char* g_FilePath = "news-sys:/history.bin";
}

ReceivedHistoryManager::ReceivedHistoryManager() NN_NOEXCEPT :
    m_Mutex(true),
    m_Count(0),
    m_BlockIndex(-1)
{
}

nn::Result ReceivedHistoryManager::AddOrUpdate(const Record& record) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();
    {
        nn::fs::FileHandle handle = {};
        size_t size;

        NN_RESULT_DO(Open(&size, &handle));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        int indexInFile;
        NN_RESULT_DO(SearchUpdatableRecord(&indexInFile, handle, size, record));

        NN_RESULT_THROW_UNLESS(indexInFile < HistoryCountMax, ResultNotFound());

        if (indexInFile != -1)
        {
            NN_RESULT_DO(nn::fs::WriteFile(handle, sizeof (Record) * indexInFile, &record, sizeof (Record),
                nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

            int index = SearchUpdatableRecordFromCache(record);

            // ファイルから検索する際、必ずキャッシュにレコードを載せるはず。
            NN_SDK_ASSERT(index != -1);

            if (index != -1)
            {
                m_Records[index] = record;
            }
        }
        else
        {
            if (size == sizeof (Record) * HistoryCountMax)
            {
                NN_RESULT_DO(DeleteHeadBlock(handle));
                size -= sizeof (m_Records);
            }

            NN_RESULT_DO(nn::fs::WriteFile(handle, size, &record, sizeof (Record),
                nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

            if (m_Count < CountPerBlock)
            {
                m_Records[m_Count++] = record;
            }
        }
    }
    FileSystem::Commit(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME);

    NN_RESULT_SUCCESS;
}

nn::Result ReceivedHistoryManager::Clear() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

    NN_RESULT_TRY(nn::fs::DeleteFile(g_FilePath))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    FileSystem::Commit(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME);

    m_BlockIndex = -1;
    m_Count = 0;

    NN_RESULT_SUCCESS;
}

nn::Result ReceivedHistoryManager::IsImportable(bool* outIsImportable, const SearchKey& key) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIsImportable);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    // キャッシュ上のものを最初に確認する。
    if (m_BlockIndex != -1)
    {
        if (SearchRecordForImport(key) != -1)
        {
            *outIsImportable = false;
            NN_RESULT_SUCCESS;
        }
    }

    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

    nn::fs::FileHandle handle = {};
    size_t size;

    NN_RESULT_DO(Open(&size, &handle));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    int searchedIndex = m_BlockIndex;

    if (size == 0)
    {
        *outIsImportable = true;
        NN_RESULT_SUCCESS;
    }

    int loop = static_cast<int>(size - 1) / sizeof (m_Records) + 1;

    for (int i = 0; i < loop; i++)
    {
        if (i == searchedIndex)
        {
            continue;
        }

        NN_RESULT_DO(Read(handle, i));

        if (SearchRecordForImport(key) != -1)
        {
            *outIsImportable = false;
            NN_RESULT_SUCCESS;
        }
    }

    *outIsImportable = true;

    NN_RESULT_SUCCESS;
}

nn::Result ReceivedHistoryManager::IsOverwritable(bool* outIsOverwritable, const SearchKey& key) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIsOverwritable);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    // キャッシュ上のものを最初に確認する。
    if (m_BlockIndex != -1)
    {
        int index = SearchRecordForOverwrite(key);

        if (index != -1)
        {
            *outIsOverwritable = (m_Records[index].dataId != key.dataId);
            NN_RESULT_SUCCESS;
        }
    }

    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

    nn::fs::FileHandle handle = {};
    size_t size;

    NN_RESULT_DO(Open(&size, &handle));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    if (size == 0)
    {
        *outIsOverwritable = false;
        NN_RESULT_SUCCESS;
    }

    int loop = static_cast<int>(size - 1) / sizeof (m_Records) + 1;

    int searchedIndex = m_BlockIndex;

    for (int i = 0; i < loop; i++)
    {
        if (i == searchedIndex)
        {
            continue;
        }

        NN_RESULT_DO(Read(handle, i));

        int index = SearchRecordForOverwrite(key);

        if (index != -1)
        {
            *outIsOverwritable = (m_Records[index].dataId != key.dataId);
            NN_RESULT_SUCCESS;
        }
    }

    *outIsOverwritable = false;

    NN_RESULT_SUCCESS;
}

nn::Result ReceivedHistoryManager::Open(size_t* outSize, nn::fs::FileHandle* outHandle) NN_NOEXCEPT
{
    *outSize = 0;

    NN_RESULT_TRY(nn::fs::OpenFile(outHandle, g_FilePath, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_DO(Create(outHandle));
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    int64_t size;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&size, *outHandle));

    if ((size % sizeof (Record)) != 0)
    {
        NN_DETAIL_NEWS_WARN("[news] %s is corrupted. (size %% sizeof (Record))\n", g_FilePath);
        nn::fs::CloseFile(*outHandle);

        NN_RESULT_DO(Create(outHandle));

        NN_RESULT_SUCCESS;
    }

    int count = static_cast<int>(size) / sizeof (Record);

    if (count == 0)
    {
        NN_RESULT_SUCCESS;
    }

    if (count > HistoryCountMax)
    {
        NN_DETAIL_NEWS_WARN("[news] %s is corrupted. (count > HistoryCountMax)\n", g_FilePath);
        nn::fs::CloseFile(*outHandle);

        NN_RESULT_DO(Create(outHandle));

        NN_RESULT_SUCCESS;
    }

    *outSize = static_cast<size_t>(size);

    NN_RESULT_SUCCESS;
}

nn::Result ReceivedHistoryManager::Create(nn::fs::FileHandle* outHandle) NN_NOEXCEPT
{
    NN_RESULT_TRY(nn::fs::DeleteFile(g_FilePath))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    NN_RESULT_DO(nn::fs::CreateFile(g_FilePath, 0));

    nn::fs::FileHandle handle = {};
    NN_RESULT_DO(nn::fs::OpenFile(&handle, g_FilePath, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));

    *outHandle = handle;

    NN_RESULT_SUCCESS;
}

nn::Result ReceivedHistoryManager::Read(nn::fs::FileHandle handle, int blockIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(blockIndex >= 0);

    if (m_BlockIndex == blockIndex)
    {
        NN_RESULT_SUCCESS;
    }

    size_t size;

    NN_RESULT_TRY(nn::fs::ReadFile(&size, handle, sizeof (m_Records) * blockIndex, m_Records, sizeof (m_Records)))
        NN_RESULT_CATCH_ALL
        {
            m_BlockIndex = -1;
            m_Count = 0;
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    if ((size % sizeof (Record)) != 0)
    {
        m_BlockIndex = -1;
        m_Count = 0;
        NN_RESULT_SUCCESS;
    }

    m_BlockIndex = blockIndex;
    m_Count = static_cast<int>(size) / sizeof (Record);

    NN_RESULT_SUCCESS;
}

nn::Result ReceivedHistoryManager::DeleteHeadBlock(nn::fs::FileHandle handle) NN_NOEXCEPT
{
    int64_t size;
    NN_RESULT_DO(nn::fs::GetFileSize(&size, handle));

    if (size <= sizeof (m_Records))
    {
        NN_RESULT_DO(nn::fs::SetFileSize(handle, 0));
        NN_RESULT_SUCCESS;
    }

    int loop = static_cast<int>(size - 1) / sizeof (m_Records);

    for (int i = 0; i < loop; i++)
    {
        NN_RESULT_DO(Read(handle, i + 1));

        NN_RESULT_DO(nn::fs::WriteFile(handle, sizeof (m_Records) * i, m_Records, sizeof (Record) * m_Count,
            nn::fs::WriteOption::MakeValue(0)));
    }

    NN_RESULT_DO(nn::fs::FlushFile(handle));
    NN_RESULT_DO(nn::fs::SetFileSize(handle, size - sizeof (m_Records)));

    m_BlockIndex--;

    NN_RESULT_SUCCESS;
}

nn::Result ReceivedHistoryManager::SearchUpdatableRecord(int* outIndex, nn::fs::FileHandle handle, size_t size, const Record& record) NN_NOEXCEPT
{
    if (size == 0)
    {
        *outIndex = -1;
        NN_RESULT_SUCCESS;
    }

    // キャッシュ上のものを最初に確認する。
    if (m_BlockIndex != -1)
    {
        int index = SearchUpdatableRecordFromCache(record);

        if (index != -1)
        {
            *outIndex = m_BlockIndex * CountPerBlock + index;
            NN_RESULT_SUCCESS;
        }
    }

    int loop = static_cast<int>(size - 1) / sizeof (m_Records) + 1;

    int searchedIndex = m_BlockIndex;

    for (int i = 0; i < loop; i++)
    {
        if (i == searchedIndex)
        {
            continue;
        }

        NN_RESULT_DO(Read(handle, i));

        int index = SearchUpdatableRecordFromCache(record);

        if (index != -1)
        {
            *outIndex = m_BlockIndex * CountPerBlock + index;
            NN_RESULT_SUCCESS;
        }
    }

    *outIndex = -1;

    NN_RESULT_SUCCESS;
}

int ReceivedHistoryManager::SearchUpdatableRecordFromCache(const Record& record) NN_NOEXCEPT
{
    for (int i = 0; i < m_Count; i++)
    {
        if (record.isIndividual == m_Records[i].isIndividual)
        {
            if (record.isIndividual)
            {
                if (record.userId == m_Records[i].userId && record.language == m_Records[i].language &&
                    record.domain == m_Records[i].domain && record.newsId == m_Records[i].newsId)
                {
                    return i;
                }
            }
            else
            {
                if (record.language == m_Records[i].language &&
                    record.domain == m_Records[i].domain && record.newsId == m_Records[i].newsId)
                {
                    return i;
                }
            }
        }
    }

    return -1;
}

int ReceivedHistoryManager::SearchRecordForImport(const SearchKey& key) NN_NOEXCEPT
{
    for (int i = 0; i < m_Count; i++)
    {
        if (key.isIndividual == m_Records[i].isIndividual)
        {
            if (key.isIndividual)
            {
                if (key.userId == m_Records[i].userId && key.domain == m_Records[i].domain && key.newsId == m_Records[i].newsId)
                {
                    return i;
                }
            }
            else
            {
                if (key.domain == m_Records[i].domain && key.newsId == m_Records[i].newsId)
                {
                    return i;
                }
            }
        }
    }

    return -1;
}

int ReceivedHistoryManager::SearchRecordForOverwrite(const SearchKey& key) NN_NOEXCEPT
{
    for (int i = 0; i < m_Count; i++)
    {
        if (key.isIndividual == m_Records[i].isIndividual)
        {
            if (key.isIndividual)
            {
                if (key.userId == m_Records[i].userId && key.language == m_Records[i].language &&
                    key.domain == m_Records[i].domain && key.newsId == m_Records[i].newsId)
                {
                    return i;
                }
            }
            else
            {
                if (key.language == m_Records[i].language &&
                    key.domain == m_Records[i].domain && key.newsId == m_Records[i].newsId)
                {
                    return i;
                }
            }
        }
    }

    return -1;
}

}}}}}
