﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/service/core/news_NewsListReader.h>
#include <nn/news/detail/service/msgpack/news_MsgpackReader.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

NewsListReader::NewsListReader() NN_NOEXCEPT :
    m_ListInfo(),
    m_ListInfoBits(),
    m_Entry(),
    m_EntryBits(),
    m_Entries(nullptr),
    m_Index(0),
    m_Offset(0),
    m_Count(0),
    m_CurrentLanguage(""),
    m_IsLanguageEntryParsing(false),
    m_IsConditionsParsing(false)
{
}

void NewsListReader::SetLanguage(const char* language) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(language);

    m_CurrentLanguage = language;
}

nn::Result NewsListReader::Read(int* outCount, nne::nlib::InputStream& stream, DataEntry* entries, int offset, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);

    m_Entries = entries;
    m_Offset = offset;
    m_Count = count;

    m_Index = 0;

    detail::service::msgpack::MsgpackReader reader;

    NN_RESULT_TRY(reader.Parse(stream, *this, sizeof (Url)))
        NN_RESULT_CATCH(ResultAbortedByCallback)
        {
            NN_RESULT_THROW(ResultInvalidFormat());
        }
    NN_RESULT_END_TRY;

    *outCount = m_Index;

    NN_RESULT_SUCCESS;
}

const NewsListReader::ListInfo& NewsListReader::GetListInfo() const NN_NOEXCEPT
{
    return m_ListInfo;
}

bool NewsListReader::OnStartObject(const detail::service::msgpack::JsonPath& jsonPath) NN_NOEXCEPT
{
    if (jsonPath.Match("$.directories[0].data_list[*]"))
    {
        std::memset(&m_EntryBits, 0, sizeof (m_EntryBits));
        std::memset(&m_Entry, 0, sizeof (m_Entry));
    }
    else if (!m_EntryBits.conditions &&
        jsonPath.Match("$.directories[0].data_list[*].conditions"))
    {
        m_IsConditionsParsing = true;
        m_EntryBits.conditions = true;
    }

    return true;
}

bool NewsListReader::OnEndObject(const detail::service::msgpack::JsonPath& jsonPath) NN_NOEXCEPT
{
    if (jsonPath.Match("$.directories[0].data_list[*]"))
    {
        if (m_EntryBits.formatVersion && m_EntryBits.semanticsVersion && m_EntryBits.newsId)
        {
            if (m_EntryBits.dataId && m_EntryBits.url && m_EntryBits.size)
            {
                if (m_Offset == 0)
                {
                    if (m_Index < m_Count)
                    {
                        std::memcpy(&m_Entries[m_Index++], &m_Entry, sizeof (m_Entry));
                    }
                }
                else
                {
                    m_Offset--;
                }
            }
            else if (!(m_EntryBits.dataId || m_EntryBits.url || m_EntryBits.size))
            {
                // デフォルト言語・本体設定言語どちらも存在しない場合、そのエントリーを無視する。
            }
            else
            {
                return false;
            }
        }
        else
        {
            return false;
        }
    }
    else if (m_IsConditionsParsing &&
        jsonPath.Match("$.directories[0].data_list[*].conditions"))
    {
        m_IsConditionsParsing = false;
    }

    return true;
}

bool NewsListReader::OnStartArray(const detail::service::msgpack::JsonPath& jsonPath) NN_NOEXCEPT
{
    if (jsonPath.Compare("$.directories[0].data_list[]"))
    {
        // 必須項目の存在確認。
        if (!(m_ListInfoBits.topicId && m_ListInfoBits.isInService && m_ListInfoBits.isNintendoAccountRequired))
        {
            return false;
        }
    }

    return true;
}

bool NewsListReader::OnBoolean(const detail::service::msgpack::JsonPath& jsonPath, bool value) NN_NOEXCEPT
{
    if (!m_ListInfoBits.isNintendoAccountRequired &&
        jsonPath.Compare("$.na_required"))
    {
        m_ListInfo.isNintendoAccountRequired = value;
        m_ListInfoBits.isNintendoAccountRequired = true;
    }
    else if (!m_ListInfoBits.isTestDistribution &&
        jsonPath.Compare("$.test_distribution"))
    {
        m_ListInfo.isTestDistribution = value;
        m_ListInfoBits.isTestDistribution = true;
    }
    else if (m_IsLanguageEntryParsing && !m_EntryBits.isOverwrite &&
        jsonPath.Match("$.directories[0].data_list[*].languages[*].overwrite"))
    {
        m_Entry.isOverwrite = value;
        m_EntryBits.isOverwrite = true;
    }

    return true;
}

bool NewsListReader::OnInteger(const detail::service::msgpack::JsonPath& jsonPath, const detail::service::msgpack::AnyInteger& value) NN_NOEXCEPT
{
    if (!m_EntryBits.newsId &&
        jsonPath.Match("$.directories[0].data_list[*].news_id"))
    {
        m_Entry.newsId = value.u;
        m_EntryBits.newsId = true;
    }
    else if (!m_EntryBits.formatVersion &&
        jsonPath.Match("$.directories[0].data_list[*].version.format"))
    {
        m_Entry.version.format = static_cast<int8_t>(value.s);
        m_EntryBits.formatVersion = true;
    }
    else if (!m_EntryBits.semanticsVersion &&
        jsonPath.Match("$.directories[0].data_list[*].version.semantics"))
    {
        m_Entry.version.semantics = static_cast<int8_t>(value.s);
        m_EntryBits.semanticsVersion = true;
    }
    else if (m_IsLanguageEntryParsing && !m_EntryBits.dataId &&
        jsonPath.Match("$.directories[0].data_list[*].languages[*].data_id"))
    {
        m_Entry.dataId = value.u;
        m_EntryBits.dataId = true;
    }
    else if (m_IsLanguageEntryParsing && !m_EntryBits.size &&
        jsonPath.Match("$.directories[0].data_list[*].languages[*].size"))
    {
        m_Entry.size = value.s;
        m_EntryBits.size = true;
    }
    else if (m_IsConditionsParsing &&
        jsonPath.Match("$.directories[0].data_list[*].conditions.serial[*]"))
    {
        if (value.s >= 0 && value.s < 10)
        {
            m_Entry.conditions.serialBits |= (1 << static_cast<nn::Bit16>(value.s));
            m_Entry.conditions.isSerialBits = true;
        }
        else
        {
            return false;
        }
    }

    return true;
}

bool NewsListReader::OnString(const detail::service::msgpack::JsonPath& jsonPath, const char* value, size_t length, bool isOverflowed) NN_NOEXCEPT
{
    NN_UNUSED(isOverflowed);

    if (!m_ListInfoBits.topicId &&
        jsonPath.Compare("$.topic_id"))
    {
        if (!StringVerifier::VerifyTopicId(value))
        {
            return false;
        }
        nn::util::Strlcpy(m_ListInfo.topicId.value, value, sizeof (m_ListInfo.topicId.value));
        m_ListInfoBits.topicId = true;
    }
    else if (!m_ListInfoBits.isInService &&
        jsonPath.Compare("$.service_status"))
    {
        if (nn::util::Strncmp(value, "in_service", sizeof ("in_service")) == 0)
        {
            m_ListInfo.isInService = true;
        }
        else if (nn::util::Strncmp(value, "expired", sizeof ("expired")) == 0)
        {
            m_ListInfo.isInService = false;
        }
        else
        {
            return false;
        }
        m_ListInfoBits.isInService = true;
    }
    else if (!m_EntryBits.defaultLanguage &&
        jsonPath.Match("$.directories[0].data_list[*].default_language"))
    {
        nn::util::Strlcpy(m_Entry.defaultLanguage, value, sizeof (m_Entry.defaultLanguage));
        m_EntryBits.defaultLanguage = true;
    }
    else if (jsonPath.Match("$.directories[0].data_list[*].languages[*].language"))
    {
        m_IsLanguageEntryParsing = false;

        if (nn::util::Strncmp(value, m_CurrentLanguage, static_cast<int>(length) + 1) == 0)
        {
            nn::util::Strlcpy(m_Entry.language, value, sizeof (m_Entry.language));
            m_EntryBits.language = true;

            m_IsLanguageEntryParsing = true;

            // デフォルト言語のエントリーを先にパースしていたら上書きする。
            if (m_EntryBits.defaultLanguageEntry)
            {
                m_Entry.dataId = 0;
                m_Entry.url.value[0] = '\0';
                m_Entry.size = 0;
                m_Entry.isOverwrite = false;

                m_EntryBits.dataId = false;
                m_EntryBits.url = false;
                m_EntryBits.size = false;
                m_EntryBits.isOverwrite = false;
            }
        }
        else if (!m_EntryBits.defaultLanguageEntry &&
            nn::util::Strncmp(value, m_Entry.defaultLanguage, static_cast<int>(length) + 1) == 0)
        {
            // 目当ての言語のエントリーを先にパースしていたら言語はコピーしない。
            if (!m_EntryBits.language)
            {
                nn::util::Strlcpy(m_Entry.language, value, sizeof (m_Entry.language));
                m_EntryBits.language = true;

                m_IsLanguageEntryParsing = true;
            }

            m_EntryBits.defaultLanguageEntry = true;
        }
    }
    else if (m_IsLanguageEntryParsing && !m_EntryBits.url &&
        jsonPath.Match("$.directories[0].data_list[*].languages[*].url"))
    {
        nn::util::Strlcpy(m_Entry.url.value, value, sizeof (m_Entry.url.value));
        m_EntryBits.url = true;
    }

    return true;
}

}}}}}
