﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/service/core/news_NewsImporter.h>
#include <nn/news/detail/service/core/news_IncrementalId.h>
#include <nn/news/detail/service/core/news_NewlyArrivedEventListener.h>
#include <nn/news/detail/service/core/news_NewsMetadataReader.h>
#include <nn/news/detail/service/core/news_ReceivedHistoryManager.h>
#include <nn/news/detail/service/core/news_OverwriteEventListener.h>
#include <nn/news/detail/service/msgpack/news_FileInputStream.h>
#include <nn/time/time_Result.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

namespace
{
    nn::Result DeleteExpiredNews() NN_NOEXCEPT
    {
        nn::time::PosixTime now;

        NN_RESULT_TRY(nn::time::StandardNetworkSystemClock::GetCurrentTime(&now))
            NN_RESULT_CATCH(nn::time::ResultClockInvalid)
            {
                // ネットワーク時刻が取得できない場合、有効期限判定は行わない。
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        while (NN_STATIC_CONDITION(1))
        {
            NewsRecord record;
            int count;

            char wherePhrase[64] = {};
            nn::util::SNPrintf(wherePhrase, sizeof (wherePhrase), "expire_at > 0 AND expire_at < %lld", now.value);

            NN_RESULT_DO(NewsDatabase::GetInstance().GetList(&count, &record, wherePhrase, "", 0, 1));

            if (count == 0)
            {
                NN_RESULT_SUCCESS;
            }

            NN_DETAIL_NEWS_INFO("[news] Delete expired news. id=%s, received_at=%lld\n",
                record.newsId.value, record.receivedTime.value);

            nn::util::SNPrintf(wherePhrase, sizeof (wherePhrase), "news_id = '%s'", record.newsId.value);

            NN_RESULT_DO(NewsDatabase::GetInstance().Delete(wherePhrase));

            char path[Path::NewsDataPathLengthMax + 1] = {};
            Path::MakeNewsDataPath(path, sizeof (path), record);

            NN_RESULT_TRY(nn::fs::DeleteFile(path))
                NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
                {
                }
            NN_RESULT_END_TRY;
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result DeleteOldNews() NN_NOEXCEPT
    {
        NewsRecord record;
        int count;

        NN_RESULT_DO(NewsDatabase::GetInstance().GetList(&count, &record, "",
            "deletion_priority DESC, received_at ASC", 0, 1));

        if (count == 0)
        {
            NN_RESULT_SUCCESS;
        }

        NN_DETAIL_NEWS_INFO("[news] Delete old news. id=%s, received_at=%lld\n",
            record.newsId.value, record.receivedTime.value);

        char wherePhrase[64] = {};
        nn::util::SNPrintf(wherePhrase, sizeof (wherePhrase), "news_id = '%s'", record.newsId.value);

        NN_RESULT_DO(NewsDatabase::GetInstance().Delete(wherePhrase));

        char path[Path::NewsDataPathLengthMax + 1] = {};
        Path::MakeNewsDataPath(path, sizeof (path), record);

        NN_RESULT_TRY(nn::fs::DeleteFile(path))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
            }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }

    nn::Result DeleteOldNewsIfRecordFull() NN_NOEXCEPT
    {
        while (NN_STATIC_CONDITION(1))
        {
            int count;
            NN_RESULT_DO(NewsDatabase::GetInstance().Count(&count, ""));

            // インポート前に削除するため、最大件数貯まっていたら削除する。
            if (count < DataCountMax)
            {
                break;
            }

            NN_RESULT_DO(DeleteOldNews());
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result DeleteOldNewsIfNotEnoughSpace(int64_t requiredSize) NN_NOEXCEPT
    {
        while (NN_STATIC_CONDITION(1))
        {
            int64_t freeSpaceSize;
            NN_RESULT_DO(FileSystem::GetFreeSpaceSize(&freeSpaceSize, NN_DETAIL_NEWS_DATA_MOUNT_NAME));

            NN_DETAIL_NEWS_INFO("[news] Free space size = %lld, required size = %lld.\n", freeSpaceSize, requiredSize);

            if (requiredSize < freeSpaceSize)
            {
                break;
            }

            NN_RESULT_DO(DeleteOldNews());
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result CleanupAndMakeFreeSpace(int64_t requiredSize) NN_NOEXCEPT
    {
        NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

        NN_RESULT_DO(DeleteExpiredNews());
        NN_RESULT_DO(DeleteOldNewsIfRecordFull());
        NN_RESULT_DO(DeleteOldNewsIfNotEnoughSpace(requiredSize));

        // データ保存ストレージのコミットを先に行うこと。
        FileSystem::Commit(NN_DETAIL_NEWS_DATA_MOUNT_NAME);
        // システム情報ストレージのコミット前に電源断が発生した場合、起動時のリカバリー処理でレコードが削除される。
        FileSystem::Commit(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME);

        NN_RESULT_SUCCESS;
    }

    nn::Result ImportRecord(NewsRecord* outRecord,
        nne::nlib::InputStream& stream, bool isLocal,
        bool isIndividual, uint64_t userId, int8_t language, uint64_t dataId, bool isOverwrite, bool isTestDistribution) NN_NOEXCEPT
    {
        NewsDatabase::InsertRecord insertRecord = {};
        NewsMetadataReader reader;

        NN_RESULT_DO(reader.Read(&insertRecord, stream, isLocal));

        bool isNewly = true;

        if (insertRecord.newsId.value[1] == 'S')
        {
            ReceivedHistoryManager::SearchKey key = {};
            bool isImportable = false;
            bool isOverwritable = false;

            key.isIndividual = isIndividual;
            key.userId = userId;
            key.domain = insertRecord.newsId.value[0];
            key.newsId = std::strtoull(&insertRecord.newsId.value[2], nullptr, 10);
            key.language = language;
            key.dataId = dataId;

            NN_RESULT_DO(ReceivedHistoryManager::GetInstance().IsImportable(&isImportable, key));

            if (isOverwrite)
            {
                NN_RESULT_DO(ReceivedHistoryManager::GetInstance().IsOverwritable(&isOverwritable, key));
            }

            if (isImportable || isOverwritable || isTestDistribution)
            {
                isNewly = isImportable;
            }
            else
            {
                NN_DETAIL_NEWS_INFO("[news] The news (id = %s) is already imported.\n", insertRecord.newsId.value);
                NN_RESULT_THROW(ResultAlreadyImported());
            }

            if (!isImportable && !isTestDistribution)
            {
                NewsRecord record = {};
                int count;

                char wherePhrase[64] = {};
                nn::util::SNPrintf(wherePhrase, sizeof (wherePhrase), "news_id = '%s'", insertRecord.newsId.value);

                NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

                // DB にレコードがあるかどうかを確認する。
                NN_RESULT_DO(NewsDatabase::GetInstance().GetList(&count, &record, wherePhrase, "", 0, 1));

                if (count == 0)
                {
                    NN_DETAIL_NEWS_INFO("[news] %s is already deleted.\n", insertRecord.newsId.value);
                    NN_RESULT_THROW(ResultAlreadyDeleted());
                }
            }
        }

        NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

        NN_RESULT_TRY(nn::time::StandardNetworkSystemClock::GetCurrentTime(&insertRecord.receivedAt))
            NN_RESULT_CATCH(nn::time::ResultClockInvalid)
            {
                NewsRecord record;
                int count;

                // ネットワーク時刻が取得できなかった場合、DB から一番新しい受信時刻を取得し、+1 した値を採用する。
                // DB にもレコードが存在しない場合、1970-01-01 00:00:00 を初期値とする。
                if (NewsDatabase::GetInstance().GetList(&count, &record, "", "received_at DESC", 0, 1).IsSuccess() && count == 1)
                {
                    insertRecord.receivedAt.value = record.receivedTime.value + 1;
                }
                else
                {
                    // 1970-01-01 00:00:00
                    insertRecord.receivedAt.value = 0;
                }

                NN_DETAIL_NEWS_INFO("[news] The network time is invalid. receivedTime = %lld\n", insertRecord.receivedAt.value);
            }
        NN_RESULT_END_TRY;

        if (isIndividual)
        {
            nn::util::SNPrintf(insertRecord.userId.value, sizeof (insertRecord.userId.value), "%llu", userId);
        }

        if (isNewly)
        {
            insertRecord.read = 0;
            insertRecord.newly = 1;
            insertRecord.displayed = 0;
            insertRecord.optedIn = 1;
            insertRecord.pointStatus = 0;
            insertRecord.extra1 = 0;
            insertRecord.extra2 = 0;

            NN_RESULT_DO(NewsDatabase::GetInstance().Insert(insertRecord));
        }
        else
        {
            NN_RESULT_DO(NewsDatabase::GetInstance().UpdateInsertedRecord(insertRecord));
        }

        FileSystem::Commit(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME);

        nn::util::Strlcpy(outRecord->newsId.value, insertRecord.newsId.value, sizeof (insertRecord.newsId.value));
        nn::util::Strlcpy(outRecord->userId.value, insertRecord.userId.value, sizeof (insertRecord.userId.value));

        outRecord->receivedTime = insertRecord.receivedAt;

        NN_RESULT_SUCCESS;
    }

    nn::Result CreateFile(nn::fs::FileHandle* outHandle, const NewsRecord& record, size_t size) NN_NOEXCEPT
    {
        char path[Path::NewsDataPathLengthMax + 1] = {};
        Path::MakeNewsDataPath(path, sizeof (path), record);

        NN_RESULT_TRY(nn::fs::DeleteFile(path))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
            }
        NN_RESULT_END_TRY;

        NN_RESULT_DO(FileSystem::CreateFile(path, size, false));

        NN_RESULT_DO(nn::fs::OpenFile(outHandle, path, nn::fs::OpenMode_Write));

        NN_RESULT_SUCCESS;
    }

    nn::Result SaveNewsData(const NewsRecord& record, const void* data, size_t dataSize) NN_NOEXCEPT
    {
        nn::fs::FileHandle handle = {};
        NN_RESULT_DO(CreateFile(&handle, record, dataSize));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_RESULT_DO(nn::fs::WriteFile(handle, 0, data, dataSize,
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

        NN_RESULT_SUCCESS;
    }

    nn::Result SaveNewsData(const NewsRecord& record, const char* dataPath, size_t dataSize, void* work, size_t workSize) NN_NOEXCEPT
    {
        nn::fs::FileHandle srcHandle = {};
        NN_RESULT_DO(nn::fs::OpenFile(&srcHandle, dataPath, nn::fs::OpenMode_Read));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(srcHandle);
        };

        nn::fs::FileHandle destHandle = {};
        NN_RESULT_DO(CreateFile(&destHandle, record, dataSize));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(destHandle);
        };

        int64_t offset = 0;

        while (NN_STATIC_CONDITION(1))
        {
            size_t read = 0;
            NN_RESULT_DO(nn::fs::ReadFile(&read, srcHandle, offset, work, workSize));

            if (read == 0)
            {
                break;
            }

            NN_RESULT_DO(nn::fs::WriteFile(destHandle, offset, work, read, nn::fs::WriteOption::MakeValue(0)));

            offset += static_cast<int64_t>(read);
        }

        NN_RESULT_DO(nn::fs::FlushFile(destHandle));

        NN_RESULT_SUCCESS;
    }
}

nn::Result NewsImporter::ImportFromLocal(const void* data, size_t dataSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(data);
    NN_SDK_REQUIRES(dataSize > 0);

    NN_RESULT_TRY(CleanupAndMakeFreeSpace(static_cast<int64_t>(dataSize)))
        NN_RESULT_CATCH_ALL
        {
            FileSystem::Rollback(NN_DETAIL_NEWS_DATA_MOUNT_NAME);
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    nne::nlib::MemoryInputStream stream;
    stream.Init(data, dataSize);

    NN_UTIL_SCOPE_EXIT
    {
        stream.Close();
    };

    uint64_t dataId = 0;
    NN_RESULT_DO(IncrementalId::GetInstance().Issue(&dataId));

    NewsRecord record;

    NN_RESULT_TRY(ImportRecord(&record, stream, true, false, 0, 0, dataId, true, false))
        NN_RESULT_CATCH(ResultAlreadyImported)
        {
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_CATCH(ResultAlreadyDeleted)
        {
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    // 電源断時の復帰処理テスト用
    // NN_DETAIL_NEWS_INFO("[new] Sleep 5 seconds for recovery test...\n");
    // nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    NN_RESULT_TRY(SaveNewsData(record, data, dataSize))
        NN_RESULT_CATCH_ALL
        {
            FileSystem::Rollback(NN_DETAIL_NEWS_DATA_MOUNT_NAME);
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    FileSystem::Commit(NN_DETAIL_NEWS_DATA_MOUNT_NAME);

    // 自動発行の場合、受信履歴に保存しない。
    if (record.newsId.value[1] == 'S')
    {
        ReceivedHistoryManager::Record historyRecord = {};

        historyRecord.isIndividual = false;
        historyRecord.userId = 0;
        historyRecord.domain = record.newsId.value[0];
        historyRecord.newsId = std::strtoull(&record.newsId.value[2], nullptr, 10);
        historyRecord.language = 0;
        historyRecord.dataId = dataId;
        historyRecord.receivedTime = record.receivedTime;

        NN_RESULT_DO(ReceivedHistoryManager::GetInstance().AddOrUpdate(historyRecord));
    }

    NewlyArrivedEventListener::GetInstance().SignalAll();

    NN_RESULT_SUCCESS;
}

nn::Result NewsImporter::ImportFromNetwork(const char* path,
    bool isIndividual, uint64_t userId, int8_t language, uint64_t dataId, bool isOverwrite, bool isTestDistribution,
    void* work, size_t workSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    int64_t dataSize;
    NN_RESULT_DO(FileSystem::GetFileSize(&dataSize, path));

    NN_RESULT_TRY(CleanupAndMakeFreeSpace(dataSize))
        NN_RESULT_CATCH_ALL
        {
            FileSystem::Rollback(NN_DETAIL_NEWS_DATA_MOUNT_NAME);
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    detail::service::msgpack::FileInputStream stream;
    stream.SetBuffer(work, workSize);

    NN_RESULT_DO(stream.Open(path));

    NN_UTIL_SCOPE_EXIT
    {
        stream.Close();
    };

    NewsRecord record;

    NN_RESULT_TRY(ImportRecord(&record, stream, false, isIndividual, userId, language, dataId, isOverwrite, isTestDistribution))
        NN_RESULT_CATCH(ResultAlreadyImported)
        {
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_CATCH(ResultAlreadyDeleted)
        {
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    // 電源断時の復帰処理テスト用
    // NN_DETAIL_NEWS_INFO("[new] Sleep 5 seconds for recovery test...\n");
    // nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    NN_RESULT_TRY(SaveNewsData(record, path, static_cast<size_t>(dataSize), work, workSize))
        NN_RESULT_CATCH_ALL
        {
            FileSystem::Rollback(NN_DETAIL_NEWS_DATA_MOUNT_NAME);
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    FileSystem::Commit(NN_DETAIL_NEWS_DATA_MOUNT_NAME);

    // 自動発行の場合、受信履歴に保存しない。
    if (record.newsId.value[1] == 'S')
    {
        ReceivedHistoryManager::Record historyRecord = {};

        historyRecord.isIndividual = isIndividual;
        historyRecord.userId = userId;
        historyRecord.domain = record.newsId.value[0];
        historyRecord.newsId = std::strtoull(&record.newsId.value[2], nullptr, 10);
        historyRecord.language = language;
        historyRecord.dataId = dataId;
        historyRecord.receivedTime = record.receivedTime;

        NN_RESULT_DO(ReceivedHistoryManager::GetInstance().AddOrUpdate(historyRecord));
    }

    if (isOverwrite)
    {
        OverwriteEventListener::GetInstance().SignalAll();
    }
    else
    {
        NewlyArrivedEventListener::GetInstance().SignalAll();
    }

    NN_RESULT_SUCCESS;
}

}}}}}
