﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/service/core/news_NewsDatabase.h>
#include <nn/news/detail/service/util/news_Sqlite.h>
#include <sqlite3_nintendo_custom.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

namespace
{
    const char* FilePath = "news-sys:/news.db";
    const char* TempFilePath = "news-sys:/news.temp.db";

    const size_t QueryStringSize = 4096;

    const char* CreateTableQuery =
        "CREATE TABLE IF NOT EXISTS news "
        "("
            "news_id            TEXT,"      // [NS|NA|LS|LA]<uint64>
            "user_id            TEXT,"      // <uint64>
            "topic_id           TEXT,"      //
            "application_ids    TEXT,"      // /%016llx/%016llx/%016llx/%016llx/%016llx/
            "received_at        INTEGER,"   //
            "published_at       INTEGER,"   //
            "expire_at          INTEGER,"   //
            "pickup_limit       INTEGER,"   //
            "priority           INTEGER,"   //
            "deletion_priority  INTEGER,"   //
            "age_limit          INTEGER,"   //
            "surprise           INTEGER,"   //
            "bashotorya         INTEGER,"   //
            "point              INTEGER,"   //
            "read               INTEGER,"   //
            "newly              INTEGER,"   //
            "displayed          INTEGER,"   //
            "opted_in           INTEGER,"   //
            "point_status       INTEGER,"   //
            "extra_1            INTEGER,"   //
            "extra_2            INTEGER,"   //
            "UNIQUE(news_id, user_id)"
        ")";

    const char* InsertQuery =
        "INSERT INTO news "
        "("
            "news_id,"
            "user_id,"
            "topic_id,"
            "application_ids,"
            "received_at,"
            "published_at,"
            "expire_at,"
            "pickup_limit,"
            "priority,"
            "deletion_priority,"
            "age_limit,"
            "surprise,"
            "bashotorya,"
            "point,"
            "read,"
            "newly,"
            "displayed,"
            "opted_in,"
            "point_status,"
            "extra_1,"
            "extra_2"
        ")"
        " VALUES "
        "("
            "?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?"
        ")";

    const char* UpdateInsertedRecordQueryFormat =
        "UPDATE news SET "
            "topic_id = '%s',"
            "application_ids = '%s',"
            "published_at = %lld,"
            "expire_at = %lld,"
            "pickup_limit = %lld,"
            "priority = %d,"
            "deletion_priority = %d,"
            "age_limit = %d,"
            "surprise = %d,"
            "bashotorya = %d,"
            "point = %d"
        " WHERE news_id = '%s' AND user_id = '%s'";

    // 利用可能なキー
    const char* UsableKeys[] =
    {
        "news_id",
        "user_id",
        "topic_id",
        "application_ids",
        "received_at",
        "published_at",
        "expire_at",
        "pickup_limit",
        "priority",
        "deletion_priority",
        "age_limit",
        "surprise",
        "bashotorya",
        "point",
        "read",
        "newly",
        "displayed",
        "opted_in",
        "point_status",
        "extra_1",
        "extra_2",
        ""
    };

    // 更新可能なキー
    const char* UpdatableKeyForInteger[] =
    {
        "priority",
        "read",
        "newly",
        "displayed",
        "opted_in",
        "point_status",
        "extra_1",
        "extra_2",
        ""
    };

    // 更新可能なキー
    const char* UpdatableKeyForString[] =
    {
        ""
    };

    // 利用可能な関数
    const char* UsableFunctions[] =
    {
        "ABS",
        "N_IF",
        "N_SWITCH",
        "N_RAND",
        "N_PMATCH",
        ""
    };

    // 利用可能な演算
    const char* UsableOperationKeywords[] =
    {
        "AND",
        "OR",
        "NOT",
        ""
    };

    // 利用可能なソート種別
    const char* UsableSortOrderKeywords[] =
    {
        "ASC",
        "DESC",
        ""
    };
}

namespace
{
    struct TokenAnalysisContext
    {
        const char* current;
        const char* token;
        int32_t tokenLength;
        int32_t tokenCount;
        int32_t analyzedLength;
        int32_t bracket;
        bool isValid;
        bool isTerminated;
    };

    typedef bool (*WordInspector)(const char* word, size_t length);
}

namespace
{
    bool IsAlpha(char c) NN_NOEXCEPT
    {
        return ((c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z'));
    }

    bool IsDigit(char c) NN_NOEXCEPT
    {
        return (c >= '0' && c <= '9');
    }

    bool IsSpace(char c) NN_NOEXCEPT
    {
        return (c == ' ' || c == '\t');
    }

    bool IsQuote(char c) NN_NOEXCEPT
    {
        return (c == '\'');
    }

    bool IsSymbolToken(char c) NN_NOEXCEPT
    {
        return (c == '(' || c == ')' || c == ',' || c == '+' || c == '-' || c == '*' || c == '/');
    }

    bool IncludesComparisonOperator(char c) NN_NOEXCEPT
    {
        return (c == '=' || c == '<' || c == '>' || c == '!');
    }

     bool IncludesKeywordChar(char c) NN_NOEXCEPT
    {
        return (IsAlpha(c) || IsDigit(c) || c == '_');
    }

    bool IncludesStringValueChar(char c) NN_NOEXCEPT
    {
        return (IsAlpha(c) || IsDigit(c) || c == ' ' || c == '_' || c == '/');
    }

    bool FindWord(const char* word, size_t length, const char* words[]) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(word);
        NN_SDK_REQUIRES_NOT_NULL(words);

        for (int i = 0; words[i][0] != '\0'; i++)
        {
            if (length == static_cast<size_t>(nn::util::Strnlen(words[i], KeyLengthMax + 1)) &&
                nn::util::Strncmp(word, words[i], static_cast<int>(length)) == 0)
            {
                return true;
            }
        }

        return false;
    }

    void AnalyzeToken(TokenAnalysisContext* pContext, WordInspector wordInspector) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pContext);
        NN_SDK_REQUIRES_NOT_NULL(wordInspector);

        if (pContext->isTerminated)
        {
            return;
        }

        while (IsSpace(*pContext->current))
        {
            pContext->current++;
            pContext->analyzedLength++;
        }

        if (*pContext->current == '\0')
        {
            pContext->isValid = (pContext->bracket == 0);
            pContext->isTerminated = true;
            return;
        }

        pContext->token = pContext->current;
        pContext->isValid = false;

        if (IsAlpha(*pContext->current))
        {
            do
            {
                pContext->current++;
            }
            while (IncludesKeywordChar(*pContext->current));

            if (wordInspector(pContext->token, pContext->current - pContext->token))
            {
                pContext->isValid = true;
            }
        }
        else if (IsDigit(*pContext->current))
        {
            do
            {
                pContext->current++;
            }
            while (IsDigit(*pContext->current));

            pContext->isValid = true;
        }
        else if (IsQuote(*pContext->current))
        {
            do
            {
                pContext->current++;
            }
            while (IncludesStringValueChar(*pContext->current));

            if (IsQuote(*pContext->current++))
            {
                pContext->isValid = true;
            }
        }
        else if (IsSymbolToken(*pContext->current))
        {
            if (*pContext->current == '(')
            {
                pContext->bracket++;
            }
            else if (*pContext->current == ')')
            {
                pContext->bracket--;
            }
            pContext->current++;

            if (pContext->bracket >= 0 && pContext->bracket <= NestDepthMax)
            {
                pContext->isValid = true;
            }
        }
        else if (IncludesComparisonOperator(*pContext->current))
        {
            const char* first = pContext->current;

            do
            {
                pContext->current++;
            }
            while (IncludesComparisonOperator(*pContext->current));

            size_t length = pContext->current - first;

            if (length == 1 && (*first == '=' || *first == '<' || *first == '>'))
            {
                pContext->isValid = true;
            }
            else if (length == 2 && ((*first == '!' || *first == '<' || *first == '>') && (*(first + 1) == '=')))
            {
                pContext->isValid = true;
            }
        }

        if (pContext->isValid)
        {
            pContext->tokenLength = static_cast<int32_t>(pContext->current - pContext->token);
            pContext->tokenCount++;

            pContext->analyzedLength += pContext->tokenLength;
        }
        else
        {
            pContext->isTerminated = true;
        }
    } // NOLINT(impl/function_size)

    bool InspectWherePhraseKeyword(const char* word, size_t length) NN_NOEXCEPT
    {
        if (FindWord(word, length, UsableKeys))
        {
            return true;
        }
        if (FindWord(word, length, UsableFunctions))
        {
            return true;
        }
        if (FindWord(word, length, UsableOperationKeywords))
        {
            return true;
        }

        return false;
    }

    bool InspectOrderByPhraseKeyword(const char* word, size_t length) NN_NOEXCEPT
    {
        if (FindWord(word, length, UsableKeys))
        {
            return true;
        }
        if (FindWord(word, length, UsableFunctions))
        {
            return true;
        }
        if (FindWord(word, length, UsableOperationKeywords))
        {
            return true;
        }
        if (FindWord(word, length, UsableSortOrderKeywords))
        {
            return true;
        }

        return false;
    }

    bool Inspect(bool* outIsEmpty, const char* phrase, WordInspector wordInspector) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outIsEmpty);
        NN_SDK_REQUIRES_NOT_NULL(phrase);
        NN_SDK_REQUIRES_NOT_NULL(wordInspector);

        TokenAnalysisContext context = {};
        context.current = phrase;

        while (!context.isTerminated)
        {
            AnalyzeToken(&context, wordInspector);

            if (!context.isValid)
            {
                NN_DETAIL_NEWS_INFO("[news] => \"%s\"\n", phrase);

                for (int32_t i = 0; i < 11 + context.analyzedLength; i++)
                {
                    NN_DETAIL_NEWS_INFO(" ");
                }

                NN_DETAIL_NEWS_INFO("^\n");
            }
        }

        *outIsEmpty = (context.tokenCount == 0);

        return context.isValid;
    }

    nn::Result InspectKey(const char* key) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(FindWord(key, nn::util::Strnlen(key, KeyLengthMax), UsableKeys), ResultUnknownKey());

        NN_RESULT_SUCCESS;
    }

    nn::Result InspectWherePhrase(bool* outIsEmpty, const char* phrase) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(Inspect(outIsEmpty, phrase, InspectWherePhraseKeyword), ResultInvalidWherePhrase());

        NN_RESULT_SUCCESS;
    }

    nn::Result InspectOrderByPhrase(bool* outIsEmpty, const char* phrase) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(Inspect(outIsEmpty, phrase, InspectOrderByPhraseKeyword), ResultInvalidOrderByPhrase());

        NN_RESULT_SUCCESS;
    }

    bool IsUpdatableKey(const char* key, bool isIntegerValue) NN_NOEXCEPT
    {
        if (isIntegerValue)
        {
            return FindWord(key, nn::util::Strnlen(key, KeyLengthMax), UpdatableKeyForInteger);
        }
        else
        {
            return FindWord(key, nn::util::Strnlen(key, KeyLengthMax), UpdatableKeyForString);
        }
    }
}

NewsDatabase::NewsDatabase() NN_NOEXCEPT :
    m_Mutex(true)
{
}

nn::Result NewsDatabase::GetList(int* outCount, NewsRecord* outRecords,
    const char* wherePhrase, const char* orderByPhrase, int offset, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outRecords);
    NN_SDK_REQUIRES_NOT_NULL(wherePhrase);
    NN_SDK_REQUIRES_NOT_NULL(orderByPhrase);
    NN_SDK_REQUIRES(offset >= 0);
    NN_SDK_REQUIRES(count > 0);

    bool isWherePhraseEmpty = false;
    NN_RESULT_DO(InspectWherePhrase(&isWherePhraseEmpty, wherePhrase));

    bool isOrderByPhraseEmpty = false;
    NN_RESULT_DO(InspectOrderByPhrase(&isOrderByPhraseEmpty, orderByPhrase));

    NN_UTIL_LOCK_GUARD(m_Mutex);

    sqlite3* pHandle = nullptr;
    sqlite3_stmt* pStatement = nullptr;

    // 現時点では、毎回開いて閉じる。
    NN_RESULT_DO(Open(&pHandle));

    NN_UTIL_SCOPE_EXIT
    {
        if (pStatement)
        {
            sqlite3_finalize(pStatement);
        }
        Close(pHandle);
    };

    char query[QueryStringSize] = {};

    int length = nn::util::SNPrintf(query, sizeof (query),
        "SELECT news_id, user_id, topic_id, received_at, read, newly, displayed, extra_1, extra_2 FROM news %s%s%s%s%s LIMIT %d, %d",
        isWherePhraseEmpty ? "" : "WHERE ", wherePhrase, isWherePhraseEmpty ? "" : " ",
        isOrderByPhraseEmpty ? "" : "ORDER BY ", orderByPhrase, offset, count);

    NN_DETAIL_NEWS_INFO("[news] Query: %s\n", query);

    SQLITE_DO(sqlite3_prepare_v2(pHandle, query, length, &pStatement, nullptr));

    int result = SQLITE_OK;

    int actualCount = 0;

    do
    {
        result = sqlite3_step(pStatement);

        if (result == SQLITE_ROW)
        {
            NewsRecord& record = outRecords[actualCount];
            int index = 0;

            nn::util::Strlcpy(record.newsId.value, reinterpret_cast<const char*>(sqlite3_column_text(pStatement, index++)),
                sizeof (record.newsId.value));
            nn::util::Strlcpy(record.userId.value, reinterpret_cast<const char*>(sqlite3_column_text(pStatement, index++)),
                sizeof (record.userId.value));
            nn::util::Strlcpy(record.topicId.value, reinterpret_cast<const char*>(sqlite3_column_text(pStatement, index++)),
                sizeof (record.topicId.value));

            record.receivedTime.value = sqlite3_column_int64(pStatement, index++);
            record.read               = sqlite3_column_int(pStatement, index++);
            record.newly              = sqlite3_column_int(pStatement, index++);
            record.displayed          = sqlite3_column_int(pStatement, index++);
            record.extra1             = sqlite3_column_int(pStatement, index++);
            record.extra2             = sqlite3_column_int(pStatement, index++);

            actualCount++;
        }
        else
        {
            SQLITE_DO(result);
        }
    }
    while (result == SQLITE_ROW);

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result NewsDatabase::GetList(int* outCount, NewsRecordV1* outRecords,
    const char* wherePhrase, const char* orderByPhrase, int offset, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outRecords);
    NN_SDK_REQUIRES_NOT_NULL(wherePhrase);
    NN_SDK_REQUIRES_NOT_NULL(orderByPhrase);
    NN_SDK_REQUIRES(offset >= 0);
    NN_SDK_REQUIRES(count > 0);

    bool isWherePhraseEmpty = false;
    NN_RESULT_DO(InspectWherePhrase(&isWherePhraseEmpty, wherePhrase));

    bool isOrderByPhraseEmpty = false;
    NN_RESULT_DO(InspectOrderByPhrase(&isOrderByPhraseEmpty, orderByPhrase));

    NN_UTIL_LOCK_GUARD(m_Mutex);

    sqlite3* pHandle = nullptr;
    sqlite3_stmt* pStatement = nullptr;

    // 現時点では、毎回開いて閉じる。
    NN_RESULT_DO(Open(&pHandle));

    NN_UTIL_SCOPE_EXIT
    {
        if (pStatement)
        {
            sqlite3_finalize(pStatement);
        }
        Close(pHandle);
    };

    char query[QueryStringSize] = {};

    int length = nn::util::SNPrintf(query, sizeof (query),
        "SELECT news_id, user_id, received_at, read, newly, displayed, extra_1 FROM news %s%s%s%s%s LIMIT %d, %d",
        isWherePhraseEmpty ? "" : "WHERE ", wherePhrase, isWherePhraseEmpty ? "" : " ",
        isOrderByPhraseEmpty ? "" : "ORDER BY ", orderByPhrase, offset, count);

    NN_DETAIL_NEWS_INFO("[news] Query: %s\n", query);

    SQLITE_DO(sqlite3_prepare_v2(pHandle, query, length, &pStatement, nullptr));

    int result = SQLITE_OK;

    int actualCount = 0;

    do
    {
        result = sqlite3_step(pStatement);

        if (result == SQLITE_ROW)
        {
            NewsRecordV1& record = outRecords[actualCount];
            int index = 0;

            nn::util::Strlcpy(record.newsId.value, reinterpret_cast<const char*>(sqlite3_column_text(pStatement, index++)),
                sizeof (record.newsId.value));
            nn::util::Strlcpy(record.userId.value, reinterpret_cast<const char*>(sqlite3_column_text(pStatement, index++)),
                sizeof (record.userId.value));

            record.receivedTime.value = sqlite3_column_int64(pStatement, index++);
            record.read               = sqlite3_column_int(pStatement, index++);
            record.newly              = sqlite3_column_int(pStatement, index++);
            record.displayed          = sqlite3_column_int(pStatement, index++);
            record.extra1             = sqlite3_column_int(pStatement, index++);

            actualCount++;
        }
        else
        {
            SQLITE_DO(result);
        }
    }
    while (result == SQLITE_ROW);

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result NewsDatabase::Count(int* outCount, const char* wherePhrase) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(wherePhrase);

    bool isWherePhraseEmpty = false;
    NN_RESULT_DO(InspectWherePhrase(&isWherePhraseEmpty, wherePhrase));

    NN_UTIL_LOCK_GUARD(m_Mutex);

    sqlite3* pHandle = nullptr;
    sqlite3_stmt* pStatement = nullptr;

    // 現時点では、毎回開いて閉じる。
    NN_RESULT_DO(Open(&pHandle));

    NN_UTIL_SCOPE_EXIT
    {
        if (pStatement)
        {
            sqlite3_finalize(pStatement);
        }
        Close(pHandle);
    };

    char query[QueryStringSize] = {};

    int length = nn::util::SNPrintf(query, sizeof (query),
        "SELECT COUNT(*) FROM news %s%s", isWherePhraseEmpty ? "" : "WHERE ", wherePhrase);

    NN_DETAIL_NEWS_INFO("[news] Query: %s\n", query);

    SQLITE_DO(sqlite3_prepare_v2(pHandle, query, length, &pStatement, nullptr));

    int result = SQLITE_OK;

    do
    {
        result = sqlite3_step(pStatement);

        if (result == SQLITE_ROW)
        {
            *outCount = sqlite3_column_int(pStatement, 0);
        }
        else
        {
            SQLITE_DO(result);
        }
    }
    while (result == SQLITE_ROW);

    NN_RESULT_SUCCESS;
}

nn::Result NewsDatabase::Count(int* outCount, bool distinct, const char* key, const char* wherePhrase) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(wherePhrase);

    NN_RESULT_DO(InspectKey(key));

    bool isWherePhraseEmpty = false;
    NN_RESULT_DO(InspectWherePhrase(&isWherePhraseEmpty, wherePhrase));

    NN_UTIL_LOCK_GUARD(m_Mutex);

    sqlite3* pHandle = nullptr;
    sqlite3_stmt* pStatement = nullptr;

    // 現時点では、毎回開いて閉じる。
    NN_RESULT_DO(Open(&pHandle));

    NN_UTIL_SCOPE_EXIT
    {
        if (pStatement)
        {
            sqlite3_finalize(pStatement);
        }
        Close(pHandle);
    };

    char query[QueryStringSize] = {};

    int length = nn::util::SNPrintf(query, sizeof (query),
        "SELECT COUNT(%s%s) FROM news %s%s",
        distinct ? "DISTINCT " : "", key, isWherePhraseEmpty ? "" : "WHERE ", wherePhrase);

    NN_DETAIL_NEWS_INFO("[news] Query: %s\n", query);

    SQLITE_DO(sqlite3_prepare_v2(pHandle, query, length, &pStatement, nullptr));

    int result = SQLITE_OK;

    do
    {
        result = sqlite3_step(pStatement);

        if (result == SQLITE_ROW)
        {
            *outCount = sqlite3_column_int(pStatement, 0);
        }
        else
        {
            SQLITE_DO(result);
        }
    }
    while (result == SQLITE_ROW);

    NN_RESULT_SUCCESS;
}

nn::Result NewsDatabase::Update(const char* key, int32_t newValue, const char* wherePhrase) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(key);
    NN_SDK_REQUIRES_NOT_NULL(wherePhrase);

    NN_RESULT_DO(InspectKey(key));

    NN_RESULT_THROW_UNLESS(IsUpdatableKey(key, true), ResultNotUpdatable());

    bool isWherePhraseEmpty = false;
    NN_RESULT_DO(InspectWherePhrase(&isWherePhraseEmpty, wherePhrase));

    NN_UTIL_LOCK_GUARD(m_Mutex);

    sqlite3* pHandle = nullptr;
    sqlite3_stmt* pStatement = nullptr;

    NN_RESULT_DO(Open(&pHandle));

    NN_UTIL_SCOPE_EXIT
    {
        if (pStatement)
        {
            sqlite3_finalize(pStatement);
        }
        Close(pHandle);
    };

    char query[QueryStringSize] = {};

    int length = nn::util::SNPrintf(query, sizeof (query),
        "UPDATE news SET %s = %d %s%s",
        key, newValue, isWherePhraseEmpty ? "" : "WHERE ", wherePhrase);

    NN_DETAIL_NEWS_INFO("[news] Query: %s\n", query);

    SQLITE_DO(sqlite3_prepare_v2(pHandle, query, length, &pStatement, nullptr));
    SQLITE_DO(sqlite3_step(pStatement));

    NN_RESULT_SUCCESS;
}

nn::Result NewsDatabase::Update(const char* key, const char* newValue, const char* wherePhrase) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(key);
    NN_SDK_REQUIRES_NOT_NULL(newValue);
    NN_SDK_REQUIRES_NOT_NULL(wherePhrase);

    // キーの種類ごとに、適切な検証を行う。

    NN_RESULT_DO(InspectKey(key));

    NN_RESULT_THROW_UNLESS(IsUpdatableKey(key, false), ResultNotUpdatable());

    bool isWherePhraseEmpty = false;
    NN_RESULT_DO(InspectWherePhrase(&isWherePhraseEmpty, wherePhrase));

    NN_UTIL_LOCK_GUARD(m_Mutex);

    sqlite3* pHandle = nullptr;
    sqlite3_stmt* pStatement = nullptr;

    NN_RESULT_DO(Open(&pHandle));

    NN_UTIL_SCOPE_EXIT
    {
        if (pStatement)
        {
            sqlite3_finalize(pStatement);
        }
        Close(pHandle);
    };

    char query[QueryStringSize] = {};

    int length = nn::util::SNPrintf(query, sizeof (query),
        "UPDATE news SET %s = '%s' %s%s",
        key, newValue, isWherePhraseEmpty ? "" : "WHERE ", wherePhrase);

    NN_DETAIL_NEWS_INFO("[news] Query: %s\n", query);

    SQLITE_DO(sqlite3_prepare_v2(pHandle, query, length, &pStatement, nullptr));
    SQLITE_DO(sqlite3_step(pStatement));

    NN_RESULT_SUCCESS;
}

nn::Result NewsDatabase::UpdateWithAddition(const char* key, int32_t value, const char* wherePhrase) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(key);
    NN_SDK_REQUIRES_NOT_NULL(wherePhrase);

    NN_RESULT_DO(InspectKey(key));

    NN_RESULT_THROW_UNLESS(IsUpdatableKey(key, true), ResultNotUpdatable());

    bool isWherePhraseEmpty = false;
    NN_RESULT_DO(InspectWherePhrase(&isWherePhraseEmpty, wherePhrase));

    NN_UTIL_LOCK_GUARD(m_Mutex);

    sqlite3* pHandle = nullptr;
    sqlite3_stmt* pStatement = nullptr;

    NN_RESULT_DO(Open(&pHandle));

    NN_UTIL_SCOPE_EXIT
    {
        if (pStatement)
        {
            sqlite3_finalize(pStatement);
        }
        Close(pHandle);
    };

    char query[QueryStringSize] = {};

    int length = nn::util::SNPrintf(query, sizeof (query),
        "UPDATE news SET %s = %s %c %d %s%s",
        key, key, value >= 0 ? '+' : '-', value >= 0 ? value : -value, isWherePhraseEmpty ? "" : "WHERE ", wherePhrase);

    NN_DETAIL_NEWS_INFO("[news] Query: %s\n", query);

    SQLITE_DO(sqlite3_prepare_v2(pHandle, query, length, &pStatement, nullptr));
    SQLITE_DO(sqlite3_step(pStatement));

    NN_RESULT_SUCCESS;
}

nn::Result NewsDatabase::UpdateInsertedRecord(const InsertRecord& record) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    sqlite3* pHandle = nullptr;
    sqlite3_stmt* pStatement = nullptr;

    NN_RESULT_DO(Open(&pHandle));

    NN_UTIL_SCOPE_EXIT
    {
        if (pStatement)
        {
            sqlite3_finalize(pStatement);
        }
        Close(pHandle);
    };

    // application_ids
    char applicationIds[ApplicationIdCountMax * 17 + 2] = {};

    if (record.applicationIds[0].value)
    {
        int length = nn::util::Strlcpy(applicationIds, "/", sizeof (applicationIds));

        for (int i = 0; i < ApplicationIdCountMax; i++)
        {
            if (record.applicationIds[i].value == 0)
            {
                break;
            }

            length += nn::util::SNPrintf(&applicationIds[length], sizeof (applicationIds) - length,
                "%016llx/", record.applicationIds[i].value);
        }
    }

    char query[QueryStringSize] = {};

    nn::util::SNPrintf(query, sizeof (query), UpdateInsertedRecordQueryFormat,
        record.topicId.value,
        applicationIds,
        record.publishedAt.value,
        record.expireAt.value,
        record.pickupLimit.value,
        record.priority,
        record.deletionPriority,
        record.ageLimit,
        record.surprise,
        record.bashotorya,
        record.point,
        record.newsId.value, record.userId.value);

    NN_DETAIL_NEWS_INFO("[news] Query: %s\n", query);

    SQLITE_DO(sqlite3_prepare_v2(pHandle, query, -1, &pStatement, nullptr));
    SQLITE_DO(sqlite3_step(pStatement));

    NN_RESULT_SUCCESS;
}

nn::Result NewsDatabase::Insert(const InsertRecord& record) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    sqlite3* pHandle = nullptr;
    sqlite3_stmt* pStatement = nullptr;

    NN_RESULT_DO(Open(&pHandle));

    NN_UTIL_SCOPE_EXIT
    {
        if (pStatement)
        {
            sqlite3_finalize(pStatement);
        }
        Close(pHandle);
    };

    SQLITE_DO(sqlite3_prepare_v2(pHandle, InsertQuery, -1, &pStatement, nullptr));

    int index = 1;

    // news_id
    SQLITE_DO(sqlite3_bind_text(pStatement, index++, record.newsId.value, -1, SQLITE_STATIC));
    // user_id
    SQLITE_DO(sqlite3_bind_text(pStatement, index++, record.userId.value, -1, SQLITE_STATIC));
    // topic_id
    SQLITE_DO(sqlite3_bind_text(pStatement, index++, record.topicId.value, -1, SQLITE_STATIC));

    // application_ids
    if (record.applicationIds[0].value)
    {
        char applicationIds[ApplicationIdCountMax * 17 + 2] = {};

        int length = nn::util::Strlcpy(applicationIds, "/", sizeof (applicationIds));

        for (int i = 0; i < ApplicationIdCountMax; i++)
        {
            if (record.applicationIds[i].value == 0)
            {
                break;
            }

            length += nn::util::SNPrintf(&applicationIds[length], sizeof (applicationIds) - length,
                "%016llx/", record.applicationIds[i].value);
        }

        SQLITE_DO(sqlite3_bind_text(pStatement, index++, applicationIds, length, SQLITE_STATIC));
    }
    else
    {
        SQLITE_DO(sqlite3_bind_text(pStatement, index++, "", 0, SQLITE_STATIC));
    }

    // received_at
    SQLITE_DO(sqlite3_bind_int64(pStatement, index++, record.receivedAt.value));
    // published_at
    SQLITE_DO(sqlite3_bind_int64(pStatement, index++, record.publishedAt.value));
    // expire_at
    SQLITE_DO(sqlite3_bind_int64(pStatement, index++, record.expireAt.value));
    // pickup_limit
    SQLITE_DO(sqlite3_bind_int64(pStatement, index++, record.pickupLimit.value));
    // priority
    SQLITE_DO(sqlite3_bind_int(pStatement, index++, record.priority));
    // deletion_priority
    SQLITE_DO(sqlite3_bind_int(pStatement, index++, record.deletionPriority));
    // age_limit
    SQLITE_DO(sqlite3_bind_int(pStatement, index++, record.ageLimit));
    // surprise
    SQLITE_DO(sqlite3_bind_int(pStatement, index++, record.surprise));
    // bashotorya
    SQLITE_DO(sqlite3_bind_int(pStatement, index++, record.bashotorya));
    // point
    SQLITE_DO(sqlite3_bind_int(pStatement, index++, record.point));
    // read
    SQLITE_DO(sqlite3_bind_int(pStatement, index++, record.read));
    // newly
    SQLITE_DO(sqlite3_bind_int(pStatement, index++, record.newly));
    // displayed
    SQLITE_DO(sqlite3_bind_int(pStatement, index++, record.displayed));
    // opted_in
    SQLITE_DO(sqlite3_bind_int(pStatement, index++, record.optedIn));
    // point_status
    SQLITE_DO(sqlite3_bind_int(pStatement, index++, record.pointStatus));
    // extra_1
    SQLITE_DO(sqlite3_bind_int(pStatement, index++, record.extra1));
    // extra_2
    SQLITE_DO(sqlite3_bind_int(pStatement, index++, record.extra2));

    SQLITE_DO(sqlite3_step(pStatement));

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

nn::Result NewsDatabase::Delete(const char* wherePhrase) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(wherePhrase);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    sqlite3* pHandle = nullptr;
    sqlite3_stmt* pStatement = nullptr;

    NN_RESULT_DO(Open(&pHandle));

    NN_UTIL_SCOPE_EXIT
    {
        if (pStatement)
        {
            sqlite3_finalize(pStatement);
        }
        Close(pHandle);
    };

    char query[QueryStringSize] = {};

    int length = nn::util::SNPrintf(query, sizeof (query), "DELETE FROM news WHERE %s", wherePhrase);

    NN_DETAIL_NEWS_INFO("[news] Query: %s\n", query);

    SQLITE_DO(sqlite3_prepare_v2(pHandle, query, length, &pStatement, nullptr));
    SQLITE_DO(sqlite3_step(pStatement));

    NN_RESULT_SUCCESS;
}

nn::Result NewsDatabase::DeleteAll() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_TRY(nn::fs::DeleteFile(FilePath))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result NewsDatabase::Shrink(bool* outIsCommitRequired) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIsCommitRequired);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    int pageSize = 0;
    NN_RESULT_DO(GetPageSize(&pageSize));

    if (pageSize <= 1024)
    {
        *outIsCommitRequired = false;
        NN_RESULT_SUCCESS;
    }

    NN_DETAIL_NEWS_INFO("[news] The page size is larger than 1024. Shrinking '%s'...\n", FilePath);

    NN_RESULT_TRY(nn::fs::DeleteFile(TempFilePath))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    {
        // DB の再生成には 512KB ほど必要。
        // SQLITE 用のヒープを小さくする時は、グローバルヒープから取得するように変更する。
        NN_RESULT_DO(detail::service::util::Sqlite::SetupAllocator());

        SQLITE_DO(sqlite3_initialize());

        sqlite3* pHandle = nullptr;

        SQLITE_DO(sqlite3_open(TempFilePath, &pHandle));
        SQLITE_DO(sqlite3_exec(pHandle, "PRAGMA journal_mode = OFF", nullptr, nullptr, nullptr));

        NN_UTIL_SCOPE_EXIT
        {
            sqlite3_close(pHandle);
            sqlite3_shutdown();
        };

        SQLITE_DO(sqlite3_exec(pHandle, CreateTableQuery, nullptr, nullptr, nullptr));

        char attachDbQuery[64] = {};
        nn::util::SNPrintf(attachDbQuery, sizeof (attachDbQuery), "ATTACH DATABASE '%s' AS old", FilePath);

        SQLITE_DO(sqlite3_exec(pHandle, attachDbQuery, nullptr, nullptr, nullptr));
        SQLITE_DO(sqlite3_exec(pHandle, "INSERT INTO news SELECT * FROM old.news;", nullptr, nullptr, nullptr));
        SQLITE_DO(sqlite3_exec(pHandle, "DETACH DATABASE old;", nullptr, nullptr, nullptr));

        NN_RESULT_TRY(nn::fs::DeleteFile(FilePath))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
            }
        NN_RESULT_END_TRY;
    }

    NN_RESULT_DO(nn::fs::RenameFile(TempFilePath, FilePath));

    *outIsCommitRequired = true;

    NN_RESULT_SUCCESS;
}

nn::Result NewsDatabase::GetDump(size_t* outSize, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    nn::fs::FileHandle handle = {};

    NN_RESULT_TRY(nn::fs::OpenFile(&handle, FilePath, nn::fs::OpenMode_Read))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_THROW(ResultNotFound());
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::ReadFile(outSize, handle, 0, buffer, size));

    NN_RESULT_SUCCESS;
}

nn::Result NewsDatabase::Open(sqlite3** outHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outHandle);

    NN_RESULT_DO(detail::service::util::Sqlite::SetupAllocator());

    SQLITE_DO(sqlite3_initialize());

    SQLITE_DO(sqlite3_open(FilePath, outHandle));

    bool isCompleted = false;

    NN_UTIL_SCOPE_EXIT
    {
        if (!isCompleted)
        {
            Close(*outHandle);
        }
    };

    NN_RESULT_DO(detail::service::util::Sqlite::SetPerformanceImprovementParameters(*outHandle));
    NN_RESULT_DO(detail::service::util::Sqlite::RegisterCustomFunctions(*outHandle));

    SQLITE_DO(sqlite3_exec(*outHandle, CreateTableQuery, nullptr, nullptr, nullptr));

    isCompleted = true;

    NN_RESULT_SUCCESS;
}

void NewsDatabase::Close(sqlite3* pHandle) NN_NOEXCEPT
{
    sqlite3_close(pHandle);

/*
    int64_t used = sqlite3_memory_used();
    int64_t highwater = sqlite3_memory_highwater(true);

    NN_DETAIL_NEWS_INFO("[news] NewsDatabase::Close(). memory: used=%lld, highwater=%lld\n", used, highwater);
*/

    sqlite3_shutdown();
}

nn::Result NewsDatabase::GetPageSize(int* outPageSize) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::service::util::Sqlite::SetupAllocator());

    sqlite3* pHandle = nullptr;

    SQLITE_DO(sqlite3_initialize());

    SQLITE_DO(sqlite3_open(FilePath, &pHandle));

    NN_UTIL_SCOPE_EXIT
    {
        sqlite3_close(pHandle);
        sqlite3_shutdown();
    };

    NN_RESULT_DO(detail::service::util::Sqlite::GetPageSize(outPageSize, pHandle));

    NN_RESULT_SUCCESS;
}

}}}}}
