﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/service/core/news_FileSystem.h>
#include <nn/news/detail/service/core/news_FileSystemConfig.h>
#include <nn/news/detail/service/news_Common.h>
#include <nn/fs/fs_FileSystemPrivate.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

namespace
{
    struct StorageParam
    {
        const char* mountName;
        nn::fs::SystemSaveDataId saveDataId;
        int64_t storageSize;
        int64_t storageJournalSize;
    };

    const StorageParam g_StorageParams[3] =
    {
        {NN_DETAIL_NEWS_DATA_MOUNT_NAME, 0x8000000000000090, DataStorageSize, DataStorageJournalSize},
        {NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME, 0x8000000000000091, SystemStorageSize, SystemStorageJournalSize},
        {NN_DETAIL_NEWS_DOWNLOAD_MOUNT_NAME, 0x8000000000000092, DownloadStorageSize, DownloadStorageJournalSize}
    };

    nn::os::SdkMutexType g_Mutexes[3] =
    {
        NN_OS_SDK_MUTEX_INITIALIZER(), NN_OS_SDK_MUTEX_INITIALIZER(), NN_OS_SDK_MUTEX_INITIALIZER()
    };
}

namespace
{
    bool g_IsTestMode = false;
}

namespace
{
    nn::Result MountSystemSaveData(const char* name, nn::fs::SystemSaveDataId id, int64_t size, int64_t journalSize) NN_NOEXCEPT
    {
        NN_RESULT_TRY(nn::fs::MountSystemSaveData(name, id))
            NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
            {
                NN_RESULT_DO(nn::fs::CreateSystemSaveData(id, size, journalSize, 0));
                NN_RESULT_DO(nn::fs::MountSystemSaveData(name, id));

                NN_DETAIL_NEWS_INFO("[news] A new system save data (%s) is created.\n", name);
            }
            NN_RESULT_CATCH_ALL
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);
            }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }

    int GetStorageIndex(const char* mountName) NN_NOEXCEPT
    {
        for (int i = 0; i < 3; i++)
        {
            if (nn::util::Strncmp(mountName, g_StorageParams[i].mountName, nn::fs::MountNameLengthMax + 1) == 0)
            {
                return i;
            }
        }

        NN_ABORT("");

        return -1;
    }
}

void FileSystem::EnableTestMode() NN_NOEXCEPT
{
    g_IsTestMode = true;
}

nn::Result FileSystem::Mount(const char* mountName, bool isLock) NN_NOEXCEPT
{
    if (g_IsTestMode)
    {
        NN_RESULT_SUCCESS;
    }

    int index = GetStorageIndex(mountName);

    if (isLock)
    {
        g_Mutexes[index].lock();
    }

    NN_RESULT_DO(MountSystemSaveData(g_StorageParams[index].mountName, g_StorageParams[index].saveDataId,
        g_StorageParams[index].storageSize, g_StorageParams[index].storageJournalSize));

    NN_RESULT_SUCCESS;
}

void FileSystem::Unmount(const char* mountName, bool isLocked) NN_NOEXCEPT
{
    if (g_IsTestMode)
    {
        return;
    }

    int index = GetStorageIndex(mountName);

    if (isLocked)
    {
        NN_SDK_ASSERT(g_Mutexes[index].IsLockedByCurrentThread());
    }

    nn::fs::Unmount(mountName);

    if (isLocked)
    {
        g_Mutexes[index].unlock();
    }
}

void FileSystem::Lock(const char* mountName) NN_NOEXCEPT
{
    int index = GetStorageIndex(mountName);

    g_Mutexes[index].lock();
}

void FileSystem::Unlock(const char* mountName) NN_NOEXCEPT
{
    int index = GetStorageIndex(mountName);

    NN_SDK_ASSERT(g_Mutexes[index].IsLockedByCurrentThread());
    g_Mutexes[index].unlock();
}

nn::Result FileSystem::Commit(const char* mountName) NN_NOEXCEPT
{
    if (g_IsTestMode)
    {
        NN_RESULT_SUCCESS;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CommitSaveData(mountName));

    NN_RESULT_SUCCESS;
}

void FileSystem::Rollback(const char* mountName) NN_NOEXCEPT
{
    if (g_IsTestMode)
    {
        return;
    }

    int index = GetStorageIndex(mountName);

    nn::fs::Unmount(mountName);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountSystemSaveData(mountName, g_StorageParams[index].saveDataId));
}

nn::Result FileSystem::GetFreeSpaceSize(int64_t* outSize, const char* mountName) NN_NOEXCEPT
{
    if (g_IsTestMode)
    {
        *outSize = g_StorageParams[GetStorageIndex(mountName)].storageSize;
    }
    else
    {
        char root[nn::fs::MountNameLengthMax + 3] = {};
        nn::util::SNPrintf(root, sizeof (root), "%s:/", mountName);

        NN_RESULT_DO(nn::fs::GetFreeSpaceSize(outSize, root));
    }

    NN_RESULT_SUCCESS;
}

nn::Result FileSystem::CreateDirectoryRecursively(const char* path, bool isFilePath) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    char currentPath[nn::fs::EntryNameLengthMax + 1] = {};
    bool root = true;

    char c = path[0];

    for (int i = 0; path[i] != '\0'; i++)
    {
        c = path[i];

        if (c == '/')
        {
            if (!root)
            {
                NN_RESULT_TRY(nn::fs::CreateDirectory(currentPath))
                    NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
                    {
                    }
                NN_RESULT_END_TRY;
            }
            else
            {
                root = false;
            }
        }

        currentPath[i] = c;
    }

    if (!root && !isFilePath && c != '/')
    {
        NN_RESULT_TRY(nn::fs::CreateDirectory(currentPath))
            NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
            {
            }
        NN_RESULT_END_TRY;
    }

    NN_RESULT_SUCCESS;
}

nn::Result FileSystem::DeleteDirectoryRecursively(const char* path) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    return nn::fs::DeleteDirectoryRecursively(path);
}

nn::Result FileSystem::CreateFile(const char* path, int64_t size, bool ignoreIfExists) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    NN_RESULT_TRY(nn::fs::CreateFile(path, size))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_DO(CreateDirectoryRecursively(path, true));
            NN_RESULT_DO(nn::fs::CreateFile(path, size));
        }
        NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
        {
            if (!ignoreIfExists)
            {
                NN_RESULT_RETHROW;
            }
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result FileSystem::GetFileSize(int64_t* outSize, const char* path) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSize);
    NN_SDK_REQUIRES_NOT_NULL(path);

    nn::fs::FileHandle handle = {};
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::GetFileSize(outSize, handle));

    NN_RESULT_SUCCESS;
}

}}}}}
