﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <algorithm>
#include <atomic>

#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>
#include <curl/curl.h>
#include <nn/ssl/ssl_Context.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/netdiag/netdiag_Result.h>
#include <nn/netdiag/netdiag_ResultPrivate.h>
#include <nn/netdiag/netdiag_BandWidthApi.h>
#include <nn/netdiag/netdiag_BandWidthTypes.h>
#include <nn/netdiag/netdiag_BandWidthTypesPrivate.h>
#include <nn/netdiag/detail/netdiag_CurlMulti.h>
#include <nn/netdiag/detail/netdiag_Log.h>

#include "netdiag_Util.h"

namespace nn { namespace netdiag { namespace detail {

namespace {

    const size_t DataSizeForUpload   =  1 * 1024 * 1024; // アップロード計測のデータサイズ： 1 MiB
    const size_t DataSizeForDownload = 30 * 1024 * 1024; // ダウンロード計測のデータサイズ：30 MiB

    //----------------------------------------------------------------
    // Download 時の Write コールバックユーザデータ
    struct CurlWriteFunctionUserData
    {
        size_t leftSize;
    };

    // Download 時の Write コールバック
    size_t CurlWriteFunction(void* ptr, size_t size, size_t nmemb, void *userdata) NN_NOEXCEPT
    {
        CurlWriteFunctionUserData* userData = static_cast<CurlWriteFunctionUserData*>(userdata);
        //NN_DETAIL_NETDIAG_INFO("[netdiag] size=%d nmemb=%d left=%d\n", size, nmemb, userData->leftSize );

        auto count = size * nmemb;
        if ( count > 0 && userData->leftSize > 0 )
        {
            // ここではデータのコピーなど行わない。単に残りバイト数を計算するのみ
            size_t writeSize = std::min( count, userData->leftSize );
            userData->leftSize -= writeSize;
        }
        else
        {
            return 0;
        }
        return count;
    }

    //----------------------------------------------------------------
    // Upload 時の Read コールバックユーザデータ
    struct CurlReadFunctionUserData
    {
        size_t leftSize;
    };

    // Upload 時の Read コールバック
    size_t CurlReadFunction(void* ptr, size_t size, size_t nmemb, void* userdata) NN_NOEXCEPT
    {
        CurlReadFunctionUserData* userData = static_cast<CurlReadFunctionUserData*>(userdata);
        //NN_DETAIL_NETDIAG_INFO("[netdiag] size=%d nmemb=%d left=%d\n", size, nmemb, userData->leftSize );

        auto count = size * nmemb;
        if ( count > 0 && userData->leftSize > 0 )
        {
            // ここではデータのコピーなど行わない。単に残りバイト数を計算するのみ
            size_t readSize = std::min( count, userData->leftSize );
            userData->leftSize -= readSize;

            return readSize;
        }
        return 0;
    }

} // namespace

//----------------------------------------------------------------
// SSL 設定
// 今は使用していませんが https:// を使う場合は必要
#if 0
namespace {
    CURLcode CurlSslContextFunction(CURL* pCurl, void* pSslContext, void* pUserData) NN_NOEXCEPT
    {
        NN_UNUSED(pCurl);
        NN_UNUSED(pUserData);

        auto& context = *reinterpret_cast<nn::ssl::Context*>(pSslContext);
        if (context.Create(nn::ssl::Context::SslVersion_Auto).IsFailure())
        {
            return CURLE_ABORTED_BY_CALLBACK;
        }
        return CURLE_OK;
    }
} // namespace

void SetCurlSslSetting( CURL* curl ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( curl );
    curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 1L);
    curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 2L);
    curl_easy_setopt(curl, CURLOPT_SSL_VERIFYDATE, 0L);
    curl_easy_setopt(curl, CURLOPT_SSL_CTX_FUNCTION, CurlSslContextFunction);
}
#endif

//----------------------------------------------------------------
// proxy 設定
// デバッグ用途
#if 0
void SetCurlProxySetting( CURL* curl ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( curl );
    curl_easy_setopt(curl, CURLOPT_PROXYAUTOCONFIG, 0);
    curl_easy_setopt(curl, CURLOPT_PROXY, "proxy.nintendo.co.jp");
    curl_easy_setopt(curl, CURLOPT_PROXYPORT, 8080);
    curl_easy_setopt(curl, CURLOPT_PROXYAUTH, CURLAUTH_BASIC);
}
#endif

//----------------------------------------------------------------
// curl によるスピード計測
void GetMeasuringResult( CURL* curl, BandWidth* pUploadSpeed, BandWidth* pDownloadSpeed ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( curl );
    if ( pUploadSpeed )
    {
        curl_easy_getinfo(curl, CURLINFO_SPEED_UPLOAD, pUploadSpeed );
        NN_DETAIL_NETDIAG_INFO( "[netdiag] Upload speed : %f (bytes/sec)\n", *pUploadSpeed );
    }
    if ( pDownloadSpeed )
    {
        curl_easy_getinfo(curl, CURLINFO_SPEED_DOWNLOAD, pDownloadSpeed );
        NN_DETAIL_NETDIAG_INFO( "[netdiag] Download speed : %f (bytes/sec)\n", *pDownloadSpeed );
    }

#if 0
    // デバッグ用途: 転送時間の表示
    double transferTime;
    curl_easy_getinfo( curl, CURLINFO_TOTAL_TIME, &transferTime );
    NN_DETAIL_NETDIAG_INFO( "[netdiag] Transfer time : %f (sec)\n", transferTime );
#endif
}

//----------------------------------------------------------------
// アップロードの帯域計測
//
nn::Result MeasureBandWidthForUpload( BandWidth* pBandWidth, uint32_t timeOut, bool* pIsTimedout ) NN_NOEXCEPT
{
    ClearInterruptCurlMulti(); // 中断フラグのクリア

    CURL* curl;
    NN_RESULT_DO(AcquireCurlHandle(&curl));
    NN_UTIL_SCOPE_EXIT
    {
        curl_easy_cleanup(curl);
    };

    //SetCurlSslSetting( curl );
    //SetCurlProxySetting( curl );

    curl_easy_setopt(curl, CURLOPT_URL, MeasuringServerForUpload); // アップロードサーバ
    curl_easy_setopt(curl, CURLOPT_USERAGENT, UserAgentForMeasuring); // User-Agent 指定
    curl_easy_setopt(curl, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_0); // HTTP バージョン
#if defined(NN_SDK_BUILD_DEBUG)
    curl_easy_setopt(curl, CURLOPT_VERBOSE, 1L);
#endif
    curl_easy_setopt(curl, CURLOPT_POST, 1L);

    CurlReadFunctionUserData userData = { DataSizeForUpload };
    curl_easy_setopt(curl, CURLOPT_READDATA, &userData);
    curl_easy_setopt(curl, CURLOPT_READFUNCTION, CurlReadFunction);
    curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE, DataSizeForUpload);

    // タイムアウト設定
    if ( timeOut > 0 )
    {
        curl_easy_setopt(curl, CURLOPT_TIMEOUT_MS, timeOut);
    }

    CURLcode curlCode;
    NN_RESULT_DO(PerformCurlMulti(curl, &curlCode));
    NN_RESULT_TRY(HandleCurlError(curlCode))
        NN_RESULT_CATCH(ResultCurlErrorOperationTimedout)
        {
            if (pIsTimedout)
            {
                *pIsTimedout = true;
            }
            GetMeasuringResult( curl, pBandWidth, nullptr );
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY

    // レスポンスコードの確認
    long responseCode;
    curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &responseCode);
    NN_RESULT_DO(HandleHttpStatus(responseCode));

    // 計測
    GetMeasuringResult( curl, pBandWidth, nullptr );
    NN_RESULT_SUCCESS;
}

//----------------------------------------------------------------
// ダウンロードの帯域計測
//
nn::Result MeasureBandWidthForDownload( BandWidth* pBandWidth, uint32_t timeOut, bool* pIsTimedout ) NN_NOEXCEPT
{
    ClearInterruptCurlMulti(); // 中断フラグのクリア

    CURL* curl;
    NN_RESULT_DO(AcquireCurlHandle(&curl));
    NN_UTIL_SCOPE_EXIT
    {
        curl_easy_cleanup(curl);
    };

    //SetCurlSslSetting( curl );
    //SetCurlProxySetting( curl );

    curl_easy_setopt(curl, CURLOPT_URL, MeasuringServerForDownload); // ダウンロードサーバ
    curl_easy_setopt(curl, CURLOPT_USERAGENT, UserAgentForMeasuring); // User-Agent
    curl_easy_setopt(curl, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_0); // HTTP バージョン
#if defined(NN_SDK_BUILD_DEBUG)
    curl_easy_setopt(curl, CURLOPT_VERBOSE, 1L);
#endif

    curl_easy_setopt(curl, CURLOPT_FRESH_CONNECT, 1L); // キャッシュしない

    CurlWriteFunctionUserData userData = { DataSizeForDownload };
    curl_easy_setopt(curl, CURLOPT_WRITEDATA, &userData);
    curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION,  CurlWriteFunction);

    // タイムアウト設定
    if ( timeOut > 0 )
    {
        curl_easy_setopt(curl, CURLOPT_TIMEOUT_MS, timeOut);
    }

    CURLcode curlCode;
    NN_RESULT_DO(PerformCurlMulti(curl, &curlCode));
    NN_RESULT_TRY(HandleCurlError(curlCode))
        NN_RESULT_CATCH(ResultCurlErrorOperationTimedout)
        {
            if (pIsTimedout)
            {
                *pIsTimedout = true;
            }
            GetMeasuringResult( curl, nullptr, pBandWidth );
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY

    // レスポンスコードの確認
    long responseCode;
    curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &responseCode);
    NN_RESULT_DO(HandleHttpStatus(responseCode));

    // 計測
    GetMeasuringResult( curl, nullptr, pBandWidth );
    NN_RESULT_SUCCESS;
}

//----------------------------------------------------------------
// 帯域測定の中断指示を出す
//
void InterruptMeasureBandWidth() NN_NOEXCEPT
{
    InterruptCurlMulti();
}

}}} // nn::netdiag::detail
