﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/mii/mii_PrivateDatabase.h>
#include <nn/mii/detail/mii_Debug.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_StringUtil.h>
//#include <nn/gfx/gfx_SamplerInfo.h>
#include <nn/mii/detail/mii_Color.h>
#include <nn/mii/detail/mii_DatabaseImpl.h>
#include <nn/mii/detail/mii_DatabaseInterfaceVersion.h>
#include <nn/mii/detail/mii_BufferAllocator.h>
#include <nn/mii/mii_Result.h>
#include <nn/mii/mii_Special.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/mii/detail/service/mii_ServiceProviderClient.h>
#include <nn/mii/detail/service/mii_IDatabaseService.sfdl.h>

#include <cstring>
#include <new>

namespace nn { namespace mii {

namespace detail
{
    extern ServiceProviderClient g_ServiceProviderClient;
}

PrivateDatabase::PrivateDatabase() NN_NOEXCEPT
{
    new(&m_Storage) nn::sf::SharedPointer<detail::IDatabaseService>();
}

PrivateDatabase::~PrivateDatabase() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsAvailable());
    reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage)->~SharedPointer();
}

bool PrivateDatabase::IsBrokenDatabaseWithClearFlag() NN_NOEXCEPT
{
    if (detail::g_ServiceProviderClient.Initialize().IsFailure())
    {
        return false;
    }

    nn::sf::SharedPointer<detail::IDatabaseService> pIDatabaseService;

    if (detail::g_ServiceProviderClient.GetDatabaseServiceSharedPointer(&pIDatabaseService, GetSpecialMiiKeyCode()).IsFailure())
    {
        return false;
    }

    bool isBrokenDatabase;

    if (pIDatabaseService->IsBrokenDatabaseWithClearFlag(&isBrokenDatabase).IsFailure())
    {
        return false;
    }

    return isBrokenDatabase;
}

nn::Result PrivateDatabase::Initialize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsAvailable());

    NN_RESULT_DO(detail::g_ServiceProviderClient.Initialize());
    NN_RESULT_DO(
        detail::g_ServiceProviderClient.GetDatabaseServiceSharedPointer(
            reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage),
            GetSpecialMiiKeyCode()
        )
    );
    NN_RESULT_DO(
        (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->SetInterfaceVersion(detail::DatabaseInterfaceVersion_Current)
    );
    NN_RESULT_SUCCESS;
}

void PrivateDatabase::Finalize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    *reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage) = nullptr;
}
bool PrivateDatabase::IsAvailable() const NN_NOEXCEPT
{
    return *reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage) != nullptr;
}
bool PrivateDatabase::IsUpdated(int sourceFlags) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    bool isUpdated = false;
    auto result = (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->IsUpdated(&isUpdated, sourceFlags);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return isUpdated;
}

bool PrivateDatabase::IsFullDatabase() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    bool isFullDatabase = false;
    auto result = (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->IsFullDatabase(&isFullDatabase);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return isFullDatabase;
}

int PrivateDatabase::GetCount(int sourceFlags) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    int32_t count;
    auto result = (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->GetCount(&count, sourceFlags);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return count;
}

nn::Result PrivateDatabase::Get(int* pOutCount,StoreDataElement* pElements,int count,int sourceFlags) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    return (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->Get2(pOutCount, nn::sf::OutArray<StoreDataElement>(pElements, count), sourceFlags);
}
nn::Result PrivateDatabase::Get(int* pOutCount,StoreData* pElements,int count,int sourceFlags) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(pElements);

    return (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->Get3(pOutCount, nn::sf::OutArray<StoreData>(pElements,count),sourceFlags);
}

nn::Result PrivateDatabase::UpdateLatest(StoreData* pInfo,int sourceFlags) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    NN_SDK_REQUIRES_NOT_NULL(pInfo);

    return (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->UpdateLatest1(pInfo,*pInfo,sourceFlags);
}

int PrivateDatabase::FindIndex(const nn::mii::CreateId& id, bool enabledSpecialMii) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    int index = 0;
    (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->FindIndex(&index,id,enabledSpecialMii);
    return index;
}

nn::Result PrivateDatabase::Move(int newIndex,const CreateId& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    return (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->Move(newIndex,id);
}
nn::Result PrivateDatabase::AddOrReplace(const StoreData& storeData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    return (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->AddOrReplace(storeData);
}
nn::Result PrivateDatabase::Delete(const CreateId& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    return (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->Delete(id);
}
nn::Result PrivateDatabase::DestroyFile() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    return (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->DestroyFile();
}
nn::Result PrivateDatabase::DeleteFile() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    return (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->DeleteFile();
}
void PrivateDatabase::Format() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    auto result = (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->Format();

    // 正常系として非テストモード時にResultDevelopmentOnlyが返るので、ABORTから除外
    if(result.IsFailure() && !nn::mii::ResultDevelopmentOnly::Includes(result))
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
}
nn::Result PrivateDatabase::Import(const void* pBuffer,size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);

    return (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->Import(nn::sf::InBuffer(reinterpret_cast<const char*>(pBuffer),size));
}
void PrivateDatabase::Export(void* pBuffer,size_t size) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);

    auto result = (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->Export(nn::sf::OutBuffer(reinterpret_cast<char*>(pBuffer),size));

    // 正常系として非テストモード時にResultDevelopmentOnlyが返るので、ABORTから除外
    if(result.IsFailure() && !nn::mii::ResultDevelopmentOnly::Includes(result))
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
}


}}
