﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/mii/mii_Database.h>
#include <nn/mii/detail/mii_Debug.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_StringUtil.h>
//#include <nn/gfx/gfx_SamplerInfo.h>
#include <nn/mii/detail/mii_Color.h>
#include <nn/mii/detail/mii_DatabaseImpl.h>
#include <nn/mii/detail/mii_DatabaseInterfaceVersion.h>
#include <nn/mii/detail/mii_BufferAllocator.h>
#include <nn/mii/mii_Result.h>
#include <nn/mii/mii_Special.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/mii/detail/service/mii_ServiceProviderClient.h>
#include <nn/mii/detail/service/mii_IDatabaseService.sfdl.h>

#include <cstring>
#include <new>

namespace nn { namespace mii {

namespace detail
{
    extern ServiceProviderClient g_ServiceProviderClient;
}

Database::Database() NN_NOEXCEPT
{
    new(&m_Storage) nn::sf::SharedPointer<detail::IDatabaseService>();
}

Database::~Database() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsAvailable());
    reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage)->~SharedPointer();
}

nn::Result Database::Initialize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsAvailable());

    NN_RESULT_DO(detail::g_ServiceProviderClient.Initialize());
    NN_RESULT_DO(
        detail::g_ServiceProviderClient.GetDatabaseServiceSharedPointer(
            reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage),
            GetSpecialMiiKeyCode()
        )
    );
    NN_RESULT_DO(
        (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->SetInterfaceVersion(detail::DatabaseInterfaceVersion_Current)
    );

    NN_RESULT_SUCCESS;
}

void Database::Finalize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    *reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage) = nullptr;
}
bool Database::IsAvailable() const NN_NOEXCEPT
{
    return *reinterpret_cast<const nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage) != nullptr;
}
bool Database::IsUpdated(int sourceFlags) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    bool isUpdated = false;
    auto result = (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->IsUpdated(&isUpdated, sourceFlags);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return isUpdated;
}

bool Database::IsFullDatabase() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    bool isFullDatabase = false;
    auto result = (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->IsFullDatabase(&isFullDatabase);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return isFullDatabase;
}


int Database::GetCount(int sourceFlags) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    int32_t count;
    auto result = (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->GetCount(&count, sourceFlags);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return count;
}

nn::Result Database::GetIndex(int* pOut, const CharInfo& info) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    return (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->GetIndex(pOut, info);
}

nn::Result Database::Get(int* pOutCount,CharInfoElement* pElements,int count,int sourceFlags) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    return (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->Get(pOutCount,nn::sf::OutArray<CharInfoElement>(pElements,count),sourceFlags);
}
nn::Result Database::Get(int* pOutCount,CharInfo* pElements,int count,int sourceFlags) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(pElements);

    return (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->Get1(pOutCount,nn::sf::OutArray<CharInfo>(pElements,count),sourceFlags);
}

nn::Result Database::UpdateLatest(CharInfo* pInfo,int sourceFlags) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    NN_SDK_REQUIRES_NOT_NULL(pInfo);
    return (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->UpdateLatest(pInfo,*pInfo,sourceFlags);
}

nn::Result Database::Convert(CharInfo* pInfo,const Ver3StoreData& ver3StoreData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    NN_SDK_REQUIRES_NOT_NULL(pInfo);
    return (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->Convert(pInfo, ver3StoreData);
}

void Database::BuildRandom(CharInfo* pInfo,const RandomDesc& desc) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    NN_SDK_REQUIRES_NOT_NULL(pInfo);

    auto result = (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->BuildRandom(pInfo,desc.GetAge(),desc.GetGender(),desc.GetRace());
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}
void Database::BuildDefault(CharInfo* pInfo,int index) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    NN_SDK_REQUIRES_NOT_NULL(pInfo);
    NN_SDK_REQUIRES_RANGE(index, 0, nn::mii::DefaultMiiCount);

    auto result = (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->BuildDefault(pInfo,index);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}
void Database::Format() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    auto result = (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->Format();

    // 正常系として非テストモード時にResultDevelopmentOnlyが返るので、ABORTから除外
    if(result.IsFailure() && !nn::mii::ResultDevelopmentOnly::Includes(result))
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
}
nn::Result Database::Import(const void* pBuffer,size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    return (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->Import(nn::sf::InBuffer(reinterpret_cast<const char*>(pBuffer),size));
}
void Database::Export(void* pBuffer,size_t size) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAvailable());
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);

    auto result = (*reinterpret_cast<nn::sf::SharedPointer<detail::IDatabaseService>*>(&m_Storage))->Export(nn::sf::OutBuffer(reinterpret_cast<char*>(pBuffer),size));

    // 正常系として非テストモード時にResultDevelopmentOnlyが返るので、ABORTから除外
    if(result.IsFailure() && !nn::mii::ResultDevelopmentOnly::Includes(result))
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
}


}}
