﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Utf8StringUtil.h>
#include "detail/lm_LogPacketTransmitter.h"
#include "detail/lm_UnitHeapAllocator.h"
#include "detail/lm_ShimLibraryGlobal.h"
#include "detail/lm_ShimLibraryForLogGetter.h"

namespace nn { namespace diag { namespace detail {
    void ReplaceDefaultLogObserver(LogObserver logObserver) NN_NOEXCEPT;
    void ResetDefalutLogObserver() NN_NOEXCEPT;
    void GetProcessNamePointer(const char** out, size_t* outLength) NN_NOEXCEPT;
}}} // nn::diag::detail

namespace nn { namespace lm {

namespace {
    const size_t TransmissionBufferSize = 1024u;
    const size_t TransmissionBufferCount = 4u;
    const size_t HeapStorageAlign = NN_ALIGNOF(detail::LogPacketHeader);
    typedef detail::UnitHeapAllocator<TransmissionBufferSize, TransmissionBufferCount, HeapStorageAlign> TransmissionBufferAllocator;

    bool LogPacketTransmitterFlushFunction(const uint8_t* pointer, size_t size) NN_NOEXCEPT
    {
        auto result = detail::ShimLibraryGlobal::GetInstance().GetLogger()->Log(sf::InBuffer(reinterpret_cast<const char*>(pointer), size));

        // 実装は常に成功を返すが、sf が成功以外の result を返すこともあるため
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        return result.IsSuccess();
    }

    using PushTextFunction = void (detail::LogPacketTransmitter::*)(const char*, size_t);

    void InvokePushTextWithUtf8Sanitizing(detail::LogPacketTransmitter& transmitter, PushTextFunction push, const char* string) NN_NOEXCEPT
    {
        const auto length = std::strlen(string);

        if (length == 0 || util::VerifyUtf8String(string, length))
        {
            (transmitter.*push)(string, length);
        }
        else
        {
            (transmitter.*push)("(Invalid UTF8 string)", sizeof("(Invalid UTF8 string)") - 1);
        }
    }

    void LogManagerLogObserver(const diag::LogMetaData& logMetaData, const diag::LogBody& logBody, void*) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!logMetaData.useDefaultLocaleCharset); // 実機では必ず UTF8 を使用している。

        auto buffer = static_cast<char*>(TransmissionBufferAllocator::GetInstance().AllocateSync());
        NN_UTIL_SCOPE_EXIT { TransmissionBufferAllocator::GetInstance().Free(buffer); };

        detail::LogPacketTransmitter transmitter(
            buffer, TransmissionBufferSize, LogPacketTransmitterFlushFunction,
            static_cast<uint8_t>(logMetaData.severity), static_cast<uint8_t>(logMetaData.verbosity), 0,
            logBody.isHead, logBody.isTail);

        if (logBody.isHead)
        {
            transmitter.PushUserSystemClock(os::GetSystemTick().ToTimeSpan().GetSeconds());
            transmitter.PushLineNumber(logMetaData.sourceInfo.lineNumber);
            InvokePushTextWithUtf8Sanitizing(transmitter, &detail::LogPacketTransmitter::PushFileName, logMetaData.sourceInfo.fileName);
            InvokePushTextWithUtf8Sanitizing(transmitter, &detail::LogPacketTransmitter::PushFunctionName, logMetaData.sourceInfo.functionName);
            InvokePushTextWithUtf8Sanitizing(transmitter, &detail::LogPacketTransmitter::PushModuleName, logMetaData.moduleName);
            InvokePushTextWithUtf8Sanitizing(transmitter, &detail::LogPacketTransmitter::PushThreadName, os::GetThreadNamePointer(os::GetCurrentThread()));
            const char* processName;
            size_t processNameLength;
            diag::detail::GetProcessNamePointer(&processName, &processNameLength);
            transmitter.PushProcessName(processName, processNameLength); // RO セグメント上に静的に生成された文字列を参照しているため、サニタイズ不要。
        }

        // TORIAEZU: スタック不足によりメッセージが分割している場合は、妥当な UTF8 文字列にならないためサニタイズしない。
        transmitter.PushTextLog(logBody.message, logBody.messageBytes);
    }
} // anonymous

void Initialize() NN_NOEXCEPT
{
    // LogManager との IPC セッション切断を利用して、ログ出力の完了を通知するため、
    // ログ出力しない場合でも、LogManager と IPC セッションを張る必要がある。
    detail::ShimLibraryGlobal::GetInstance();

    diag::detail::ReplaceDefaultLogObserver(LogManagerLogObserver);
}

void Finalize() NN_NOEXCEPT
{
    diag::detail::ResetDefalutLogObserver();
}

void SetDestination(Bit32 destination) NN_NOEXCEPT
{
    detail::ShimLibraryGlobal::GetInstance().GetLogger()->SetDestination(destination);
}

void StartLogging() NN_NOEXCEPT
{
    detail::ShimLibraryForLogGetter::GetInstance().GetLogger()->StartLogging();
}

void StopLogging() NN_NOEXCEPT
{
    detail::ShimLibraryForLogGetter::GetInstance().GetLogger()->StopLogging();
}

void GetLog(char* data, size_t size, int64_t *writeSize, uint32_t* dropCount) NN_NOEXCEPT
{
    detail::ShimLibraryForLogGetter::GetInstance().GetLogger()->GetLog(sf::OutBuffer(reinterpret_cast<char*>(data), size), sf::Out<int64_t>(writeSize), sf::Out<uint32_t>(dropCount));
}

}} // nn::lm
