﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <nn/nn_Common.h>
#include <nn/os/os_Mutex.h>

namespace nn { namespace lm { namespace impl {

/*!
    @brief  ログを SD カードに書き出します。
*/
class SdCardLogger
{
    NN_DISALLOW_COPY( SdCardLogger );
    NN_DISALLOW_MOVE( SdCardLogger );

private:
    /*!
    @brief      コンストラクタです。
    */
    SdCardLogger() NN_NOEXCEPT;

public:
    /*!
    @brief      デストラクタです。
    */
    ~SdCardLogger() NN_NOEXCEPT;

public:
    /*!
    @brief      ロギングの状態が変化したときに呼ばれるオブザーバーです。

    @param[in]  isAvailable     ロギングが有効化されたときに true、
                                無効化されたときに false。
    */
    typedef void (*LoggingObserver)(bool isAvailable);

public:
    /*!
    @brief      インスタンスを取得します。

    @return     インスタンス。
    */
    static SdCardLogger& GetInstance() NN_NOEXCEPT;

    /*!
    @brief      SD カードにログを書き出します。

    @param[in]  message        書き出すメッセージ。
    @param[in]  messageSize    書き出すメッセージのバイトサイズ。

    @return     成功時に true、失敗時に false。
    */
    bool Write(const uint8_t* message, size_t messageSize) NN_NOEXCEPT;

    /*!
    @brief      ロギングの状態が変化したときに呼ばれるオブザーバーを登録します。

    @param[in]  observer    登録するオブザーバー。

    @details    既にオブザーバーが登録されている場合は、上書きします。@n
                引数に nullptr を指定すると、オブザーバーの登録を解除します。@n
                オブザーバーの型は @ref LoggingObserver を参照してください。
    */
    void SetLoggingObserver(LoggingObserver observer) NN_NOEXCEPT;

private:
    /*!
    @brief      SD カードにログを書き出す設定値を取得します。

    @return     設定値を返します。

    @details    SD カードにログを書き出す設定値を取得します。
                書き出す設定の場合は true、そうでないときは false を返します。
    */
    bool GetEnabled() const NN_NOEXCEPT;

    /*!
    @brief      SD カードにログを書き出すかどうかを設定します。

    @param[in]  isEnabled   設定値。

    @details    SD カードにログを書き出すかどうかを設定します。
                書き出す設定の場合は true、そうでないときは false 引数に指定します。
    */
    void SetEnabled(bool isEnabled) NN_NOEXCEPT;

    bool Initialize() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

private:
    /*!
    @brief      指定した名前のディレクトリが存在するか確認して、
                存在しなければ生成します。

    @param[in]  dirname    ディレクトリ名。

    @return     成功時に true、失敗時に false。
    */
    static bool EnsureLogDirectory(const char* driname) NN_NOEXCEPT;

    /*!
    @brief      指定した名前のディレクトリに、ログファイルを生成します。

    @param[out] out         生成したログファイルのフルパス。
    @param[in]  size        out のバイトサイズ。
    @param[in]  dirname     ディレクトリ名。

    @return     成功時に true、失敗時に false。
    */
    static bool GenerateLogFile(char* out, size_t size, const char* dirname) NN_NOEXCEPT;

    static bool WriteLogFileHeader(const char* path) NN_NOEXCEPT;
    static bool WriteLogFileBody(const char* path, int64_t offset, const uint8_t* message, size_t messageSize) NN_NOEXCEPT;
    static bool GetSdCardLoggingEnabled() NN_NOEXCEPT;
    static bool GetSdCardLogOutputDirectory(char* out, size_t size) NN_NOEXCEPT;

private:
    os::Mutex m_LoggingObserverMutex;

    bool m_IsEnabled;
    bool m_IsSdCardMounted;
    char m_LogFilePath[128];
    int64_t m_LogFileOffset;

    LoggingObserver m_LoggingObserver;
};

}}} // nn::lm::impl
