﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SystemThreadDefinition.h>
#include "lm_SdCardLogger.h"
#include "../detail/lm_Log.h"

#include <nn/fs.h>
#include <nn/fs/fs_IEventNotifier.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/system/settings_SerialNumber.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/time/time_TimeZoneApi.h>

namespace nn { namespace lm { namespace impl {

namespace {
    const char MountName[] = "sdcard";
    const char LogFileExtension[] = "nxbinlog";

    const char SettingName[] = "lm";
#if !defined(NN_DETAIL_LM_ENABLE_SD_LOGGING_FORCIBLY)
    const char SettingKeyLoggingEnabled[] = "enable_sd_card_logging";
#endif
    const char SettingKeyOutputDirectory[] = "sd_card_log_output_directory";

    const size_t LogFileHeaderSize = 8;
    const Bit32 LogFileHeaderMagicNumber = 0x70687068; // "phph"
    const Bit8 LogFileHeaderVersion = 1;

    struct LogFileHeader
    {
        Bit32 magicNumber;
        Bit8 version;
        Bit8 reserved[3];
    };
    NN_STATIC_ASSERT(sizeof(LogFileHeader) == LogFileHeaderSize);
}

namespace
{
    std::unique_ptr<fs::IEventNotifier> g_SdCardDetectionEventNotifier;
    os::SystemEvent g_SdCardDetectionEvent;

    class SdCardDetectionEventInitializer
    {
    public:
        SdCardDetectionEventInitializer() NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenSdCardDetectionEventNotifier(&g_SdCardDetectionEventNotifier));
            NN_ABORT_UNLESS_RESULT_SUCCESS(g_SdCardDetectionEventNotifier->BindEvent(g_SdCardDetectionEvent.GetBase(), nn::os::EventClearMode_ManualClear));
        }
    };

    bool IsSdCardInserted() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(SdCardDetectionEventInitializer, s_SdCardDetectionEventInitializer);
        NN_UNUSED(s_SdCardDetectionEventInitializer);
        NN_FUNCTION_LOCAL_STATIC(bool, s_CacheValue, = nn::fs::IsSdCardInserted());

        if (g_SdCardDetectionEvent.TryWait())
        {
            g_SdCardDetectionEvent.Clear();
            s_CacheValue = nn::fs::IsSdCardInserted();
        }

        return s_CacheValue;
    }
}

namespace
{
    bool MakeLogFilePathWithoutExtension(char* out, size_t size, const char* dirname) NN_NOEXCEPT
    {
        time::PosixTime currentTime;
        if (time::StandardUserSystemClock::GetCurrentTime(&currentTime).IsFailure())
        {
            NN_SDK_ASSERT(false);
            return false;
        }
        const auto calendar = time::ToCalendarTimeInUtc(currentTime);

        settings::system::SerialNumber serialNumber;
        settings::system::GetSerialNumber(&serialNumber);

        const auto count = static_cast<size_t>(util::SNPrintf(out, size, "%s:/%s/%s_%04d%02d%02d%02d%02d%02d",
            MountName, dirname, serialNumber.string,
            calendar.year, calendar.month, calendar.day,
            calendar.hour, calendar.minute, calendar.second));
        if (!(count < size))
        {
            NN_SDK_ASSERT(false);
            return false;
        }

        return true;
    }
}

SdCardLogger::SdCardLogger() NN_NOEXCEPT
    : m_LoggingObserverMutex(false)
    , m_IsEnabled(false)
    , m_IsSdCardMounted(false)
    , m_LogFileOffset(0)
    , m_LoggingObserver(nullptr)
{}

SdCardLogger::~SdCardLogger() NN_NOEXCEPT
{
    Finalize();
}

SdCardLogger& SdCardLogger::GetInstance() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(SdCardLogger, s_Instance);
    return s_Instance;
}

bool SdCardLogger::GetEnabled() const NN_NOEXCEPT
{
    return m_IsEnabled;
}

void SdCardLogger::SetEnabled(bool isEnabled) NN_NOEXCEPT
{
    const auto isChanged = m_IsEnabled != isEnabled;
    if (!isChanged)
    {
        return;
    }

    m_IsEnabled = isEnabled;

    NN_DETAIL_LM_INFO("Sd card logging is %s.\n", isEnabled ? "enabled" : "disabled");

    std::lock_guard<nn::os::Mutex> lock(m_LoggingObserverMutex);

    if (m_LoggingObserver)
    {
        m_LoggingObserver(isEnabled);
    }
}

void SdCardLogger::SetLoggingObserver(LoggingObserver observer) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_LoggingObserverMutex);
    m_LoggingObserver = observer;
}

bool SdCardLogger::WriteLogFileBody(const char* path, int64_t offset, const uint8_t* message, size_t messageSize) NN_NOEXCEPT
{
    const auto result = [&]() -> Result
    {
        fs::FileHandle handle;

        NN_RESULT_DO(fs::OpenFile(&handle, path, fs::OpenMode_Write | fs::OpenMode_AllowAppend));

        NN_UTIL_SCOPE_EXIT { fs::CloseFile(handle); };

        NN_RESULT_DO(fs::WriteFile(handle, offset, message, messageSize, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));

        NN_RESULT_SUCCESS;
    }();

    if (result.IsFailure())
    {
        if (fs::ResultPortSdCardDeviceRemoved::Includes(result))
        {
            NN_DETAIL_LM_ERROR("Sd card is detached.\n");
        }
        else
        {
            NN_DETAIL_LM_ERROR("Failed to write log body. path=\"%s\", error=%03d-%04d\n",
                path, result.GetModule(), result.GetDescription());
        }
        return false;
    }

    return true;
}

bool SdCardLogger::EnsureLogDirectory(const char* dirname) NN_NOEXCEPT
{
    char path[128];
    const auto count = static_cast<size_t>(util::SNPrintf(path, sizeof(path), "%s:/%s", MountName, dirname));
    if (!(count < sizeof(path)))
    {
        NN_SDK_ASSERT(false);
        return false;
    }

    const auto result = fs::CreateDirectory(path);
    if (!(result.IsSuccess() || fs::ResultPathAlreadyExists::Includes(result)))
    {
        NN_DETAIL_LM_ERROR("Failed to create log directory. path=\"%s\", error=%03d-%04d\n",
            path, result.GetModule(), result.GetDescription());
        return false;
    }

    return true;
}

bool SdCardLogger::GenerateLogFile(char* out, size_t size, const char* dirname) NN_NOEXCEPT
{
    char pathWithoutExtension[128];
    if (!MakeLogFilePathWithoutExtension(pathWithoutExtension, sizeof(pathWithoutExtension), dirname))
    {
        return false;
    }

    const int IndexMax = 99;
    for (int index = 1; index <= IndexMax; index++)
    {
        const auto count = index == 1
            ? static_cast<size_t>(util::SNPrintf(out, size, "%s.%s", pathWithoutExtension, LogFileExtension))
            : static_cast<size_t>(util::SNPrintf(out, size, "%s_%d.%s", pathWithoutExtension, index, LogFileExtension));
        if (!(count < size))
        {
            NN_SDK_ASSERT(false);
            return false;
        }

        const auto result = fs::CreateFile(out, 0);
        if (result.IsSuccess())
        {
            return true;
        }
        else if (fs::ResultPathAlreadyExists::Includes(result))
        {
            // 存在するときはインクリメントしていく。
            continue;
        }
        else
        {
            NN_DETAIL_LM_ERROR("Failed to create log file. path=\"%s\", error=%03d-%04d\n",
                out, result.GetModule(), result.GetDescription());
            return false;
        }
    }

    return false;
}

bool SdCardLogger::WriteLogFileHeader(const char* path) NN_NOEXCEPT
{
    const auto result = [&]() -> Result
    {
        fs::FileHandle handle;

        NN_RESULT_DO(fs::OpenFile(&handle, path, fs::OpenMode_Write | fs::OpenMode_AllowAppend));

        NN_UTIL_SCOPE_EXIT { fs::CloseFile(handle); };

        LogFileHeader header;
        header.magicNumber = LogFileHeaderMagicNumber;
        header.version = LogFileHeaderVersion;

        NN_RESULT_DO(fs::WriteFile(handle, 0, &header, sizeof(header), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));

        NN_RESULT_SUCCESS;
    }();

    if (result.IsFailure())
    {
        if (fs::ResultPortSdCardDeviceRemoved::Includes(result))
        {
            NN_DETAIL_LM_ERROR("Sd card is detached.\n");
        }
        else
        {
            NN_DETAIL_LM_ERROR("Failed to write log header. path=\"%s\", error=%03d-%04d\n",
                path, result.GetModule(), result.GetDescription());
        }
        return false;
    }

    return true;
}

bool SdCardLogger::GetSdCardLoggingEnabled() NN_NOEXCEPT
{
#if defined(NN_DETAIL_LM_ENABLE_SD_LOGGING_FORCIBLY)
    return true;
#else
    bool isEnabled;
    const auto readSize = settings::fwdbg::GetSettingsItemValue(&isEnabled, sizeof(isEnabled), SettingName, SettingKeyLoggingEnabled);
    if (readSize != sizeof(isEnabled))
    {
        NN_SDK_ASSERT(false);
        return false;
    }
    return isEnabled;
#endif
}

bool SdCardLogger::GetSdCardLogOutputDirectory(char* out, size_t size) NN_NOEXCEPT
{
    const auto valueSize = settings::fwdbg::GetSettingsItemValueSize(SettingName, SettingKeyOutputDirectory);
    if (!(valueSize <= size))
    {
        NN_SDK_ASSERT(false);
        return false;
    }
    const auto readSize = settings::fwdbg::GetSettingsItemValue(out, size, SettingName, SettingKeyOutputDirectory);
    if (readSize != valueSize)
    {
        NN_SDK_ASSERT(false);
        return false;
    }
    return true;
}

bool SdCardLogger::Initialize() NN_NOEXCEPT
{
    if (GetEnabled())
    {
        return true;
    }

    if (!IsSdCardInserted())
    {
        return false;
    }

    const auto result = fs::MountSdCardForDebug(MountName);
    if (result.IsFailure())
    {
        NN_DETAIL_LM_ERROR("Failed to mount sd card. error=%03d-%04d\n",
            result.GetModule(), result.GetDescription());
        return false;
    }
    m_IsSdCardMounted = true;

    // ログ置き場ディレクトリの用意
    char outputDirectory[128];
    if (!GetSdCardLogOutputDirectory(outputDirectory, sizeof(outputDirectory)))
    {
        return false;
    }
    if (!EnsureLogDirectory(outputDirectory))
    {
        return false;
    }

    // ファイル名の決定
    if (!GenerateLogFile(m_LogFilePath, sizeof(m_LogFilePath), outputDirectory))
    {
        return false;
    }

    // バイナリログのヘッダを書く
    if (!WriteLogFileHeader(m_LogFilePath))
    {
        return false;
    }
    m_LogFileOffset = LogFileHeaderSize;

    return true;
}

void SdCardLogger::Finalize() NN_NOEXCEPT
{
    SetEnabled(false);
    if (m_IsSdCardMounted)
    {
        fs::Unmount(MountName);
        m_IsSdCardMounted = false;
    }
}

bool SdCardLogger::Write(const uint8_t* message, size_t messageSize) NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(bool, s_IsSdCardLoggingEnabled, = GetSdCardLoggingEnabled());
    if (!s_IsSdCardLoggingEnabled)
    {
        return false;
    }

    bool isSuccess = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (!isSuccess && m_IsSdCardMounted)
        {
            fs::Unmount(MountName);
            m_IsSdCardMounted = false;
        }
        SetEnabled(isSuccess);
    };

    if (!Initialize())
    {
        return false;
    }

    if (!WriteLogFileBody(m_LogFilePath, m_LogFileOffset, message, messageSize))
    {
        return false;
    }
    m_LogFileOffset += messageSize;

    isSuccess = true;

    return true;
}

}}} // nn::lm::impl
