﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/lm/lm_LogDestinationType.h>
#include "lm_LoggerImpl.h"
#include "lm_EventLogTransmitter.h"
#include "lm_LogBuffer.h"
#include "lm_LogGetterImpl.h"
#include "lm_LogPacketParser.h"
#include "../detail/lm_Log.h"
#include "../detail/lm_LogPacketHeader.h"

namespace nn { namespace lm { namespace impl {

bool IsFlushAvailable() NN_NOEXCEPT;

bool g_IsLoggingToCustomSink = false;

namespace {
#if defined(NN_DETAIL_LM_ENABLE_BYPASS_LOGGING)
    Bit32 g_LogDestination = nn::lm::LogDestination::LogDestination_TargetManager | nn::lm::LogDestination::LogDestination_UartIfSleep;
#else
    Bit32 g_LogDestination = nn::lm::LogDestination::LogDestination_TargetManager;
#endif

    void PutLogToUart(const sf::InBuffer& message) NN_NOEXCEPT
    {
#if defined(NN_SDK_BUILD_RELEASE)
        NN_UNUSED(message);
#else
        auto pPacket    = message.GetPointerUnsafe();
        auto packetSize = message.GetSize();
        auto pHeader    = reinterpret_cast<const detail::LogPacketHeader*>(pPacket);

        char moduleName[16] = {};
        LogPacketParser::ParseModuleName(moduleName, sizeof (moduleName), pPacket, packetSize);

        diag::LogMetaData logMetaData = {};
        logMetaData.moduleName = moduleName;
        logMetaData.severity   = static_cast<diag::LogSeverity>(pHeader->GetSeverity());
        logMetaData.verbosity  = pHeader->GetVerbosity();

        LogPacketParser::ParseTextLogWithContext(pPacket, packetSize,
            [](const char* pText, size_t textSize, void* argument)
            {
                auto& logMetaData = *static_cast<diag::LogMetaData*>(argument);
                diag::detail::PutImpl(logMetaData, pText, textSize);
            }, &logMetaData);
#endif
    }

    void PutLogToTargetManager(const sf::InBuffer& message) NN_NOEXCEPT
    {
        bool isPushSuccess;

        if (IsFlushAvailable())
        {
            // フラッシュ不可能になると、ブロッキングが解除されて false が返る。
            isPushSuccess = LogBuffer::GetDefaultInstance().Push(message.GetPointerUnsafe(), message.GetSize());
        }
        else
        {
            isPushSuccess = LogBuffer::GetDefaultInstance().TryPush(message.GetPointerUnsafe(), message.GetSize());
        }

        if (!isPushSuccess)
        {
            EventLogTransmitter::GetDefaultInstance().IncreaseLogPacketDropCount();
        }
    }

    void PutLogToCustomSink(const sf::InBuffer& message)
    {
        LogPacketParser::ParseTextLogWithContext(message.GetPointerUnsafe(), message.GetSize(),
            [](const char* pText, size_t textSize, void*)
            {
                // バッファがいっぱいの場合にブロックすると他のログも出なくなるのでブロックしない
                auto isPushSuccess = LogGetterImpl::GetBuffer().TryPush(pText, textSize);

                if (!isPushSuccess)
                {
                    LogGetterImpl::IncreaseLogPacketDropCount();
                }
            }, nullptr);
    }

    bool SetProcessId(const sf::InBuffer& message, nn::Bit64 processId) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_ALIGNED(message.GetPointerUnsafe(), NN_ALIGNOF(detail::LogPacketHeader));

        auto pHeader = const_cast<detail::LogPacketHeader*>(
            reinterpret_cast<const detail::LogPacketHeader*>(message.GetPointerUnsafe()));

        if (detail::LogPacketHeaderSize + pHeader->GetPayloadSize() != message.GetSize())
        {
            return false;
        }

        pHeader->SetProcessId(processId);

        return true;
    }
}

LoggerImpl::LoggerImpl(LogServiceImpl* parent, Bit64 processId) NN_NOEXCEPT
    : m_Parent(parent, true)
    , m_ProcessId(processId)
{
    EventLogTransmitter::GetDefaultInstance().PushLogSessionBegin(m_ProcessId);
}

LoggerImpl::~LoggerImpl() NN_NOEXCEPT
{
    EventLogTransmitter::GetDefaultInstance().PushLogSessionEnd(m_ProcessId);
}

Result LoggerImpl::Log(sf::InBuffer message) NN_NOEXCEPT
{
    if (!SetProcessId(message, m_ProcessId))
    {
        return ResultSuccess(); // 実装は常に成功を返す
    }

    if (g_LogDestination & LogDestination::LogDestination_TargetManager)
    {
        PutLogToTargetManager(message);
    }

    if (g_LogDestination & LogDestination::LogDestination_Uart)
    {
        PutLogToUart(message);
    }
    else if (IsFlushAvailable() && (g_LogDestination & LogDestination::LogDestination_UartIfSleep))
    {
        // スリープ中に UART に出す場合
        PutLogToUart(message);
    }

    if (g_IsLoggingToCustomSink)
    {
        PutLogToCustomSink(message);
    }

    return ResultSuccess(); // 実装は常に成功を返す
}

Result LoggerImpl::SetDestination(Bit32 destination) NN_NOEXCEPT
{
    g_LogDestination = destination;
    return ResultSuccess();
}

}}} // nn::lm::impl
