﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <nn/nn_Common.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_ConditionVariable.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Thread.h>

namespace nn { namespace lm { namespace impl {

/*!
    @brief  Log Server との接続状態を抽象化します。
*/
class LogServerProxy
{
private:
    NN_DISALLOW_COPY( LogServerProxy );
    NN_DISALLOW_MOVE( LogServerProxy );

private:
    /*!
        @brief      コンストラクタです。
    */
    LogServerProxy() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static LogServerProxy& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(LogServerProxy, instance);
        return instance;
    }

public:
    /*!
        @brief      Log Server との接続状況が変化したときの呼ばれるオブザーバーです。

        @param[in]  connected   接続時は true、切断時は false。
    */
    typedef void (*ConnectionObserver)(bool connected);

public:
    /*!
        @brief      本クラスを開始させます。
    */
    void Start() NN_NOEXCEPT;

    /*!
        @brief      本クラスを中断させます。
    */
    void Stop() NN_NOEXCEPT;

    /*!
        @brief      Log Server との接続状況を取得します。

        @return     Log Server と接続時に true、切断時に false。
    */
    bool IsConnected() NN_NOEXCEPT;

    /*!
        @brief      Log Server との接続を待機します。
    */
    void WaitToConnect() NN_NOEXCEPT;

    /*!
        @brief      Log Server との接続状況が変化したときに呼ばれるオブザーバーを登録します。

        @param[in]  observer    登録するオブザーバー。

        @detail
                    既にオブザーバーが登録されている場合は、上書きします。@n
                    引数に nullptr を指定すると、オブザーバーの登録を解除します。@n
                    オブザーバーの型は @ref ConnectionObserver を参照してください。
    */
    void SetConnectionObserver(ConnectionObserver observer) NN_NOEXCEPT;

    /*!
        @brief      Log Server にデータを送信します。

        @param[in]  data    送信するデータのポインタ。
        @param[in]  size    送信するデータのサイズ。

        @return     成功時は true, 失敗時は false。
    */
    bool Send(const uint8_t* data, size_t size) NN_NOEXCEPT;


private:
    void LoopAuto() NN_NOEXCEPT;
    //
    void SignalConnection() NN_NOEXCEPT;
    //
    void InvokeConnectionObserver(bool connected) NN_NOEXCEPT;

private:
    std::aligned_storage<4 * 1024, nn::os::ThreadStackAlignment>::type m_ThreadStack;
    nn::os::ThreadType m_Thread;
    //
    nn::os::ConditionVariable m_ConditionConnected;
    //
    nn::os::Event m_StopEvent;
    //
    nn::os::Mutex m_ConnectionMutex;
    nn::os::Mutex m_ObserverMutex;
    //
    std::atomic<int> m_ServerSocket;
    std::atomic<int> m_ClientSocket;
    //
    ConnectionObserver m_ConnectionObserver;

};

}}} // nn::lm::impl
