﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <type_traits>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/psc.h>
#include <nn/lm/detail/lm_ILogService.h>
#include <nn/lm/detail/lm_ILogGetter.h>
#include <nn/sf/sf_HipcServer.h>
#include "../detail/lm_Log.h"
#include "../lm_CreateLogService.h"

#if !defined(NN_DETAIL_LM_FOR_PRODUCT)
#include "lm_LogBuffer.h"
#include "../lm_CreateLogGetter.h"
#endif

namespace nn { namespace lm {

extern const char* LogServiceName;
extern const char* LogGetterServiceName;

namespace impl {

namespace {
    const size_t DefaultSessionCountMax = 30;
    const size_t LogGetterSettionCountMax = 1;
    const size_t SessionCountMax = DefaultSessionCountMax + LogGetterSettionCountMax;
    const size_t PortCountMax = 2;

    struct ServerManagerOptions
    {
        static const size_t PointerTransferBufferSize = 1024;
    };

    class ServerManager
        : public sf::HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, ServerManagerOptions>
    {
    };

    std::aligned_storage<sizeof(ServerManager), NN_ALIGNOF(ServerManager)>::type g_LogServerManagerStorage;
    ServerManager* g_pServerManager;

    nn::psc::PmModule           g_PmModule;
    nn::os::MultiWaitHolderType g_PmModuleEventHolder;

    std::atomic<bool> g_IsSleeping(false);
}

bool IsSleeping() NN_NOEXCEPT
{
    return g_IsSleeping;
}

void InitializeIpcServer() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(!g_pServerManager);

#if defined(NN_DETAIL_LM_FOR_PRODUCT)
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_PmModule.Initialize(
            nn::psc::PmModuleId_Lm, nullptr, 0, nn::os::EventClearMode_ManualClear));
#else
    const nn::psc::PmModuleId dependencies[] = { nn::psc::PmModuleId_TmHostIo, nn::psc::PmModuleId_Fs };
    const auto dependencyCount = sizeof(dependencies) / sizeof(dependencies[0]);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_PmModule.Initialize(
            nn::psc::PmModuleId_Lm, dependencies, dependencyCount, nn::os::EventClearMode_ManualClear));
#endif

    nn::os::InitializeMultiWaitHolder(
        &g_PmModuleEventHolder, g_PmModule.GetEventPointer()->GetBase());

    nn::os::SetMultiWaitHolderUserData(
        &g_PmModuleEventHolder, nn::psc::PmModuleId_Lm);

    g_pServerManager = new (&g_LogServerManagerStorage) ServerManager;
    g_pServerManager->AddUserWaitHolder(&g_PmModuleEventHolder);
    g_pServerManager->RegisterObjectForPort(CreateLogServiceByDfc(), DefaultSessionCountMax, LogServiceName);
#if !defined(NN_DETAIL_LM_FOR_PRODUCT)
    g_pServerManager->RegisterObjectForPort(CreateLogGetterByDfc(), LogGetterSettionCountMax, LogGetterServiceName);
#endif
    g_pServerManager->Start();
}

void LoopIpcServer() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pServerManager);

    auto previousState = nn::psc::PmState_Unknown;

    while (NN_STATIC_CONDITION(true))
    {
        auto p = g_pServerManager->Wait();

        if (p == nullptr)
        {
            break;
        }

        switch (nn::os::GetMultiWaitHolderUserData(p))
        {
        case ServerManager::HipcSimpleAllInOneServerManager::AcceptTag:
        case ServerManager::HipcSimpleAllInOneServerManager::InvokeTag:
            {
                g_pServerManager->ProcessAuto(p);
            }
            break;

        case nn::psc::PmModuleId_Lm:
            {
                g_PmModule.GetEventPointer()->Clear();
                g_pServerManager->AddUserWaitHolder(&g_PmModuleEventHolder);

                nn::psc::PmState    state;
                nn::psc::PmFlagSet  flags;

                NN_ABORT_UNLESS_RESULT_SUCCESS(
                    g_PmModule.GetRequest(&state, &flags));

                if (previousState == nn::psc::PmState_EssentialServicesAwake &&
                    state         == nn::psc::PmState_MinimumAwake)
                {
                    NN_DETAIL_LM_INFO("Leave sleep mode.\n");
                    g_IsSleeping = false;
                }
                else if (
                    previousState == nn::psc::PmState_MinimumAwake &&
                    state         == nn::psc::PmState_SleepReady)
                {
                    NN_DETAIL_LM_INFO("Enter sleep mode.\n");
                    g_IsSleeping = true;
                }
                else if (
                    state == nn::psc::PmState_ShutdownReady)
                {
                    NN_DETAIL_LM_INFO("Enter sleep mode.\n");
                    g_IsSleeping = true;
                }

                previousState = state;

                g_PmModule.Acknowledge(state, nn::ResultSuccess());
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

void StopIpcServer() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pServerManager);

    g_pServerManager->RequestStop();
}

void FinalizeIpcServer() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pServerManager);

    g_pServerManager->~ServerManager();
    g_pServerManager = nullptr;
}

} // impl

}} // nn::lm
