﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include "lm_CustomSinkBuffer.h"
#include "lm_LogGetterImpl.h"

namespace nn { namespace lm { namespace impl {

CustomSinkBuffer::CustomSinkBuffer(void* buffer, size_t bufferSize, FlushFunction function) NN_NOEXCEPT :
    m_Buffer(reinterpret_cast<uint8_t*>(buffer)),
    m_BufferSize(bufferSize),
    m_FlushFunction(function)
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER(m_BufferSize, 0u);
    NN_SDK_REQUIRES_NOT_NULL(function);
}

// @brief       バッファにログを格納します。
// @detail      格納に成功した場合は true を返します。
//              バッファに空きがない場合は失敗して false を返します。
bool CustomSinkBuffer::TryPush(const void* pSrc, size_t srcSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(srcSize, m_BufferSize);
    NN_SDK_REQUIRES(pSrc || srcSize == 0u);

    if (srcSize == 0u)
    {
        return true;
    }

    if(!(srcSize <= m_BufferSize - m_UsedBufferSize))
    {
        return false;
    }

    std::memcpy(m_Buffer + m_UsedBufferSize, pSrc, srcSize);
    m_UsedBufferSize += srcSize;

    return true;
}

// @brief       バッファリングされたログをフラッシュします。
// @detail      フラッシュに成功した場合は true を返します。
//              バッファにログがないは、失敗して false を返します。
bool CustomSinkBuffer::TryFlush() NN_NOEXCEPT
{
    if (!(m_UsedBufferSize > 0u))
    {
        return false;
    }

    if (!m_FlushFunction(m_Buffer, m_UsedBufferSize))
    {
        return false;
    }

    m_UsedBufferSize = 0;

    return true;
}

}}} // nn::lm::impl
