﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <type_traits>
#include <nn/os.h>
#include <nn/lmem/lmem_UnitHeap.h>

namespace nn { namespace lm { namespace detail {

template <size_t UnitSize, int UnitCount, int Alignment>
class UnitHeapAllocator
{
public:
    static UnitHeapAllocator& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(UnitHeapAllocator, s_Instance);
        return s_Instance;
    }

private:
    UnitHeapAllocator() NN_NOEXCEPT
    {
        nn::os::InitializeSemaphore(&m_Semaphore, UnitCount, UnitCount);
        m_HeapHandle = lmem::CreateUnitHeap(&m_Storage, StorageSize, UnitSize, lmem::CreationOption_ThreadSafe, Alignment, lmem::InfoPlacement_Tail);
        NN_SDK_ASSERT_NOT_NULL(m_HeapHandle);
        NN_SDK_ASSERT_EQUAL(lmem::GetUnitHeapAllocatableCount(m_HeapHandle), UnitCount);
    }

public:
    ~UnitHeapAllocator() NN_NOEXCEPT
    {
        nn::lmem::DestroyUnitHeap(m_HeapHandle);
        nn::os::FinalizeSemaphore(&m_Semaphore);
    }

    void* AllocateSync() NN_NOEXCEPT
    {
        nn::os::AcquireSemaphore(&m_Semaphore);
        auto buffer = lmem::AllocateFromUnitHeap(m_HeapHandle);
        NN_SDK_ASSERT_NOT_NULL(buffer);
        return buffer;
    }

    void Free(void* buffer) NN_NOEXCEPT
    {
        lmem::FreeToUnitHeap(m_HeapHandle, buffer);
        nn::os::ReleaseSemaphore(&m_Semaphore);
    }

private:
    static const size_t StorageSize = (UnitSize * UnitCount) + sizeof(nn::lmem::HeapCommonHead);
    typename std::aligned_storage<StorageSize, Alignment>::type m_Storage;

    nn::lmem::HeapHandle    m_HeapHandle;
    nn::os::SemaphoreType   m_Semaphore;
};

}}} // nn::lm::detail
