﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/settings/system/settings_Language.h>

#include <nn/nn_Abort.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_Endian.h>
#include <nn/util/util_ResEndian.h>

#include <nn/lidbe/lidbe_IdbeAccessorApi.h>


// WUPのデータサイズ
const int IdbeWupIconDataSize = 65536;
// CTRのデータサイズ
const int IdbeCtrIconDataSize = 4608;

namespace nn { namespace lidbe {

//LangageCode変換関数
inline util::optional<settings::Language> ToLanguageCode(const settings::LanguageCode& language) NN_NOEXCEPT
{
    if( language == settings::Language_Japanese)             return settings::Language_Japanese;
    if( language == settings::Language_AmericanEnglish)      return settings::Language_AmericanEnglish;
    if( language == settings::Language_French)               return settings::Language_French;
    if( language == settings::Language_German)               return settings::Language_German;
    if( language == settings::Language_Italian)              return settings::Language_Italian;
    if( language == settings::Language_Spanish)              return settings::Language_Spanish;
    if( language == settings::Language_Korean)               return settings::Language_AmericanEnglish;
    if( language == settings::Language_Dutch)                return settings::Language_Dutch;
    if( language == settings::Language_Portuguese)           return settings::Language_Portuguese;
    if( language == settings::Language_Russian)              return settings::Language_Russian;
    if( language == settings::Language_BritishEnglish)       return settings::Language_AmericanEnglish;
    if( language == settings::Language_CanadianFrench)       return settings::Language_French;
    if( language == settings::Language_LatinAmericanSpanish) return settings::Language_Spanish;
    if( language == settings::Language_SimplifiedChinese)    return settings::Language_AmericanEnglish;
    if( language == settings::Language_TraditionalChinese)   return settings::Language_AmericanEnglish;

    return settings::Language_AmericanEnglish;
}

//タイトル取得関数
const ApplicationTitle& ApplicationIdbeProperty::GetDefaultTitle() const NN_NOEXCEPT
{
    settings::LanguageCode language;
    settings::GetLanguageCode(&language);
    auto& t = title[static_cast<int>(ToLanguageCode(language).value())];

    if (t.shortName[0] != NULL)
    {
        return t;
    }

    return title[static_cast<int>(settings::Language_AmericanEnglish)];
}

void ApplicationIdbeAccessor::Initialize(void* data, size_t size) NN_NOEXCEPT
{
    ApplicationIdbeAccessor::m_Data = data;
    ApplicationIdbeAccessor::m_Size = size;

    if (ApplicationIdbeAccessor::m_Size == IdbeSizeWup)
    {
        struct WupIconDataBase* pWupIdbe = static_cast<struct WupIconDataBase*>(ApplicationIdbeAccessor::m_Data);

        uint16_t* pString = reinterpret_cast<uint16_t*>(pWupIdbe->info);          //!<WUPのEndianの変換（タイトル情報）
        for (uint32_t count = 0; count < sizeof(pWupIdbe->info) / sizeof(uint16_t); ++count)
        {
            nn::util::SwapEndian(pString);
            pString++;
        }
    }
    NN_ABORT_UNLESS((ApplicationIdbeAccessor::m_Size == IdbeSizeWup)||(ApplicationIdbeAccessor::m_Size == IdbeSizeCtr));
}

//アプリケーション管理データ取得関数
const ApplicationIdbeProperty& ApplicationIdbeAccessor::GetProperty() const NN_NOEXCEPT
{

    if (ApplicationIdbeAccessor::m_Size == IdbeSizeWup)
    {
        struct WupIconDataBase* pWupIdbe = static_cast<struct WupIconDataBase*>(m_Data);
        return *reinterpret_cast<ApplicationIdbeProperty*>(pWupIdbe->info);
    }
    else
    {
        struct CtrIconDataBase* pCtrIdbe = static_cast<struct CtrIconDataBase*>(m_Data);
        return *reinterpret_cast<ApplicationIdbeProperty*>(pCtrIdbe->info);
    }
}

//アプリケーションのアイコンデータ取得関数
const void* ApplicationIdbeAccessor::GetIconData() const NN_NOEXCEPT
{
    if (ApplicationIdbeAccessor::m_Size == IdbeSizeWup)
    {
        struct WupIconDataBase* pWupIdbe = static_cast<struct WupIconDataBase*>(m_Data);
        return reinterpret_cast<void*>(pWupIdbe->data.image);
    }
    else
    {
        struct CtrIconDataBase* pCtrIdbe = static_cast<struct CtrIconDataBase*>(m_Data);
        return reinterpret_cast<void*>(pCtrIdbe->data.bigIcon);
    }
}

//アプリケーションのアイコンサイズ取得関数
size_t ApplicationIdbeAccessor::GetIconSize() const NN_NOEXCEPT
{
    if (ApplicationIdbeAccessor::m_Size == IdbeSizeWup)
    {
        return IdbeWupIconDataSize;
    }
    else
    {
        return IdbeCtrIconDataSize;
    }
}


}}
