﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Mutex.h>
#include <nn/sf/sf_Types.h>                  // for nn::sf::SharedPointer
#include <nn/sf/sf_NativeHandle.h>

#include <nn/lbl/lbl.h>
#include <nn/lbl/lbl_SettingsForVrMode.h>
#include <nn/lbl/lbl_Type.h>
#include <nn/lbl/lbl_VrMode.h>

#include <nn/lbl/detail/lbl_ILblController.h>
#include "detail/lbl_ServiceType.h"

#include "lbl_CreateLblControllerByHipc.h"
#include "server/lbl_CreateLblControllerByDfc.h"

#define NN_LBL_ACCESS_BY_HIPC

namespace nn { namespace lbl {

namespace {

// Shim ライブラリ実装用のサービスオブジェクトへの共有ポインタ
// DFC と HIPC とで共通でよい。
nn::sf::SharedPointer<nn::lbl::detail::ILblController> g_LblController;

// Initialize の参照カウント
int g_InitializeCount = 0;

// 参照カウントを守る Mutex
nn::os::Mutex g_InitializeCountMutex(false);

}

void Initialize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_InitializeCountMutex);

    if(g_InitializeCount == 0)
    {

        NN_SDK_ASSERT(!g_LblController);

        // マクロによって、HIPC/DFC を切り替える
        #ifdef NN_LBL_ACCESS_BY_HIPC
            g_LblController = nn::lbl::CreateLblControllerByHipc();
        #else
            g_LblController = nn::lbl::CreateLblControllerByDfc();
        #endif
    }

    g_InitializeCount++;
}

// 以下は DFC と HIPC とで共通コード
void Finalize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount > 0);

    g_InitializeCount--;

    if(g_InitializeCount == 0)
    {

        NN_SDK_ASSERT(g_LblController);
        // 共有ポインタへの nullptr の代入で解放できる。
        g_LblController = nullptr;
    }
}

void SaveCurrentSetting() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    g_LblController->SaveCurrentSetting();
}

void LoadCurrentSetting() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    g_LblController->LoadCurrentSetting();
}

void SetCurrentBrightnessSetting(float brightnessSetting) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    g_LblController->SetCurrentBrightnessSetting(brightnessSetting);
}

float GetCurrentBrightnessSetting() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    return g_LblController->GetCurrentBrightnessSetting();
}

void SetCurrentBrightnessSettingForVrMode(float brightnessSetting) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    g_LblController->SetCurrentBrightnessSettingForVrMode(brightnessSetting);
}

float GetCurrentBrightnessSettingForVrMode() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    return g_LblController->GetCurrentBrightnessSettingForVrMode();
}

void ApplyCurrentBrightnessSettingToBacklight() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    g_LblController->ApplyCurrentBrightnessSettingToBacklight();
}

float GetBrightnessSettingAppliedToBacklight() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    return g_LblController->GetBrightnessSettingAppliedToBacklight();
}

void SwitchBacklightOn(nn::TimeSpan fadeInTime) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    g_LblController->SwitchBacklightOn(fadeInTime);
}

void SwitchBacklightOff(nn::TimeSpan fadeOutTime) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    g_LblController->SwitchBacklightOff(fadeOutTime);
}

BacklightSwitchStatus GetBacklightSwitchStatus() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    return static_cast<BacklightSwitchStatus>(g_LblController->GetBacklightSwitchStatus());
}

void EnableDimming() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    g_LblController->EnableDimming();
}

void DisableDimming() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    g_LblController->DisableDimming();
}

bool IsDimmingEnabled() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    return g_LblController->IsDimmingEnabled();
}

void EnableVrMode() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    g_LblController->EnableVrMode();
}

void DisableVrMode() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    g_LblController->DisableVrMode();
}

bool IsVrModeEnabled() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    return g_LblController->IsVrModeEnabled();
}

void EnableAutoBrightnessControl() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    g_LblController->EnableAutoBrightnessControl();
}

void DisableAutoBrightnessControl() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    g_LblController->DisableAutoBrightnessControl();
}

bool IsAutoBrightnessControlEnabled() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    return g_LblController->IsAutoBrightnessControlEnabled();
}

void SetAmbientLightSensorValue(float sensorValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    g_LblController->SetAmbientLightSensorValue(sensorValue);
}

float GetAmbientLightSensorValue(bool* pOutOverflowAlert) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    return g_LblController->GetAmbientLightSensorValue(pOutOverflowAlert);
}

bool IsAmbientLightSensorAvailable() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    return g_LblController->IsAmbientLightSensorAvailable();
}

void SetBrightnessReflectionDelayLevel(BrightnessChangeDirection direction, float delayLevel) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    g_LblController->SetBrightnessReflectionDelayLevel(direction, delayLevel);
}

float GetBrightnessReflectionDelayLevel(BrightnessChangeDirection direction) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    return g_LblController->GetBrightnessReflectionDelayLevel(direction);
}

void SetCurrentBrightnessMapping(const MappingCoefficient& brightnessMapping) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    g_LblController->SetCurrentBrightnessMapping(brightnessMapping);
}

void GetCurrentBrightnessMapping(MappingCoefficient* pBrightnessMapping) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    nn::sf::Out<MappingCoefficient> outMapping(pBrightnessMapping);
    g_LblController->GetCurrentBrightnessMapping(outMapping);
}

void SetCurrentAmbientLightSensorMapping(const MappingCoefficient& sensorMapping) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    g_LblController->SetCurrentAmbientLightSensorMapping(sensorMapping);
}

void GetCurrentAmbientLightSensorMapping(MappingCoefficient* pSensorMapping) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_LblController);
    nn::sf::Out<MappingCoefficient> outMapping(pSensorMapping);
    g_LblController->GetCurrentAmbientLightSensorMapping(outMapping);
}

}}
