﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/settings/system/settings_Backlight.h>

#include "lbl_FirmwareDebugSettingsAccessor.h"

namespace nn { namespace lbl { namespace impl { namespace detail {

class SettingsHolder
{
    NN_DISALLOW_COPY(SettingsHolder);
    NN_DISALLOW_MOVE(SettingsHolder);

public:
    SettingsHolder() NN_NOEXCEPT;

    //! ファームウェアデバッグ設定をロードして現在の設定に反映
    void LoadFirmwareDebugSettings() NN_NOEXCEPT;

    //! 本体設定をロードして現在の設定に反映
    void LoadSystemSettings() NN_NOEXCEPT;

    //! 現在の本体設定を保存し永続化
    void SaveSystemSettings() NN_NOEXCEPT;

    //! 自動輝度調整の有効状態
    bool IsAutoBrightnessEnabled() const NN_NOEXCEPT;
    void EnableAutoBrightness(bool enable) NN_NOEXCEPT;

    //! 輝度設定値
    float GetBrightness() const NN_NOEXCEPT;
    void SetBrightness(float brightness) NN_NOEXCEPT;

    //! VR モードにおける輝度設定値
    float GetBrightnessForVrMode() const NN_NOEXCEPT;
    void SetBrightnessForVrMode(float brightness) NN_NOEXCEPT;

    //! 輝度レベルからバックライト出力への変換係数 (二次関数)
    const nn::util::Float3& GetBrightnessCoefficients() const NN_NOEXCEPT;
    void SetBrightnessCoefficients(const nn::util::Float3& brightnessCoefficients) NN_NOEXCEPT;

#if 0 // 使用されなくなったフィールド
    //! 明るさセンサー値から輝度レベルへの変換係数 (二次関数)
    const nn::util::Float3& GetSensorCoefficients() const NN_NOEXCEPT;
    void SetSensorCoefficients(const nn::util::Float3& sensorCoefficients) NN_NOEXCEPT;

    //! 暗い状態から明るい状態への自動輝度調整の輝度値変化の遅延量
    float GetDelayDarkToBright() const NN_NOEXCEPT;
    void SetDelayDarkToBright(float delayDarkToBright) NN_NOEXCEPT;

    //! 明るい状態から暗い状態への自動輝度調整の輝度値変化の遅延量
    float GetDelayBrightToDark() const NN_NOEXCEPT;
    void SetDelayBrightToDark(float delayBrightToDark) NN_NOEXCEPT;
#endif

    //! 照度センサが使用可能か
    bool IsAlsEnabled() const NN_NOEXCEPT;

    //! 通常時輝度設定値のスケールと VR モード用輝度設定値のスケールの比率
    float GetVrBrightnessScale() const NN_NOEXCEPT;
    float GetVrLowerBrightness() const NN_NOEXCEPT;

    //! 低輝度化されているときの輝度
    float GetDimmingBrightness() const NN_NOEXCEPT;

    //! VRモード切り替え時の明るさ変化にかかるフレーム
    float GetVrBrightnessTransitionFrame() const NN_NOEXCEPT;
private:
    void LoadBrightnessMappingSettings() NN_NOEXCEPT;
    void LoadAlsEnabledSettings() NN_NOEXCEPT;
    void LoadVrBrightnessScaleSettings() NN_NOEXCEPT;
    void LoadDimmingBrightnessSettings() NN_NOEXCEPT;
    void LoadVrBrightnessTransitionFrameSettings() NN_NOEXCEPT;
private:
    // 本体設定のデータ
    nn::settings::system::BacklightSettingsEx m_SystemSettings = {};

    // ファームウェアデバッグ設定のアクセサとデータ
    FirmwareDebugSettingsAccessor m_FirmwareDebugSettingsAccessor;
    bool m_IsDebugBrightnessMappingEnabled = false;
    nn::util::Float3 m_DebugBrightnessCoefficients;
    bool m_IsFirmwareDebugSettingsOnceLoaded = false;
    bool m_IsAlsEnabled = true;
    float m_VrBrightnessScale = 0.593f;
    float m_VrLowerBrightness = 0.02f;
    float m_DimmingBrightness = 0.150f;
    float m_VrBrightnessTransitionFrame = 10.0f;
};

}}}} // namespace nn::lbl::impl::detail
