﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>

#include "lbl_SettingsHolder.h"

namespace nn { namespace lbl { namespace impl { namespace detail {

SettingsHolder::SettingsHolder() NN_NOEXCEPT
{
}

void SettingsHolder::LoadBrightnessMappingSettings() NN_NOEXCEPT
{
    m_FirmwareDebugSettingsAccessor.ReadSetting(
        &m_IsDebugBrightnessMappingEnabled, "debug_brightness_mapping_enabled");
    if (m_IsDebugBrightnessMappingEnabled)
    {
        int coefficient = 0;
        m_FirmwareDebugSettingsAccessor.ReadSetting(&coefficient,
            "debug_brightness_mapping_1st_coefficient");
        m_DebugBrightnessCoefficients.v[0] = static_cast<float>(coefficient) / 1000.0f;
        m_FirmwareDebugSettingsAccessor.ReadSetting(&coefficient,
            "debug_brightness_mapping_2nd_coefficient");
        m_DebugBrightnessCoefficients.v[1] = static_cast<float>(coefficient) / 1000.0f;
        m_FirmwareDebugSettingsAccessor.ReadSetting(&coefficient,
            "debug_brightness_mapping_3rd_coefficient");
        m_DebugBrightnessCoefficients.v[2] = static_cast<float>(coefficient) / 1000.0f;
        NN_DETAIL_LBL_TRACE("Debug brightness mapping coefficients (%f, %f, %f) is used.\n",
            m_DebugBrightnessCoefficients.v[0],
            m_DebugBrightnessCoefficients.v[1],
            m_DebugBrightnessCoefficients.v[2]);
    }
}

void SettingsHolder::LoadAlsEnabledSettings() NN_NOEXCEPT
{
    // 照度センサが有効なプラットフォームかどうかを取得
    // fwdbg で明示的に指示されていたら、プラットフォーム構成情報は無視して強制無効化
    bool forceDisableAls = false;
    m_FirmwareDebugSettingsAccessor.ReadSetting(&forceDisableAls, "force_disable_als");
    if ( forceDisableAls )
    {
        m_IsAlsEnabled = false;
        NN_DETAIL_LBL_TRACE("Ambient light sensor is forcibly DISABLED.\n");
    }
    else
    {
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        m_FirmwareDebugSettingsAccessor.ReadSetting(&m_IsAlsEnabled, "has_als", "platformconfig");
        NN_DETAIL_LBL_TRACE("has_als : %s\n", m_IsAlsEnabled ? "true" : "false");
#else
        // Win 環境はプラットフォーム構成情報が読めることを期待せず、常時使用可能とする
        m_IsAlsEnabled = true;
#endif
    }
}

void SettingsHolder::LoadVrBrightnessScaleSettings() NN_NOEXCEPT
{
#if !defined(NN_BUILD_CONFIG_OS_WIN)
    int vrBrightnessScalePermil = 593;
    m_FirmwareDebugSettingsAccessor.ReadSetting(&vrBrightnessScalePermil, "vr_brightness_scale");
    NN_DETAIL_LBL_TRACE("vr_brightness_scale : %d [permil]\n", vrBrightnessScalePermil);
    m_VrBrightnessScale = static_cast<float>(vrBrightnessScalePermil) / 1000.0f;

    int vrLowerBrightnessPermil = 20;
    m_FirmwareDebugSettingsAccessor.ReadSetting(&vrLowerBrightnessPermil, "vr_lower_brightness");
    NN_DETAIL_LBL_TRACE("vr_lower_brightness : %d [permil]\n", vrLowerBrightnessPermil);
    m_VrLowerBrightness = static_cast<float>(vrLowerBrightnessPermil) / 1000.0f;
#else
    // Win 環境はプラットフォーム構成情報が読めることを期待しない
#endif
}

void SettingsHolder::LoadDimmingBrightnessSettings() NN_NOEXCEPT
{
#if !defined(NN_BUILD_CONFIG_OS_WIN)
    int dimmingBrightnessScalePermil = 150;
    m_FirmwareDebugSettingsAccessor.ReadSetting(&dimmingBrightnessScalePermil, "dimming_brightness");
    NN_DETAIL_LBL_TRACE("dimming_brightness : %d [permil]\n", dimmingBrightnessScalePermil);
    m_DimmingBrightness = static_cast<float>(dimmingBrightnessScalePermil) / 1000.0f;
#else
    // Win 環境はプラットフォーム構成情報が読めることを期待しない
#endif
}

void SettingsHolder::LoadVrBrightnessTransitionFrameSettings() NN_NOEXCEPT
{
    int vrBrightnessTransitionFrame = 10;
    m_FirmwareDebugSettingsAccessor.ReadSetting(&vrBrightnessTransitionFrame, "vr_brightness_transition_frame");
    m_VrBrightnessTransitionFrame = static_cast<float>(vrBrightnessTransitionFrame);
    NN_DETAIL_LBL_TRACE("vr_brightness_transition_frame : %d [frame]\n", m_VrBrightnessTransitionFrame);
}
void SettingsHolder::LoadFirmwareDebugSettings() NN_NOEXCEPT
{
    LoadBrightnessMappingSettings();
    LoadAlsEnabledSettings();
    LoadVrBrightnessScaleSettings();
    LoadDimmingBrightnessSettings();
    LoadVrBrightnessTransitionFrameSettings();

    m_IsFirmwareDebugSettingsOnceLoaded = true;
}

void SettingsHolder::LoadSystemSettings() NN_NOEXCEPT
{
    nn::settings::system::GetBacklightSettings(&m_SystemSettings);
#if defined(NN_BUILD_CONFIG_OS_WIN)
    // Win 環境では較正値を読めないので、
    // ここで元々の値 v[2] から 0.02 引いた較正値(1.0f, 0.0f, 0.0f)を割り当てる。
    m_SystemSettings.brightnessCoefficients.v[0] = 1.0f;
    m_SystemSettings.brightnessCoefficients.v[1] = 0.0f;
    m_SystemSettings.brightnessCoefficients.v[2] = 0.0f;
#else
    m_SystemSettings.brightnessCoefficients.v[2] -= 0.02f;
#endif
}

void SettingsHolder::SaveSystemSettings() NN_NOEXCEPT
{
    nn::settings::system::BacklightSettingsEx systemSettings = m_SystemSettings;
    systemSettings.brightnessCoefficients.v[2] += 0.02f;
    nn::settings::system::SetBacklightSettings(systemSettings);
}

bool SettingsHolder::IsAutoBrightnessEnabled() const NN_NOEXCEPT
{
    return m_SystemSettings.flags.Test<nn::settings::system::BacklightFlag::Automatic>();
}

void SettingsHolder::EnableAutoBrightness(bool enable) NN_NOEXCEPT
{
    m_SystemSettings.flags.Set<nn::settings::system::BacklightFlag::Automatic>(enable);
}

float SettingsHolder::GetBrightness() const NN_NOEXCEPT
{
    auto brightness = m_SystemSettings.brightness;
    // 有限値であることを保証. inf や nan が取得されないようにする
    if( !std::isfinite( brightness ) )
    {
        NN_DETAIL_LBL_WARN("Invalid value: brightness = %f (should be finite)\n.", brightness);
        brightness = 0.f;
    }
    return brightness;
}

void SettingsHolder::SetBrightness(float brightness) NN_NOEXCEPT
{
    // 有限値であることを保証. inf や nan が設定されないようにする
    if( !std::isfinite( brightness ) )
    {
        NN_DETAIL_LBL_WARN("Invalid value: brightness = %f (should be finite)\n.", brightness);
        brightness = 0.f;
    }
    m_SystemSettings.brightness = brightness;
}

float SettingsHolder::GetBrightnessForVrMode() const NN_NOEXCEPT
{
    auto brightness = m_SystemSettings.brightnessVr;
    // 有限値であることを保証. inf や nan が取得されないようにする
    if( !std::isfinite( brightness ) )
    {
        NN_DETAIL_LBL_WARN("Invalid value: brightness = %f (should be finite)\n.", brightness);
        brightness = 0.f;
    }
    return brightness;
}

void SettingsHolder::SetBrightnessForVrMode(float brightness) NN_NOEXCEPT
{
    // 有限値であることを保証. inf や nan が設定されないようにする
    if( !std::isfinite( brightness ) )
    {
        NN_DETAIL_LBL_WARN("Invalid value: brightness = %f (should be finite)\n.", brightness);
        brightness = 0.f;
    }
    m_SystemSettings.brightnessVr = brightness;
}

const nn::util::Float3& SettingsHolder::GetBrightnessCoefficients() const NN_NOEXCEPT
{
    if (m_IsDebugBrightnessMappingEnabled)
    {
        return m_DebugBrightnessCoefficients;
    }
    return m_SystemSettings.brightnessCoefficients;
}

void SettingsHolder::SetBrightnessCoefficients(const nn::util::Float3& brightnessCoefficients) NN_NOEXCEPT
{
    if (m_IsDebugBrightnessMappingEnabled)
    {
        m_DebugBrightnessCoefficients = brightnessCoefficients;
    }
    else
    {
        m_SystemSettings.brightnessCoefficients = brightnessCoefficients;
    }
}

#if 0 // 使用されなくなったフィールド
const nn::util::Float3& SettingsHolder::GetSensorCoefficients() const NN_NOEXCEPT
{
    return m_SystemSettings.sensorCoefficients;
}

void SettingsHolder::SetSensorCoefficients(const nn::util::Float3& sensorCoefficients) NN_NOEXCEPT
{
    m_SystemSettings.sensorCoefficients = sensorCoefficients;
}

float SettingsHolder::GetDelayDarkToBright() const NN_NOEXCEPT
{
    return m_SystemSettings.delayDarkToBright;
}

void SettingsHolder::SetDelayDarkToBright(float delayDarkToBright) NN_NOEXCEPT
{
    m_SystemSettings.delayDarkToBright = delayDarkToBright;
}

float SettingsHolder::GetDelayBrightToDark() const NN_NOEXCEPT
{
    return m_SystemSettings.delayBrightToDark;
}

void SettingsHolder::SetDelayBrightToDark(float delayBrightToDark) NN_NOEXCEPT
{
    m_SystemSettings.delayBrightToDark = delayBrightToDark;
}
#endif

bool SettingsHolder::IsAlsEnabled() const NN_NOEXCEPT
{
    // 最初のアクセスの前に最低でも一度はロードを試みていること
    NN_SDK_ASSERT(m_IsFirmwareDebugSettingsOnceLoaded);
    return m_IsAlsEnabled;
}

float SettingsHolder::GetVrBrightnessScale() const NN_NOEXCEPT
{
    // 最初のアクセスの前に最低でも一度はロードを試みていること
    NN_SDK_ASSERT(m_IsFirmwareDebugSettingsOnceLoaded);
    return m_VrBrightnessScale;
}

float SettingsHolder::GetVrLowerBrightness() const NN_NOEXCEPT
{
    // 最初のアクセスの前に最低でも一度はロードを試みていること
    NN_SDK_ASSERT(m_IsFirmwareDebugSettingsOnceLoaded);
    return m_VrLowerBrightness;
}

float SettingsHolder::GetDimmingBrightness() const NN_NOEXCEPT
{
    // 最初のアクセスの前に最低でも一度はロードを試みていること
    NN_SDK_ASSERT(m_IsFirmwareDebugSettingsOnceLoaded);
    return m_DimmingBrightness;
}

float SettingsHolder::GetVrBrightnessTransitionFrame() const NN_NOEXCEPT
{
    // 最初のアクセスの前に最低でも一度はロードを試みていること
    NN_SDK_ASSERT(m_IsFirmwareDebugSettingsOnceLoaded);
    return m_VrBrightnessTransitionFrame;
}
}}}} // namespace nn::lbl::impl::detail
