﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/lbl/lbl_Type.h>

#include "lbl_DeviceAccessorBacklight.h"
#include "lbl_Fader.h"
#include "lbl_AutoBrightnessAdjuster.h"
#include "lbl_SettingsHolder.h"

namespace nn { namespace lbl { namespace impl { namespace detail {

class BrightnessController
{
    NN_DISALLOW_COPY(BrightnessController);

public:
    explicit BrightnessController(const SettingsHolder* pSettingsHolder) NN_NOEXCEPT;

    void Initialize() NN_NOEXCEPT;

    void Finalize() NN_NOEXCEPT;

    void StartFade(bool isOn, nn::TimeSpan fadeTime) NN_NOEXCEPT;

    nn::lbl::BacklightSwitchStatus GetFadeState() const NN_NOEXCEPT
    {
        return m_Fader.GetState();
    }

    void EnableDimming(bool enable) NN_NOEXCEPT
    {
        m_IsDimmingEnabled = enable;
    }

    bool IsDimmingEnabled() const NN_NOEXCEPT
    {
        return m_IsDimmingEnabled;
    }

    void EnableVrMode(bool enable) NN_NOEXCEPT
    {
        m_IsInVrMode = enable;
    }

    bool IsVrModeEnabled() const NN_NOEXCEPT
    {
        return m_IsInVrMode;
    }

    float GetAppliedBrightness() const NN_NOEXCEPT
    {
        return m_AppliedBrightness;
    }

    void UpdateAndApplyBrightness(float illuminanceLux) NN_NOEXCEPT;

    void ApplyImmediatelyNext() NN_NOEXCEPT;

private:
    int TransformBrightnessToDuty(double rawBrightness, double activeRatio) NN_NOEXCEPT;
    bool IsAutoAdjustEnabled() const NN_NOEXCEPT;
    //! VRモード時の輝度を計算します
    double CalcVrModeBrightness() const NN_NOEXCEPT;
    //! VRモード切替中の明るさの変化率を更新します
    double UpdateBrightnessTransitionRatio() NN_NOEXCEPT;

private:
    //! 低輝度化モードが有効か
    bool m_IsDimmingEnabled = false;

    //! 現在の適用輝度値
    float m_AppliedBrightness = 0.0f;

    //! 次回の輝度反映で即座にターゲット輝度に合わせるフラグ
    bool m_ApplyImmediatelyNext = false;

    //! 制御モード
    bool m_IsInVrMode = false;

    //! 設定値へのアクセサ (参照のみ)
    const SettingsHolder* m_pSettingsHolder = nullptr;

    //! フェードイン・アウト演出のマネージャ
    Fader m_Fader;

    //! バックライトアクセサ
    DeviceAccessorBacklight m_DeviceAccessorBacklight;

    //! 照度値に基づく自動輝度調整計算器
    AutoBrightnessAdjuster m_AutoBrightnessAdjuster;

    //! VRモードへの変化量 (0.0 ~ 1.0)
    double m_VrBrightnessTransitionRatio = 0.0;
};

}}}} // namespace nn::lbl::impl::detail
