﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Web 認証アプレットの起動時引数の受け取りをおこなうクラスの実装
 */

#include <nn/la/la_NifmToWifiWebAuthArgumentsReader.h>

#include <nn/applet/applet.h>
#include <nn/ae/ae.h>

#include <nn/util/util_ScopeExit.h>

#include <nn/nn_Macro.h>
#include <nn/nn_SdkLog.h>

#include <nn/la/la_Configs.h>
#include <nn/la/la_Result.h>

#include <algorithm>
#include <cstring>


#define NN_DETAIL_LA_RETURN_FALSE_UNLESS_PROPER_SIZE(nifmToWifiWebAuthVersion, storageSize, handlingNifmToWifiWebAuthVersion)             \
    if (nifmToWifiWebAuthVersion == handlingNifmToWifiWebAuthVersion)                                                                     \
    {                                                                                                                                     \
        NN_SDK_ASSERT_EQUAL(storageSize, sizeof(NN_DETAIL_LA_NIFMTOWIFIWEBAUTH_ARGUMENTS_DATA_TYPE(handlingNifmToWifiWebAuthVersion)));   \
        if (storageSize != sizeof(NN_DETAIL_LA_NIFMTOWIFIWEBAUTH_ARGUMENTS_DATA_TYPE(handlingNifmToWifiWebAuthVersion)))                  \
        {                                                                                                                                 \
            return false;                                                                                                                 \
        }                                                                                                                                 \
    }                                                                                                                                     \
    else

#define NN_DETAIL_LA_RETURN_FALSE_UNLESS_PROPER_SIZE_FOR_FUTURE_VERSIONS(nifmToWifiWebAuthVersion, storageSize)               \
    {                                                                                                                         \
        NN_SDK_ASSERT_GREATER(nifmToWifiWebAuthVersion, NN_DETAIL_LA_CURRENT_NIFMTOWIFIWEBAUTH_VERSION);                      \
        NN_SDK_ASSERT_GREATER_EQUAL(storageSize, sizeof(NN_DETAIL_LA_CURRENT_NIFMTOWIFIWEBAUTH_ARGUMENTS_DATA_TYPE));         \
        if (storageSize < sizeof(NN_DETAIL_LA_CURRENT_NIFMTOWIFIWEBAUTH_ARGUMENTS_DATA_TYPE))                                 \
        {                                                                                                                     \
            return false;                                                                                                     \
        }                                                                                                                     \
    }


namespace nn { namespace la {

NifmToWifiWebAuthArgumentsReader::NifmToWifiWebAuthArgumentsReader() NN_NOEXCEPT
    : m_IsInitialized(false)
{
    std::memset(m_NifmToWifiWebAuthArgumentsData.authUrl, 0, sizeof(m_NifmToWifiWebAuthArgumentsData.authUrl));
    std::memset(m_NifmToWifiWebAuthArgumentsData.connTestUrl, 0, sizeof(m_NifmToWifiWebAuthArgumentsData.connTestUrl));
    m_NifmToWifiWebAuthArgumentsData.profileId = nn::util::InvalidUuid;
    m_NifmToWifiWebAuthArgumentsData.submitId = 0;

    // 旧バージョンになかったパラメータは、コンストラクタで埋めたデフォルト値が用いられる
}

NifmToWifiWebAuthArgumentsReader::~NifmToWifiWebAuthArgumentsReader() NN_NOEXCEPT
{
}

const char* NifmToWifiWebAuthArgumentsReader::GetAuthUrlPointer() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);

    return m_NifmToWifiWebAuthArgumentsData.authUrl;
}

const char* NifmToWifiWebAuthArgumentsReader::GetConnTestUrlPointer() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);

    return m_NifmToWifiWebAuthArgumentsData.connTestUrl;
}

const nn::util::Uuid& NifmToWifiWebAuthArgumentsReader::GetProfileId() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);

    return m_NifmToWifiWebAuthArgumentsData.profileId;
}

uint32_t NifmToWifiWebAuthArgumentsReader::GetSubmitId() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);

    return m_NifmToWifiWebAuthArgumentsData.submitId;
}

bool NifmToWifiWebAuthArgumentsReader::TryPopFromInChannel() NN_NOEXCEPT
{
    nn::ae::StorageHandle storageHandle;

    if (!nn::ae::TryPopFromInChannel(&storageHandle))
    {
        return false;
    }

    NN_UTIL_SCOPE_EXIT
    {
        nn::applet::ReleaseStorage(storageHandle);
    };

    return this->ReadFromStorage(storageHandle);
}

bool NifmToWifiWebAuthArgumentsReader::ReadFromStorage(nn::ae::StorageHandle storageHandle) NN_NOEXCEPT
{
    size_t storageSize = nn::applet::GetStorageSize(storageHandle);

    uint32_t nifmToWifiWebAuthVersion;

    if (storageSize < sizeof(nifmToWifiWebAuthVersion))
    {
        return false;
    }

    nn::applet::ReadFromStorage(storageHandle, 0, &nifmToWifiWebAuthVersion, sizeof(nifmToWifiWebAuthVersion));

    storageSize -= sizeof(nifmToWifiWebAuthVersion);
    nn::applet::ReadFromStorage(storageHandle, sizeof(nifmToWifiWebAuthVersion), &m_NifmToWifiWebAuthArgumentsData, std::min(sizeof(m_NifmToWifiWebAuthArgumentsData), storageSize));

#if 0
    // 旧バージョンのハンドリング
    NN_DETAIL_LA_RETURN_FALSE_UNLESS_PROPER_SIZE(nifmToWifiWebAuthVersion, storageSize, 0)
    NN_DETAIL_LA_RETURN_FALSE_UNLESS_PROPER_SIZE(nifmToWifiWebAuthVersion, storageSize, 1)
#endif
    NN_DETAIL_LA_RETURN_FALSE_UNLESS_PROPER_SIZE(nifmToWifiWebAuthVersion, storageSize, NN_DETAIL_LA_CURRENT_NIFMTOWIFIWEBAUTH_VERSION)
    NN_DETAIL_LA_RETURN_FALSE_UNLESS_PROPER_SIZE_FOR_FUTURE_VERSIONS(nifmToWifiWebAuthVersion, storageSize)

    m_IsInitialized = true;

    return true;
}

}} // namespace nn::la

