﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ネット接続アプレットの起動時引数の受け取りをおこなうクラスの実装
 */

#include <nn/la/la_NifmToNetConnectArgumentsReader.h>

#include <nn/applet/applet.h>
#include <nn/ae/ae.h>

#include <nn/util/util_ScopeExit.h>

#include <nn/nn_Macro.h>
#include <nn/nn_SdkLog.h>

#include <nn/la/la_Configs.h>
#include <nn/la/la_Result.h>

#include <algorithm>
#include <cstring>


#define NN_DETAIL_LA_RETURN_FALSE_UNLESS_PROPER_SIZE(nifmToNetConnectVersion, storageSize, handlingNifmToNetConnectVersion)            \
    if (nifmToNetConnectVersion == handlingNifmToNetConnectVersion)                                                                    \
    {                                                                                                                                  \
        NN_SDK_ASSERT_EQUAL(storageSize, sizeof(NN_DETAIL_LA_NIFMTONETCONNECT_ARGUMENTS_DATA_TYPE(handlingNifmToNetConnectVersion)));  \
        if (storageSize != sizeof(NN_DETAIL_LA_NIFMTONETCONNECT_ARGUMENTS_DATA_TYPE(handlingNifmToNetConnectVersion)))                 \
        {                                                                                                                              \
            return false;                                                                                                              \
        }                                                                                                                              \
    }                                                                                                                                  \
    else

#define NN_DETAIL_LA_RETURN_FALSE_UNLESS_PROPER_SIZE_FOR_FUTURE_VERSIONS(nifmToNetConnectVersion, storageSize)              \
    {                                                                                                                       \
        NN_SDK_ASSERT_GREATER(nifmToNetConnectVersion, NN_DETAIL_LA_CURRENT_NIFMTONETCONNECT_VERSION);                      \
        NN_SDK_ASSERT_GREATER_EQUAL(storageSize, sizeof(NN_DETAIL_LA_CURRENT_NIFMTONETCONNECT_ARGUMENTS_DATA_TYPE));        \
        if (storageSize < sizeof(NN_DETAIL_LA_CURRENT_NIFMTONETCONNECT_ARGUMENTS_DATA_TYPE))                                \
        {                                                                                                                   \
            return false;                                                                                                   \
        }                                                                                                                   \
    }


namespace nn { namespace la {

NifmToNetConnectArgumentsReader::NifmToNetConnectArgumentsReader() NN_NOEXCEPT
    : m_IsInitialized(false)
{
    m_NifmToNetConnectArgumentsData.networkConnectionResult = ResultSuccess();
    m_NifmToNetConnectArgumentsData.networkProfileId = nn::util::InvalidUuid;

    // 旧バージョンになかったパラメータは、コンストラクタで埋めたデフォルト値が用いられる
}

NifmToNetConnectArgumentsReader::~NifmToNetConnectArgumentsReader() NN_NOEXCEPT
{
}

Result NifmToNetConnectArgumentsReader::GetNetworkConnectionResult() NN_NOEXCEPT
{
    return m_NifmToNetConnectArgumentsData.networkConnectionResult;
}

nn::util::Uuid NifmToNetConnectArgumentsReader::GetNetworkProfileId() NN_NOEXCEPT
{
    return m_NifmToNetConnectArgumentsData.networkProfileId;
}

bool NifmToNetConnectArgumentsReader::TryPopFromInChannel() NN_NOEXCEPT
{
    nn::ae::StorageHandle storageHandle;

    if (!nn::ae::TryPopFromInChannel(&storageHandle))
    {
        return false;
    }

    NN_UTIL_SCOPE_EXIT
    {
        nn::applet::ReleaseStorage(storageHandle);
    };

    return this->ReadFromStorage(storageHandle);
}

bool NifmToNetConnectArgumentsReader::ReadFromStorage(nn::ae::StorageHandle storageHandle) NN_NOEXCEPT
{
    size_t storageSize = nn::applet::GetStorageSize(storageHandle);

    uint32_t nifmToNetConnectVersion;

    if (storageSize < sizeof(nifmToNetConnectVersion))
    {
        return false;
    }

    nn::applet::ReadFromStorage(storageHandle, 0, &nifmToNetConnectVersion, sizeof(nifmToNetConnectVersion));

    storageSize -= sizeof(nifmToNetConnectVersion);
    nn::applet::ReadFromStorage(storageHandle, sizeof(nifmToNetConnectVersion), &m_NifmToNetConnectArgumentsData, std::min(sizeof(m_NifmToNetConnectArgumentsData), storageSize));

#if 0
    // 旧バージョンのハンドリング
    NN_DETAIL_LA_RETURN_FALSE_UNLESS_PROPER_SIZE(nifmToNetConnectVersion, storageSize, 0)
    NN_DETAIL_LA_RETURN_FALSE_UNLESS_PROPER_SIZE(nifmToNetConnectVersion, storageSize, 1)
#endif
    NN_DETAIL_LA_RETURN_FALSE_UNLESS_PROPER_SIZE(nifmToNetConnectVersion, storageSize, NN_DETAIL_LA_CURRENT_NIFMTONETCONNECT_VERSION)
    NN_DETAIL_LA_RETURN_FALSE_UNLESS_PROPER_SIZE_FOR_FUTURE_VERSIONS(nifmToNetConnectVersion, storageSize)

    m_IsInitialized = true;

    return true;
}

}} // namespace nn::la

