﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ライブラリアプレットの共通引数の受け渡しをおこなうクラスの実装
 */

#include <nn/la/la_CommonArgumentsWriter.h>

#include <nn/applet/applet.h>

#include <nn/nn_Macro.h>
#include <nn/nn_SdkLog.h>

#include <nn/la/la_Configs.h>
#include <nn/la/la_Result.h>
#include <nn/oe/oe_LibraryAppletControlTypes.h>
#include <nn/oe/oe_OperationStateControlSystem.h>

#include <cstring>

namespace nn { namespace la {

namespace {

const detail::CommonArgumentsHeader CurrentCommonArgumentsHeader =
{
    NN_DETAIL_LA_CURRENT_COMMON_VERSION_MAJOR << 16 | NN_DETAIL_LA_CURRENT_COMMON_VERSION_MINOR,
    sizeof(detail::CommonArgumentsHeader) + sizeof(detail::CommonArgumentsDataLatest)
};

}

CommonArgumentsWriter::CommonArgumentsWriter(uint32_t laVersion) NN_NOEXCEPT
{
    m_CommonArgumentsData.laVersion = laVersion;
    m_CommonArgumentsData.colorIndex = nn::oe::GetExpectedThemeColor();
    m_CommonArgumentsData.isPlayStartupSound = false;
    m_CommonArgumentsData.callerTimeStamp = nn::os::GetSystemTick().GetInt64Value();
}

CommonArgumentsWriter::~CommonArgumentsWriter() NN_NOEXCEPT
{
}

void CommonArgumentsWriter::SetLaVersion( uint16_t laVersionMajor, uint16_t laVersionMinor) NN_NOEXCEPT
{
    m_CommonArgumentsData.SetLaVersion(laVersionMajor, laVersionMinor);
}

void CommonArgumentsWriter::SetLaVersion(uint32_t laVersion) NN_NOEXCEPT
{
    m_CommonArgumentsData.laVersion = laVersion;
}

void CommonArgumentsWriter::SetPlayStartupSound(bool isPlayStartupSound) NN_NOEXCEPT
{
    m_CommonArgumentsData.isPlayStartupSound = isPlayStartupSound;
}

void CommonArgumentsWriter::PushToInChannel(nn::applet::LibraryAppletHandle libraryAppletHandle) const NN_NOEXCEPT
{
    nn::applet::StorageHandle storageHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateStorage(&storageHandle, sizeof(CurrentCommonArgumentsHeader) + sizeof(m_CommonArgumentsData)));
    this->WriteToStorage( storageHandle );
    nn::applet::PushToInChannel(libraryAppletHandle, storageHandle);
}

size_t CommonArgumentsWriter::WriteToStorage(nn::applet::StorageHandle storageHandle) const NN_NOEXCEPT
{
    m_CommonArgumentsData.callerTimeStamp = nn::os::GetSystemTick().GetInt64Value();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::WriteToStorage(storageHandle, 0, &CurrentCommonArgumentsHeader, sizeof(CurrentCommonArgumentsHeader)));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::WriteToStorage(storageHandle, sizeof(CurrentCommonArgumentsHeader), &m_CommonArgumentsData, sizeof(m_CommonArgumentsData)));

    return GetExportSize();
}

size_t CommonArgumentsWriter::GetExportSize() const NN_NOEXCEPT
{
    return sizeof(detail::CommonArgumentsHeader) + sizeof(m_CommonArgumentsData);
}

void CommonArgumentsWriter::Export(void* buffer, size_t size) const NN_NOEXCEPT
{
    uint8_t* buffer_ = static_cast<uint8_t*>(buffer);
    NN_ABORT_UNLESS_GREATER_EQUAL(size, GetExportSize());

    std::memcpy(&buffer_[0], &CurrentCommonArgumentsHeader, sizeof(CurrentCommonArgumentsHeader));
    std::memcpy(&buffer_[sizeof(CurrentCommonArgumentsHeader)], &m_CommonArgumentsData, sizeof(m_CommonArgumentsData));
}

}} // namespace nn::la

