﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ライブラリアプレットの共通引数の受け取りをおこなうクラスの実装
 */

#include <nn/la/la_CommonArgumentsReader.h>
#include <nn/la/la_CommonArgumentsWriter.h>
#include <nn/oe/oe_LibraryAppletControlApis.h>
#include <nn/oe/oe_OperationStateControlSystem.h>

#include <nn/applet/applet.h>
#include <nn/ae/ae.h>

#include <nn/util/util_ScopeExit.h>

#include <nn/nn_Macro.h>
#include <nn/nn_SdkLog.h>

#include <nn/la/la_Configs.h>
#include <nn/la/la_Result.h>
#include <nn/la/la_Api.h>

#include <algorithm>
#include <cstring>

namespace nn { namespace la {

template < typename TCommonArgument >
void
CommonArgumentsReader::UpdateCommonDataToLatest( TCommonArgument& oldVersion ) NN_NOEXCEPT
{
    // コピー回数が少なくて済むように、コピーせず単なるキャストにしておく。
    // キャストで問題が出るような更新があった場合はちゃんと互換を保つようにする。
    typename TCommonArgument::NextType& nextVersion = reinterpret_cast< typename TCommonArgument::NextType& >( oldVersion );
    // ここで何か処理が必要な場合は、そのバージョン用の特殊化を用意して実装する。
    UpdateCommonDataToLatest( nextVersion );
}

//
// 最新バージョンまで上がったら何もしない。
//
template <>
void
CommonArgumentsReader::UpdateCommonDataToLatest< detail::CommonArgumentsDataLatest >( detail::CommonArgumentsDataLatest& oldVersion ) NN_NOEXCEPT
{
}


template < typename TCommonArgument >
bool
CommonArgumentsReader::ReadCommonDataFromStorage(const detail::CommonArgumentsHeader& header, nn::ae::StorageHandle storageHandle, size_t storageSize) NN_NOEXCEPT
{
    if (header.GetMajorVersion() == TCommonArgument::cMajorVersion)
    {
        if ( sizeof(TCommonArgument) > storageSize ) { return false; }
        if ( sizeof(TCommonArgument) + sizeof(detail::CommonArgumentsHeader) > header.commonSize ) { return false; }

        std::memset( &m_CommonArgumentsData, 0, sizeof(detail::CommonArgumentsDataLatest) );
        nn::applet::ReadFromStorage( storageHandle, sizeof(detail::CommonArgumentsHeader), &m_CommonArgumentsData, sizeof(TCommonArgument) );

        UpdateCommonDataToLatest( reinterpret_cast< TCommonArgument& >( m_CommonArgumentsData ) );
    }

    return true;
}

//
// 最新バージョンの CommonArgumentsData を読む用の特殊化。
//
template <>
bool
CommonArgumentsReader::ReadCommonDataFromStorage< detail::CommonArgumentsDataLatest >(const detail::CommonArgumentsHeader& header, nn::ae::StorageHandle storageHandle, size_t storageSize) NN_NOEXCEPT
{
    // MajorVersion が自分より新しい事は市場ではありえないので、マイナーバージョン違いと同じ扱い。
    if (header.GetMajorVersion() >= detail::CommonArgumentsDataLatest::cMajorVersion)
    {
        int64_t offset = sizeof(detail::CommonArgumentsHeader);

        // マイナーバージョン違いもメンバ追加程度は互換をとれるようにする。
        // 互換がとれなくなったら ASSERT で止めてしまう。
        std::memset( &m_CommonArgumentsData, 0, sizeof(detail::CommonArgumentsDataLatest) );
        nn::applet::ReadFromStorage( storageHandle, offset, &m_CommonArgumentsData, std::min( sizeof(detail::CommonArgumentsDataLatest), storageSize ) );
        // LA 側が新しい場合は新メンバは 0 で初期化されている。
        // LA 側が古い場合は新メンバは読まれない。
    }

    return true;
}

size_t  CommonArgumentsReader::GetCommonSize() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);

    return m_CommonSizeInStorage;
}

uint32_t CommonArgumentsReader::GetLaVersion() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);

    return m_CommonArgumentsData.laVersion;
}

uint16_t CommonArgumentsReader::GetLaVersionMajor() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);

    return m_CommonArgumentsData.GetLaVersionMajor();
}

uint16_t CommonArgumentsReader::GetLaVersionMinor() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);

    return m_CommonArgumentsData.GetLaVersionMinor();
}

int32_t CommonArgumentsReader::GetColorIndex() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);

    return m_CommonArgumentsData.colorIndex;
}

bool CommonArgumentsReader::IsPlayStartupSound() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);

    return m_CommonArgumentsData.isPlayStartupSound;
}

nn::os::Tick CommonArgumentsReader::GetCallerTimeStamp() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);

    return nn::os::Tick( m_CommonArgumentsData.callerTimeStamp );
}

bool CommonArgumentsReader::TryPopFromInChannel() NN_NOEXCEPT
{
    nn::ae::StorageHandle storageHandle;

    if (!nn::ae::TryPopFromInChannel(&storageHandle))
    {
        return false;
    }

    NN_UTIL_SCOPE_EXIT
    {
        nn::applet::ReleaseStorage(storageHandle);
    };

    return this->ReadFromStorage( storageHandle );
}


bool CommonArgumentsReader::ReadFromStorage( nn::ae::StorageHandle storageHandle ) NN_NOEXCEPT
{
    size_t storageSize = nn::applet::GetStorageSize(storageHandle);

    struct detail::CommonArgumentsHeader header;

    if (storageSize < sizeof(header))
    {
        return false;
    }

    nn::applet::ReadFromStorage(storageHandle, 0, &header, sizeof(header));
    storageSize -= sizeof(header);

    m_CommonSizeInStorage = header.commonSize;

#if 0
    // 旧メジャーバージョンのハンドリング
    if ( ! this->ReadCommonDataFromStorage< detail::CommonArgumentsDataV0 >( header, storageHandle, storageSize ) ) { return false; }
    if ( ! this->ReadCommonDataFromStorage< detail::CommonArgumentsDataV1 >( header, storageHandle, storageSize ) ) { return false; }

#endif

    if ( ! this->ReadCommonDataFromStorage< detail::CommonArgumentsDataLatest >( header, storageHandle, storageSize ) ) { return false; }

    nn::oe::SetExpectedThemeColorForSystem( static_cast<nn::oe::ThemeColorType>( m_CommonArgumentsData.colorIndex ) );

    m_IsInitialized = true;

    return true;
}

void CommonArgumentsReader::Import(const void* buffer, size_t size) NN_NOEXCEPT
{
    const uint8_t* buffer_ = static_cast<const uint8_t*>(buffer);

    // 開発用途なので、とりあえずバージョン互換までチェックしない。
    NN_ABORT_UNLESS_GREATER_EQUAL(size, sizeof(detail::CommonArgumentsHeader) + sizeof(m_CommonArgumentsData));

    detail::CommonArgumentsHeader header;

    std::memcpy(&header, &buffer_[0], sizeof(header));
    std::memcpy(&m_CommonArgumentsData, &buffer_[sizeof(header)], sizeof(m_CommonArgumentsData));

    nn::oe::SetExpectedThemeColorForSystem( static_cast<nn::oe::ThemeColorType>( m_CommonArgumentsData.colorIndex ) );

    m_IsInitialized = true;
}

}} // namespace nn::la

