﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ライブラリアプレットからの応答の受け取りをおこなうクラスの実装
 */

#include <nn/la/la_AppletToNifmArgumentsReader.h>

#include <nn/nifm/nifm_Result.h>
#include <nn/nifm/nifm_ResultPrivate.h>

#include <nn/applet/applet.h>

#include <nn/util/util_ScopeExit.h>

#include <nn/nn_Macro.h>
#include <nn/nn_SdkLog.h>

#include <nn/la/la_Configs.h>
#include <nn/la/la_Result.h>

#include <algorithm>
#include <cstring>


#define NN_DETAIL_LA_RETURN_FALSE_UNLESS_PROPER_SIZE(appletToNifmVersion, storageSize, handlingAppletToNifmVersion)            \
    if (appletToNifmVersion == handlingAppletToNifmVersion)                                                                    \
    {                                                                                                                          \
        NN_SDK_ASSERT_EQUAL(storageSize, sizeof(NN_DETAIL_LA_APPLETTONIFM_ARGUMENTS_DATA_TYPE(handlingAppletToNifmVersion)));  \
        if (storageSize != sizeof(NN_DETAIL_LA_APPLETTONIFM_ARGUMENTS_DATA_TYPE(handlingAppletToNifmVersion)))                 \
        {                                                                                                                      \
            return false;                                                                                                      \
        }                                                                                                                      \
    }                                                                                                                          \
    else

#define NN_DETAIL_LA_RETURN_FALSE_UNLESS_PROPER_SIZE_FOR_FUTURE_VERSIONS(appletToNifmVersion, storageSize)          \
    {                                                                                                               \
        NN_SDK_ASSERT_GREATER(appletToNifmVersion, NN_DETAIL_LA_CURRENT_APPLETTONIFM_VERSION);                      \
        NN_SDK_ASSERT_GREATER_EQUAL(storageSize, sizeof(NN_DETAIL_LA_CURRENT_APPLETTONIFM_ARGUMENTS_DATA_TYPE));    \
        if (storageSize < sizeof(NN_DETAIL_LA_CURRENT_APPLETTONIFM_ARGUMENTS_DATA_TYPE))                            \
        {                                                                                                           \
            return false;                                                                                           \
        }                                                                                                           \
    }


namespace nn { namespace la {

AppletToNifmArgumentsReader::AppletToNifmArgumentsReader() NN_NOEXCEPT
    : m_IsInitialized(false)
{
    m_AppletToNifmArgumentsData.result = ResultSuccess();

    // 旧バージョンになかったパラメータは、コンストラクタで埋めたデフォルト値が用いられる
}

AppletToNifmArgumentsReader::~AppletToNifmArgumentsReader() NN_NOEXCEPT
{
}

Result AppletToNifmArgumentsReader::GetResult() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_AppletToNifmArgumentsData.result.IsFailure());

    return m_AppletToNifmArgumentsData.result;
}

bool AppletToNifmArgumentsReader::TryPopFromOutChannel(nn::applet::LibraryAppletHandle appletHandle) NN_NOEXCEPT
{
    nn::applet::StorageHandle storageHandle;

    if(!nn::applet::TryPopFromOutChannel(&storageHandle, appletHandle))
    {
        return false;
    }

    NN_UTIL_SCOPE_EXIT
    {
        nn::applet::ReleaseStorage(storageHandle);
    };

    return ReadFromStorage(storageHandle);
}

bool AppletToNifmArgumentsReader::ReadFromStorage(const nn::applet::StorageHandle storageHandle) NN_NOEXCEPT
{
    size_t storageSize = nn::applet::GetStorageSize(storageHandle);

    // NetConnect アプレットが，最初のストレージに書き込む sizeof(int32_t) サイズの値を
    // AppletToNifmArgumentsDataV* の代わりの値として読み込みます。
    if (storageSize == sizeof(int32_t))
    {
        return ReadNetConnectReturnParam(storageHandle);
    }

    uint32_t appletToNifmVersion;

    if (storageSize < sizeof(appletToNifmVersion))
    {
        return false;
    }

    nn::applet::ReadFromStorage(storageHandle, 0, &appletToNifmVersion, sizeof(appletToNifmVersion));

    storageSize -= sizeof(appletToNifmVersion);
    nn::applet::ReadFromStorage(storageHandle, sizeof(appletToNifmVersion), &m_AppletToNifmArgumentsData, std::min(sizeof(m_AppletToNifmArgumentsData), storageSize));

#if 0
    // 旧バージョンのハンドリング
    NN_DETAIL_LA_RETURN_FALSE_UNLESS_PROPER_SIZE(appletToNifmVersion, storageSize, 0)
    NN_DETAIL_LA_RETURN_FALSE_UNLESS_PROPER_SIZE(appletToNifmVersion, storageSize, 1)
#endif
    NN_DETAIL_LA_RETURN_FALSE_UNLESS_PROPER_SIZE(appletToNifmVersion, storageSize, NN_DETAIL_LA_CURRENT_APPLETTONIFM_VERSION)
    NN_DETAIL_LA_RETURN_FALSE_UNLESS_PROPER_SIZE_FOR_FUTURE_VERSIONS(appletToNifmVersion, storageSize)

    m_IsInitialized = true;

    return true;
}

bool AppletToNifmArgumentsReader::ReadNetConnectReturnParam(nn::applet::StorageHandle storageHandle) NN_NOEXCEPT
{
    int32_t retValue = -1;
    nn::applet::ReadFromStorage(storageHandle, 0, &retValue, sizeof(retValue));

    m_IsInitialized = true;

    // NetConnect アプレットが返す値の意味は以下の通りで、ここでは 0 をエラーハンドリング成功と見做します。
    // 0:接続成功, 1:もどる, 2:あとで, 3:サービスディスカバリ
    if (retValue == 0)
    {
        m_AppletToNifmArgumentsData.result = nn::nifm::ResultErrorHandlingCompleted();
    }
    else
    {
        m_AppletToNifmArgumentsData.result = nn::nifm::ResultInternetLayerNotAvailable();
    }

    return true;
}

}} // namespace nn::la

