﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <type_traits>

#include <nn/ro/ro_Types.h>

#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace jitsrv {

class DllPlugin
{
public:

    /*
        @pre NN_SDK_REQUIRES_ALIGNED(buffer, os::MemoryPageSize);
        @pre NN_SDK_REQUIRES_ALIGNED(bufferSize, os::MemoryPageSize);
        @pre buffer is allocate by os::AllocateMemoryBlock()

        @details
         nrr および nro は、内部で buffer にコピーされるため、アライン等がそろっている必要はなく、
         呼び出し後、破棄しても構わない。
    */
    Result Initialize(const void* nrrData, size_t nrrSize, const void* nroData, size_t nroSize, void* buffer, size_t bufferSize) NN_NOEXCEPT;

    Result Load() NN_NOEXCEPT;
    void Unload() NN_NOEXCEPT;

    ~DllPlugin() NN_NOEXCEPT;

    Result GetSymbol(uintptr_t* pOut, const char* name, bool allowNotFound = false) NN_NOEXCEPT;

    template <typename T>
    Result GetSymbol(T** pOut, const char* name, bool allowNotFound = false) NN_NOEXCEPT
    {
        uintptr_t addr;
        NN_RESULT_DO(GetSymbol(&addr, name, allowNotFound));
        *pOut = reinterpret_cast<T*>(addr);
        NN_RESULT_SUCCESS;
    }

private:

    enum class State
    {
        Initial,
        NotLoaded,
        Loaded,
    };

    State m_State = State::Initial;
    ro::RegistrationInfo m_RegistrationInfo;

    void* m_NroBuffer = nullptr;
    void* m_BssBuffer = nullptr;
    size_t m_PageAlignedBssSize = 0;

    ro::Module m_Module;

};

#define NN_JIT_PLUGIN_GET_SYMBOL(dllPlugin, f, symbol, allowNotFound) \
    ::std::decay<decltype(symbol)>::type f; \
    NN_RESULT_DO((dllPlugin).GetSymbol(&f, NN_MACRO_STRINGIZE(symbol)));

#define NN_JIT_PLUGIN_TRY_GET_SYMBOL(dllPlugin, f, symbol, allowNotFound) \
    ::std::decay<decltype(symbol)>::type f; \
    if (!(dllPlugin).GetSymbol(&f, NN_MACRO_STRINGIZE(symbol)).IsSuccess()) \
    { \
        f = nullptr; \
    }

}}
