﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "jitsrv_DetailUtility.h"

#include <mutex>
#include <nn/os.h>
#include <nn/svc/svc_Base.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_TinyMt.h>

namespace nn { namespace jitsrv { namespace detail {

namespace {

class RandomGenerator
{
private:

    util::TinyMt m_Random;

public:

    RandomGenerator() NN_NOEXCEPT
    {
        Bit64 seed;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&seed, svc::InfoType_Random, nn::svc::Handle(0), 2));
        m_Random.Initialize(seed);
    }

    uint64_t Generate(uint64_t randMax) NN_NOEXCEPT
    {
        return m_Random.GenerateRandomU64() % randMax;
    }

};

RandomGenerator g_RandomGenerator;

}

AslrAllocator::AslrAllocator(nn::svc::Handle process) NN_NOEXCEPT
{
    Bit64 value;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_HeapRegionAddress, process, 0));
    m_HeapBegin = value;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_HeapRegionSize, process, 0));
    m_HeapSize = value;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_ReservedRegionAddress, process, 0));
    m_RsvBegin = value;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_ReservedRegionSize, process, 0));
    m_RsvSize = value;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_AslrRegionAddress, process, 0));
    m_AslrBegin = value;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_AslrRegionSize, process, 0));
    m_AslrSize = value;
}

uint64_t AslrAllocator::GetAslrRegion(uint64_t mappingSize, int index) NN_NOEXCEPT
{
    NN_SDK_ASSERT(index == 0 || index == 1);
    auto begin = index == 0 ? m_AslrBegin : m_AslrBegin + m_AslrSize / 2;
    auto size = m_AslrSize / 2;
    if (mappingSize > size)
    {
        return 0;
    }
    const auto Align1 = os::MemoryPageSize;
    const auto Align2 = 1024 * 1024 * 2;
    auto align = mappingSize % Align2 == 0 ? Align2 : Align1;
    for (int i = 0; i < 512; i++)
    {
        uint64_t addr = begin + g_RandomGenerator.Generate((size - mappingSize) / align) * align;
        if (m_HeapSize && !(addr + mappingSize - 1 < m_HeapBegin || m_HeapBegin + m_HeapSize - 1 < addr))
        {
            continue;
        }
        if (m_RsvSize && !(addr + mappingSize - 1 < m_RsvBegin || m_RsvBegin + m_RsvSize - 1 < addr))
        {
            continue;
        }
        return addr;
    }
    return 0;
}

template <typename F>
inline void IterateCacheLine(uintptr_t addr, size_t size, F f) NN_NOEXCEPT
{
    if (size == 0)
    {
        return;
    }
    uintptr_t cacheTypeRegister;
    asm volatile ("mrs %0, ctr_el0" : "=r" (cacheTypeRegister));
    size_t cacheLineSize = 4 << ((cacheTypeRegister >> 16) & 0xF);
    auto startAddr = reinterpret_cast<uintptr_t>(addr) & ~(cacheLineSize - 1);
    auto endAddr = reinterpret_cast<uintptr_t>(addr) + size;
    for (auto p = startAddr; p < endAddr; p += cacheLineSize)
    {
        f(p);
    }
}

void StoreDataCacheAndMemoryBarrier(uintptr_t addr, size_t size) NN_NOEXCEPT
{
    //データキャッシュ store
    IterateCacheLine(addr, size, [](uintptr_t p)
    {
        asm volatile ("dc cvau, %0"::"r"(p):"memory");
    });
    // ish メモリバリア
    asm volatile ("dsb ish":::"memory");
}

}}}
