﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/os.h>

#include <nn/settings/system/settings_Sleep.h>
#include <nn/settings/system/settings_Tv.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include <nn/idle/detail/idle_Log.h>
#include "idle_PolicyParam.h"
#include "idle_PolicyParamHolder.h"

namespace nn { namespace idle { namespace server {

namespace {

    // 明示的な初期化までは保守的な（ハンドラが発動しない）方向の値にしておく
    const nn::idle::server::PolicyParam InitialPolicyParam = {
        nn::settings::system::HandheldSleepPlan_Never,
        nn::settings::system::ConsoleSleepPlan_Never,
        false,
        false,
        {
            nn::os::Tick(0),
            0,
            0,
            0,
            0,
            IdleTimeDetectionExtension_Disabled,
            nn::hid::system::InputSourceIdSet().Set(),
            nn::omm::OperationMode::Handheld, // 本当は初期状態 unknown
            { 0 },
            false,
            false,
            false
        },
        false,
        0
    };

}

PolicyParamHolder::PolicyParamHolder() NN_NOEXCEPT : m_PolicyParam(InitialPolicyParam) {}

void PolicyParamHolder::Initialize() NN_NOEXCEPT
{
    m_PolicyParam = InitialPolicyParam;
}

void PolicyParamHolder::LoadSystemSettings() NN_NOEXCEPT
{
    // settings 側は本体設定画面の要求に沿ったパラメータ構成になっているので、
    // idle 側の必要なものだけで構成した構造体にマッピングして吸収する
    nn::settings::system::SleepSettings sleepSettings;
    nn::settings::system::GetSleepSettings(&sleepSettings);
    nn::settings::system::TvSettings tvSettings;
    nn::settings::system::GetTvSettings(&tvSettings);

    m_PolicyParam.handheldSleepPlan = static_cast<nn::settings::system::HandheldSleepPlan>(sleepSettings.handheldSleepPlan);
    m_PolicyParam.consoleSleepPlan = static_cast<nn::settings::system::ConsoleSleepPlan>(sleepSettings.consoleSleepPlan);
    m_PolicyParam.isAutoSleepInMediaPlaybackEnabled = sleepSettings.flags.Test<nn::settings::system::SleepFlag::SleepsWhilePlayingMedia>();
    m_PolicyParam.isTvDimmingEnabled = tvSettings.flags.Test<nn::settings::system::TvFlag::PreventsScreenBurnIn>();

    NN_DETAIL_IDLE_INFO(
        "Loaded settings: handheldSleepPlan=%d, consoleSleepPlan=%d, isAutoSleepInMediaPlaybackEnabled=%s, isTvDimmingEnabled=%s\n",
        m_PolicyParam.handheldSleepPlan,
        m_PolicyParam.consoleSleepPlan,
        m_PolicyParam.isAutoSleepInMediaPlaybackEnabled ? "true" : "false",
        m_PolicyParam.isTvDimmingEnabled ? "true" : "false"
    );
}

void PolicyParamHolder::LoadFirmwareDebugSettings() NN_NOEXCEPT
{
    // settings が書かれていない環境用に一旦初期化
    m_PolicyParam.isAutoSleepDisabledForDebug = false;
    m_PolicyParam.overrideAutoSleepTimeForDebugInSeconds = 0;
    settings::fwdbg::GetSettingsItemValue(&m_PolicyParam.isAutoSleepDisabledForDebug, sizeof(m_PolicyParam.isAutoSleepDisabledForDebug), "systemsleep", "disable_auto_sleep");
    settings::fwdbg::GetSettingsItemValue(&m_PolicyParam.overrideAutoSleepTimeForDebugInSeconds, sizeof(m_PolicyParam.overrideAutoSleepTimeForDebugInSeconds), "systemsleep", "override_auto_sleep_time");

    NN_DETAIL_IDLE_INFO(
        "Firmware debug settings: isAutoSleepDisabledForDebug=%s, overrideAutoSleepTimeForDebugInSeconds=%d [s]\n",
        m_PolicyParam.isAutoSleepDisabledForDebug ? "true" : "false",
        m_PolicyParam.overrideAutoSleepTimeForDebugInSeconds
    );
}

void PolicyParamHolder::UpdateHandlingContext(const nn::idle::HandlingContext& handlingContext) NN_NOEXCEPT
{
    m_PolicyParam.handlingContext = handlingContext;
}

}}} // namespace nn::idle::server
